/* 
   XRDrawingEngine.m

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Pascal Forget <pascal@wsc.com>
   Date: March 1996
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May 1998
   
   This file is part of the GNUstep GUI X/RAW Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>
#include <sys/timeb.h>
#include <gnustep/xraw/XR.h>
#include <gnustep/gui/PSMatrix.h>


//*****************************************************************************
//
// 	XRGState 
//
//		A gState objects stores the state of the XRAW drawing engine and acts 
//		as the the client side representation of it's X GC.
//
//*****************************************************************************

@interface XRGState : NSObject
{
@public
  XRContext *context;
  XRWindow  *focusWindow;
  XRFont    *font;
  XRColor   *color;
  
  NSPoint    originPoint;
  NSRect     canvasRect;
  NSRect     clipRect;

  PSMatrix  *matrixToWindow;
  
  Display   *xDisplay;
  Window     xWindow;
  Drawable   xDrawable;
  GC         xGC;
}

@end
@implementation XRGState

+ (void)initialize
{
  if (self == [XRGState class])
    NSDebugLog(@"Initialize XRGState class\n");
}

- (id) init
{
  XGCValues values;
  unsigned long valuemask;
  Window xDisplayRootWindow;
  XSetWindowAttributes winattrs;

  [super init];

  context = (XRContext *)[XRContext currentContext];
  xDisplayRootWindow = [context xDisplayRootWindow];
  xDisplay = [context xDisplay];

  // Create an X GC for the content view
  // set it's colors per chosen NSColor
  values.foreground = winattrs.background_pixel;
  values.background = winattrs.background_pixel;
  values.function = GXcopy;
  valuemask = (GCForeground | GCBackground | GCFunction);
  xGC = XCreateGC(xDisplay, xDisplayRootWindow, valuemask,&values);

  focusWindow = nil;
  font = nil;
  color = nil;
  xWindow = None;
  xDrawable = None;
  matrixToWindow = nil;
  
  return self;
}

- (id) mutableCopyWithZone: (NSZone *)zone
{
  XRGState *newgstate;
  long allGCBitsOne = (-1L << (GCLastBit+1)) ^ -1L;
  newgstate = (XRGState *)NSCopyObject(self, 0, zone);
  [newgstate->matrixToWindow retain];
  // FIXME: how to copy a whole GC?
  XCopyGC (xDisplay, xGC, allGCBitsOne, newgstate->xGC);
  return newgstate;
}

- (void)dealloc
{
//FIXME: why do this coredumps?
//  XFreeGC (xDisplay, xGC);
// Free X window's GC
  [matrixToWindow release];
  [super dealloc];
}

@end  /* XRGState */

//*****************************************************************************
//
// 		XRDrawingEngine 
//
//*****************************************************************************

// drawing engine globals

// FIXME these constants should be in NSColor
static XRColor *XRLightGrayColor = nil;
static XRColor *XRBlackColor = nil;
static XRColor *XRWhiteColor = nil;
static XRColor *XRDarkGrayColor = nil;

static XColor highlightColor;			// highlight XOR X color

// current graphics state.
// FIXME this should be inside GSContext
static XRGState *gstate;
static NSMutableArray *gstateStack;

//*****************************************************************************
//
//		initialize_gnustep_backend()  	initialize the AppKit back end
//
//*****************************************************************************

BOOL 
initialize_gnustep_backend(void)
{		
  NSDebugLog(@"Using the GNUstep GUI X/RAW backend.\n");
  // set the concrete context class
  [GSContext setConcreteClass: [XRContext class]];

  // Have the backend classes poseAs: their frontend super classes
  [XRScreen poseAsClass:[NSScreen class]];
  [XRFontManager poseAsClass:[NSFontManager class]];
  [XRFont poseAsClass:[NSFont class]];
  [XRApplication poseAsClass:[NSApplication class]];
  [XRView poseAsClass:[NSView class]];
  [XRMenuCell poseAsClass:[NSMenuItem class]];
  [XRMenuMatrix poseAsClass:[NSMenuMatrix class]];
  [XRMenu poseAsClass:[NSMenu class]];
  [XRColor poseAsClass:[NSColor class]];
  [XRPopUpButton poseAsClass:[NSPopUpButton class]];
  [XRImage poseAsClass: [NSImage class]];
  [XRImageRep poseAsClass: [NSImageRep class]];
  [XRBitmapImageRep poseAsClass: [NSBitmapImageRep class]];
  [XRCachedImageRep poseAsClass: [NSCachedImageRep class]];
  [XRCursor poseAsClass: [NSCursor class]];
  [XRWindow poseAsClass:[NSWindow class]];

  // Initialize the XRAW drawing engine
  NSDebugLog(@"Initializing XRAW drawing engine\n");
  XRInitializeDrawingEngine();

  // initialize the font manager with the available fonts	
  [NSFontManager sharedFontManager];

  // create the shared dragging view/window
  [XRDragView _sharedDraggingView];

  return YES;	
}


//*****************************************************************************
//
//		XRCacheSystemImages() loads commonly used system images so that
//		they are immediately available to the application.
//
//*****************************************************************************

void 
XRCacheSystemImages(void)
{
	[NSImage imageNamed: @"common_CloseH.tiff"];
	[NSImage imageNamed: @"common_Close.tiff"];
	[NSImage imageNamed: @"common_ArrowDown.tiff"];
	[NSImage imageNamed: @"common_ArrowLeft.tiff"];
	[NSImage imageNamed: @"common_ArrowRight.tiff"];
	[NSImage imageNamed: @"common_ArrowUp.tiff"];
	[NSImage imageNamed: @"common_Dimple.tiff"];
	[NSImage imageNamed: @"common_RadioOff.tiff"];
	[NSImage imageNamed: @"common_RadioOn.tiff"];
	[NSImage imageNamed: @"common_SwitchOff.tiff"];
	[NSImage imageNamed: @"common_SwitchOn.tiff"];
	[NSImage imageNamed: @"common_ret.tiff"];
}
//*****************************************************************************
//
// 		XRInitializeDrawingEngine()   initialize the XRAW drawing engine 
//
//*****************************************************************************

void 
XRInitializeDrawingEngine(void)
{
  XRContext *context;
  
  // create an abstract representation of 
  // the default drawing destination
  context = [XRContext contextWithInfo: nil];
  [XRContext setCurrentContext: context];

  // create the current gstate -- FIXME should be in XRContext
  gstate = [[XRGState alloc] init];
  gstateStack = [[NSMutableArray alloc] initWithObjects: gstate, nil];
  [gstate release];
  gstate = nil;

  // FIXME this doesn't belong in here, should be in NSColor
  // Allocate and retain the most commonly used colors		
  XRWhiteColor = [[NSColor whiteColor] retain];
  XRBlackColor = [[NSColor blackColor] retain];
  XRDarkGrayColor = [[NSColor darkGrayColor] retain];
  XRLightGrayColor = [[NSColor lightGrayColor] retain];		

// tmp FIX ME
//  selectionColor = [[NSColor selectedTextColor] retain];
//  highlightColor = [(XRColor *)selectionColor xColor];
  highlightColor = [(XRColor *)XRLightGrayColor xColor];

  // cache default system images
//  XRCacheSystemImages();
}				
//*****************************************************************************
//
// 		return a unique integer to identify each of the app's windows 
//
//*****************************************************************************

int 
XRUniqueWindowTag(void)
{
  static int uniqueTag = 0;

  return uniqueTag++;
}


//
// convert rect in focusView to focusWindow coordinates
//
NSRect convertRectToWindow(NSRect rect)
{
  NSCAssert(gstate, NSInternalInconsistencyException);
  return [gstate->matrixToWindow rectInMatrixSpace:rect];
}

//
// convert point in focusView to focusWindow coordinates
//
NSPoint convertPointToWindow(NSPoint point)
{
  NSCAssert(gstate, NSInternalInconsistencyException);
  return [gstate->matrixToWindow pointInMatrixSpace:point];
}

//*****************************************************************************
//
//      Convert a rectangle in focusWindow
//      from Openstep coordinates to X coordinates
//
//*****************************************************************************

XRectangle XRectFromNSRect(NSRect windowRect)
{
  XRectangle xRect;

  NSCAssert(gstate, NSInternalInconsistencyException);
  // Xwindows's rects start at the top
  windowRect.origin.y += windowRect.size.height;
  // Xwindows coordinates are flipped
  xRect.y      = gstate->canvasRect.size.height - windowRect.origin.y;

  xRect.x      = windowRect.origin.x;
  xRect.width  = windowRect.size.width;
  xRect.height = windowRect.size.height;

  return xRect;
}

//*****************************************************************************
//
//      Convert a point in focusWindow
//      from Openstep coordinates to X coordinates
//
//*****************************************************************************

XPoint XPointFromNSPoint(NSPoint windowPoint)
{
  XPoint xPoint;

  NSCAssert(gstate, NSInternalInconsistencyException);
  xPoint.x = windowPoint.x;
  // X is flipped
  xPoint.y = gstate->canvasRect.size.height - windowPoint.y;

  return xPoint;
}

//*****************************************************************************
//
// 		Swaps a rectangles black for white and vice-versa (not per OS spec)
//
//*****************************************************************************

void NSHighlightRect(NSRect aRect)       
{				// fill rect via XOR
  XGCValues values;
  unsigned long valuemask;
  XRectangle xRect;

  // convert to X coordinates
  xRect = XRectFromNSRect(convertRectToWindow(aRect));

  // hack for splitview
  if (gstate->color == XRDarkGrayColor)
    values.foreground = highlightColor.pixel;		  
  else	
    values.foreground = 0x1f1f00;     // default highlight color

  // XOR rect with the current draw color
  values.function = GXxor;
  values.background = 0L;
  valuemask = (GCFunction | GCForeground | GCBackground);
  XChangeGC(gstate->xDisplay, gstate->xGC, valuemask, &values);
  XFillRectangle(gstate->xDisplay, gstate->xWindow, gstate->xGC,
                 xRect.x, xRect.y, xRect.width, xRect.height);

  // XOR backing if it exists
  if ((Drawable)gstate->xWindow != (Drawable)gstate->xDrawable)
    XFillRectangle(gstate->xDisplay, gstate->xDrawable, gstate->xGC,
                   xRect.x, xRect.y, xRect.width, xRect.height);

  // return GC to it's original state
  values.function = GXcopy;
  values.foreground = [gstate->color xColor].pixel;
  valuemask = (GCFunction | GCForeground);
  XChangeGC(gstate->xDisplay, gstate->xGC, valuemask, &values);
}
//*****************************************************************************
//
// 		Fills rect using current color 
//
//*****************************************************************************

void NSRectFill(NSRect aRect)
{
  XRectangle xRect;

  // convert to X coordinates
  xRect = XRectFromNSRect(convertRectToWindow(aRect));

  [gstate->focusWindow _windowNeedsFlushInXRect:xRect];

  // fill area with the current draw color
  XFillRectangle(gstate->xDisplay, gstate->xDrawable, gstate->xGC,
                 xRect.x, xRect.y, xRect.width, xRect.height);
}
//*****************************************************************************
//
// 		Fills an array of count rectangles with the current color. 
//
//*****************************************************************************

void NSRectFillList(const NSRect *rects, int count)
{
  int i;

  for (i=0; i<count; i++)
    NSRectFill(rects[i]);
}
//*****************************************************************************
//
// 		Fills each rectangle in the array 'rects' with the gray whose
//		is stored at the corresponding location in the array 'grays'.
//		Both arrays must be count elements long.  
//
//*****************************************************************************

void NSRectFillListWithGrays(const NSRect *rects,const float *grays,int count) 
{
}


//*****************************************************************************
//
//  draw a bordered light gray rectangle with appearance of a button
//
//*****************************************************************************

#define PXBW 1.0     // bezel border width

void NSDrawButton(NSRect aRect, NSRect clipRect)
{
  XPoint pnts[6];
  XRectangle xRect;

  // convert to X coordinates
  xRect = XRectFromNSRect(convertRectToWindow(aRect));

  [gstate->focusWindow _windowNeedsFlushInXRect:xRect];

  // -1 so that we stay in bounds
  xRect.height--;
  xRect.width--;

  // draw the top and left light edges in white
  pnts[0].x = xRect.x;
  pnts[0].y = (xRect.y + xRect.height);
  pnts[1].x = 0;// one pixel wide
  pnts[1].y = -(xRect.height);
  pnts[2].x = (xRect.width);
  pnts[2].y = 0;
  XRSetCurrentColor(XRWhiteColor);
  XDrawLines (gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              pnts, 3, CoordModePrevious);

  // draw the top and left light edges in light gray
  pnts[3].x = xRect.x +PXBW;
  pnts[3].y = (xRect.y + xRect.height -PXBW);
  pnts[4].x = 0;// one pixel wide
  pnts[4].y = -(xRect.height -2*PXBW);
  pnts[5].x = (xRect.width -PXBW);
  pnts[5].y = 0;
  XRSetCurrentColor(XRLightGrayColor);
  XDrawLines (gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              &pnts[3], 3, CoordModePrevious);

  // draw bottom and right dark edges
//pnts[0].x = xRect.x;
//pnts[0].y = (xRect.y + xRect.height);
  pnts[1].x = (xRect.width);
  pnts[1].y = 0;
  pnts[2].x = 0;
  pnts[2].y = -(xRect.height);
  XRSetCurrentColor(XRBlackColor);
  XDrawLines (gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              pnts, 3, CoordModePrevious);

//pnts[3].x = xRect.x + PXBW;
//pnts[3].y = (xRect.y + xRect.height - PXBW);
  pnts[4].x = xRect.width - 2*PXBW;
  pnts[4].y = 0;
  pnts[5].x = 0;
  pnts[5].y = 2*PXBW - xRect.height;
  XRSetCurrentColor(XRDarkGrayColor);
  XDrawLines (gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              &pnts[3], 3, CoordModePrevious);

}

//*****************************************************************************
//
//  draw a bordered light gray rectangle with pushed-in button appearance
//
//*****************************************************************************

void NSDrawGrayBezel(NSRect aRect, NSRect clipRect)
{
  XPoint pnts[6];
  XRectangle xRect;

  // convert to X coordinates
  xRect = XRectFromNSRect(convertRectToWindow(aRect));

  [gstate->focusWindow _windowNeedsFlushInXRect:xRect];

  // -1 so that we stay in bounds
  xRect.height--;
  xRect.width--;

  // draw the top and left dark edges
  pnts[0].x = xRect.x;
  pnts[0].y = (xRect.y + xRect.height);
  pnts[1].x = 0;
  pnts[1].y = -(xRect.height);
  pnts[2].x = (xRect.width);
  pnts[2].y = 0;
  XRSetCurrentColor(XRDarkGrayColor);
  XDrawLines (gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              pnts , 3, CoordModePrevious);

  pnts[3].x = xRect.x + PXBW;
  pnts[3].y = (xRect.y + xRect.height - PXBW);
  pnts[4].x = 0;
  pnts[4].y = 2*PXBW - xRect.height;
  pnts[5].x = (xRect.width - 2*PXBW);
  pnts[5].y = 0;
  XRSetCurrentColor(XRBlackColor);
  XDrawLines (gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              &pnts[3] , 3, CoordModePrevious);

  // draw bottom and right light edges
  pnts[0].x = xRect.x;
//pnts[0].y = (xRect.y + xRect.height);
  pnts[1].x = xRect.width;
  pnts[1].y = 0;
  pnts[2].x = 0;
  pnts[2].y = -(xRect.height);          // white outer edge
  XRSetCurrentColor(XRWhiteColor);
  XDrawLines (gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              pnts, 3, CoordModePrevious);

  pnts[0].x = xRect.x + PXBW;
  pnts[0].y = (xRect.y + xRect.height - PXBW);
  pnts[1].x = (xRect.width - 2*PXBW);
//pnts[1].y = 0;
//pnts[2].x = 0;
  pnts[2].y = (2*PXBW - xRect.height);  // gray inner edge
  XRSetCurrentColor(XRLightGrayColor);
  XDrawLines (gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              pnts, 3, CoordModePrevious);
}


//*****************************************************************************
//
// 		draw a light gray rectangle whose border is a groove
//
//*****************************************************************************

void NSDrawGroove(NSRect aRect, NSRect clipRect)
{
  XPoint pnts[5];
  XRectangle xRect;

  // convert to X coordinates
  xRect = XRectFromNSRect(convertRectToWindow(aRect));

  [gstate->focusWindow _windowNeedsFlushInXRect:xRect];

  // -1 so that we stay in bounds
  xRect.height--;
  xRect.width--;

  // draw top / left edges dark gray one pixel wide
  pnts[0].x = xRect.x;
  pnts[0].y = (xRect.y + xRect.height);
  pnts[1].x = 0;
  pnts[1].y = -(xRect.height);
  pnts[2].x = (xRect.width);
  pnts[2].y = 0;
  XRSetCurrentColor(XRDarkGrayColor);
  XDrawLines (gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              pnts, 3, CoordModePrevious);

  // draw white rect one pixel wide
  pnts[0].x = xRect.x + PXBW;
//pnts[0].y = (xRect.y + xRect.height);
//pnts[1].x = 0;
  pnts[1].y = -(xRect.height) + PXBW;
  pnts[2].x = (xRect.width) - PXBW;
//pnts[2].y = 0;
  pnts[3].x = 0;
  pnts[3].y = (xRect.height) - PXBW;
  pnts[4].x = (PXBW - xRect.width);
  pnts[4].y = 0;
  XRSetCurrentColor(XRWhiteColor);
  XDrawLines (gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              pnts, 5, CoordModePrevious);
	
  pnts[0].x = xRect.x + 2*PXBW;
  pnts[0].y = (xRect.y + xRect.height) - PXBW;
  pnts[1].x = (xRect.width) - 3*PXBW;
  pnts[1].y = 0;
  pnts[2].x = 0;
  pnts[2].y = -(xRect.height) + 3*PXBW;
  XRSetCurrentColor(XRDarkGrayColor);
  XDrawLines (gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              pnts, 3, CoordModePrevious);
}


//*****************************************************************************
//
// 		Draws a frame of width 1.0 using the current color.  
//
//*****************************************************************************

void NSFrameRect(NSRect aRect)  
{
  XPoint pnts[5];
  XRectangle xRect;

  // convert to X coordinates
  xRect = XRectFromNSRect(convertRectToWindow(aRect));

  [gstate->focusWindow _windowNeedsFlushInXRect:xRect];

  // -1 so that we stay in bounds
  xRect.height--;
  xRect.width--;

  pnts[0].x = xRect.x;
  pnts[0].y = (xRect.y + xRect.height);
  pnts[1].x = 0;
  pnts[1].y = -(xRect.height);
  pnts[2].x = (xRect.width);
  pnts[2].y = 0;
  pnts[3].x = 0;
  pnts[3].y = (xRect.height);
  pnts[4].x = -(xRect.width);
  pnts[4].y = 0;
  // draw frame rect with current color
  XDrawLines (gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              pnts, 5, CoordModePrevious);
}


//*****************************************************************************
//
// 		Draws a frame of width frameWidth using the current color. 
//
//*****************************************************************************

void NSFrameRectWithWidth(NSRect aRect, float frameWidth) 
{
}


//*****************************************************************************
//
// 		draw a string to focus view's xDrawable 
//
//*****************************************************************************

void XRDrawString(const char *str, NSRect aRect)      
{
  XRectangle xRect;

  // convert to X coordinates
  xRect = XRectFromNSRect(convertRectToWindow(aRect));
  xRect.y += [gstate->font ascender];

//  [gstate->focusWindow _windowNeedsFlushInXRect:xRect];

  // draw string onto the drawable using the current color
  XDrawString(gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              xRect.x, xRect.y,
              str, (int)strlen(str));
}

void PSshow(const char *str)                        // emulate PSshow   
{
  XPoint xPoint;

  // convert to X coordinates
  xPoint = XPointFromNSPoint(convertPointToWindow(gstate->originPoint));

  // draw string onto the drawable using the current color
  XDrawString(gstate->xDisplay, gstate->xDrawable, gstate->xGC,
              xPoint.x, xPoint.y,
              str, (int)strlen(str));
}


//*****************************************************************************
//
// 		draw an XImage to focus view's gstate->xDrawable 
//
//*****************************************************************************

void XRDrawXImage(XImage *xImage, NSRect aRect)      
{
  XRectangle xRect;

  // convert to X coordinates
  xRect = XRectFromNSRect(convertRectToWindow(aRect));

  [gstate->focusWindow _windowNeedsFlushInXRect:xRect];

  XPutImage(gstate->xDisplay, gstate->xDrawable, gstate->xGC, xImage,
            0, 0, xRect.x, xRect.y, xRect.width, xRect.height);
}


//*****************************************************************************
//
// 		line drawing functions 
//
//*****************************************************************************

void
XRDrawLine(NSPoint a, NSPoint b)
{
  XPoint o, d;

  // convert to X coordinates
  o = XPointFromNSPoint(convertPointToWindow(a));
  d = XPointFromNSPoint(convertPointToWindow(b));

  // draw a line in drawable using current color
  XDrawLine (gstate->xDisplay, gstate->xDrawable, gstate->xGC,
             o.x, o.y, d.x, d.y);
}

void 
PSmoveto(float x, float y)          // move drawing engine to origin point
{
  NSCAssert(gstate, NSInternalInconsistencyException);
  gstate->originPoint.x = x;
  gstate->originPoint.y = y;
}

void 
PSlineto(float x, float y)          // Draw line to point using current color
{
  NSPoint bPoint;

  NSCAssert(gstate, NSInternalInconsistencyException);
  bPoint.x = x;
  bPoint.y = y;
  XRDrawLine(gstate->originPoint, bPoint);
  gstate->originPoint = bPoint;
}

void 
PSrlineto(float x, float y)         // Draw line relative to the current point
{
  NSPoint bPoint;

  NSCAssert(gstate, NSInternalInconsistencyException);
  bPoint.x = gstate->originPoint.x + x;
  bPoint.y = gstate->originPoint.y + y;
  XRDrawLine(gstate->originPoint, bPoint);
  gstate->originPoint = bPoint;
}

void PSstroke(void)		{}			// dummies defines not
void PSnewpath(void)		{}			// implemented in XRAW
void PSclosepath(void)		{}
void PSfill(void)		{}
void PSsetlinewidth(float width){}

void PSsetgray(float num)					
{
// FIXME there must be a better way of setting a gray color X
// set the drawing engine's current color to one of the
// predefined cached colors if possible
  if (num <= 0)
    XRSetCurrentColor(XRBlackColor);
  else if (num >= 1)
    XRSetCurrentColor(XRWhiteColor);
  else if ((num >= .33) && (num < .34))
    XRSetCurrentColor(XRDarkGrayColor);
  else if ((num > .66) && (num <= .67))
    XRSetCurrentColor(XRLightGrayColor);
  else
    [[NSColor colorWithDeviceWhite: num alpha: 1.0] set];
}

//*****************************************************************************
//
// 		set the font to be used by the drawing engine to aFont
//
//*****************************************************************************

void 
XRSetCurrentFont (XRFont *aFont)
{
  NSCAssert(gstate, NSInternalInconsistencyException);
  // if font has changed
//  if (gstate->font != aFont)
    {
      gstate->font = aFont;

      if (gstate->xDisplay && aFont)
        {
          // set GC's font and store ascender val
          XFontStruct *xfont = [aFont xFontStruct];
          XSetFont(gstate->xDisplay, gstate->xGC, xfont->fid);
        }
    }
}


//*****************************************************************************
//
// 		set the drawing engine's current color 
//
//*****************************************************************************

void 
XRSetCurrentColor(NSColor *color)
{
  NSCAssert(gstate, NSInternalInconsistencyException);
//  if (gstate->color != (XRColor *) color)
    {
      // color has changed
      gstate->color = (XRColor *) color;

      if (gstate->xDisplay && color)
        {
          XColor xColor = [(XRColor *) color xColor];
          XSetForeground(gstate->xDisplay, gstate->xGC, xColor.pixel);
        }
    }
}

void 
XRSyncFocusWindowColor(NSColor *color, XRWindow *window)
{
  // sync focus window's GC color if color is
  // changed outside of the drawing engine
  if (gstate && (gstate->focusWindow == window))
    gstate->color = (XRColor *) color;
}


NSColor * XRLightGray(void)			{ return XRLightGrayColor; }		
NSColor * XRBlack(void)				{ return XRBlackColor; }
NSColor * XRWhite(void)				{ return XRWhiteColor; }
NSColor * XRDarkGray(void)			{ return XRDarkGrayColor; }

//*****************************************************************************
//
// 		set and return the drawing engine's current X GC  (X graphics context)
//
//*****************************************************************************

GC XRCurrentGC(void)
{
  NSCAssert(gstate, NSInternalInconsistencyException);
  return gstate->xGC;
}

void XRSetGC(GC aGC)
{
  NSCAssert(gstate, NSInternalInconsistencyException);
  // set the drawing engine's current GC (X graphics context)
  gstate->xGC = aGC;
  if (gstate->xGC && gstate->xDisplay) 
    {				
      XFontStruct *xfont = [gstate->font xFontStruct];
      XSetFont(gstate->xDisplay, gstate->xGC, xfont->fid);
    }
}


void PSgsave(void)
{
  XRGState *newgstate;
  if (gstate == nil)
    gstate = [gstateStack objectAtIndex: 0];
  newgstate = [gstate mutableCopy];
  [gstateStack addObject: newgstate];
  gstate = newgstate;
  if (gstate)
    {
      /*
       * Reset font, color, and clipping rectangle for the gstate.
       */
      if (gstate->font)
	{
	  XFontStruct *xfont = [gstate->font xFontStruct];
	  XSetFont(gstate->xDisplay, gstate->xGC, xfont->fid);
	}
      if (gstate->xDisplay && gstate->color)
        {
          XColor xColor = [(XRColor *) gstate->color xColor];
          XSetForeground(gstate->xDisplay, gstate->xGC, xColor.pixel);
        }
      XRSetClipRect(gstate->clipRect);      					
    }
  [newgstate release];
}

void PSgrestore(void)
{
  int count = [gstateStack count];
  if (count <= 1)
    {
      NSLog (@"PSgrestore: Graphics state stack underflow");
      return;
    }
  [gstateStack removeObjectAtIndex: --count];
  gstate = [gstateStack objectAtIndex: --count];
  if (count == 0)
    gstate = nil;
  else
    {
      /*
       * Restore font, color, and clipping rectangle for the gstate.
       */
      if (gstate->font)
	{
	  XFontStruct *xfont = [gstate->font xFontStruct];
	  XSetFont(gstate->xDisplay, gstate->xGC, xfont->fid);
	}
      if (gstate->xDisplay && gstate->color)
        {
          XColor xColor = [(XRColor *) gstate->color xColor];
          XSetForeground(gstate->xDisplay, gstate->xGC, xColor.pixel);
        }
      XRSetClipRect(gstate->clipRect);      					
    }
}

// set draw focus
void XRFocusLock(XRView *view, XRWindow *window)			
{
  NSCAssert(gstate, NSInternalInconsistencyException);
  //if (gstate->focusWindow != window)
    {
      // if focus view is in a new window, get more info
      gstate->focusWindow = window;
      gstate->context = (XRContext *)[XRContext currentContext];
      gstate->xDisplay = [gstate->context xDisplay];
      gstate->xWindow = [window xWindow];
      gstate->xGC = [window xGC];
      gstate->xDrawable = [window xDrawable];
      gstate->canvasRect = [window frame];
    }
  gstate->color = nil;

  ASSIGN(gstate->matrixToWindow, [view _matrixToWindow]);

  {
    NSRect aRect;
    NSRect newClipRect;

    /*
     *	Set up rectangle for entire window.
     */
    gstate->clipRect.size = gstate->canvasRect.size;
    gstate->clipRect.origin.x = 0;
    gstate->clipRect.origin.y = 0;

    /*
     *	Now find a rectangle for the visible part of the view.
     */
    aRect = [view visibleRect];
    // if view is rotated compute the bounding box rectangle   
    if ([gstate->matrixToWindow isRotated])
      [gstate->matrixToWindow boundingRectFor: aRect result: &newClipRect];
    else
      newClipRect = aRect;
    newClipRect = convertRectToWindow(newClipRect);

    /*
     *	Set clipping rectangle as intersect of view and window.
     */
    gstate->clipRect = NSIntersectionRect(gstate->clipRect, newClipRect);
    XRSetClipRect(gstate->clipRect);
  }
}

void XRWindowGeometryChanged(XRWindow *window, Drawable x_drawable)
{
  // if focus window geometry changes, update gstate->xDrawable and canvas rect

  if (gstate && gstate->focusWindow == window)
    {
      gstate->xDrawable = x_drawable;
      gstate->canvasRect = [window frame];
    }
}
//*****************************************************************************
//
// 		copy rectangle to new coordinate point 
//
//*****************************************************************************


void XRCopyRectToPoint(NSRect srcRect, NSPoint destPoint)      
{
  NSRect destRect;
  XRectangle srcXRect;
  XRectangle destXRect;
  
//NSLog(@"[%@] [%@]", NSStringFromRect(srcRect), NSStringFromPoint(destPoint));

  // In Openstep, destPoint is in the bottom of rect. in X it should be on top.
  // convert a rectangle instead of a point for destination,
  // because if the view isn't flipped, point should be in other side of rect.
  destRect.origin = destPoint;
  destRect.size = srcRect.size;

  srcXRect  = XRectFromNSRect(convertRectToWindow(srcRect));
  destXRect = XRectFromNSRect(convertRectToWindow(destRect));

//  printf("will copy [%d %d %d %d] to [%d %d]\n",
//            srcXRect.x, srcXRect.y,
//            srcXRect.width, srcXRect.height,
//            destXRect.x, destXRect.y);
  XCopyArea(gstate->xDisplay, gstate->xDrawable, gstate->xDrawable, gstate->xGC,
            srcXRect.x, srcXRect.y,
            srcXRect.width, srcXRect.height,
            destXRect.x, destXRect.y);
}


//*****************************************************************************
//
// 		flush rect from window's back store 
//
//*****************************************************************************

void XRFlushRect(NSRect rect)      
{
  XRectangle xRect;

  NSCAssert(gstate, NSInternalInconsistencyException);
  // if window has backing, copy pixmap backing to view's Xwindow
  if ((Drawable)gstate->xWindow != (Drawable)gstate->xDrawable)
    {
      // convert to X coordinates
      xRect   = XRectFromNSRect(convertRectToWindow(rect));

      XCopyArea(gstate->xDisplay, gstate->xDrawable, gstate->xWindow, gstate->xGC,	
                xRect.x, xRect.y,
                xRect.width, xRect.height,
                xRect.x, xRect.y);
    }
}

void XRFlushXRect(XRectangle xRect)             // X coords are assumed
{
  NSCAssert(gstate, NSInternalInconsistencyException);
  // if window has backing, copy pixmap backing to view's Xwindow
  if ((Drawable)gstate->xWindow != (Drawable)gstate->xDrawable)
    {
      XCopyArea(gstate->xDisplay, gstate->xDrawable, gstate->xWindow, gstate->xGC,	
                xRect.x, xRect.y,
                xRect.width, xRect.height,
                xRect.x, xRect.y);
    }
}
//*****************************************************************************
//
// 		clipping routines 
//
//*****************************************************************************

void NSRectClip(NSRect aRect)      
{
  NSRect newClipRect;

  NSCAssert(gstate, NSInternalInconsistencyException);
  // if view is rotated compute the bounding box rectangle   
  if ([gstate->matrixToWindow isRotated])
    [gstate->matrixToWindow boundingRectFor: aRect result: &newClipRect];
  else
    newClipRect = aRect;

  newClipRect = convertRectToWindow(newClipRect);

  // intersect focussed view clip rectangle
  newClipRect = NSIntersectionRect(gstate->clipRect, newClipRect);

  XRSetClipRect(newClipRect);      					
}

void NSRectClipList(const NSRect *rects, int count)
//FIXME: the OS spec says each rect in the list should intersect the current 
//       clip rect. Rhapsody's doc says the current clip should be intersected
//       with the union of all rects. Here it's per the spec.
{
  int i;
    
  for (i=0; i<count; i++)
    NSRectClip(rects[i]);
}

void XRSetClipRect(NSRect aRect)
{
  XRectangle xRect;

  NSCAssert(gstate, NSInternalInconsistencyException);
  xRect = XRectFromNSRect(aRect);

  // Set the clipping path to the rectangles in order to limit 
  // the effects of drawing
  XSetClipRectangles(gstate->xDisplay, gstate->xGC, 0, 0, &xRect, 1, Unsorted);	
}

void XRRemoveClipPath(void)
{
  NSCAssert(gstate, NSInternalInconsistencyException);
  XRSetClipRect(gstate->clipRect);
}

//*****************************************************************************
//
// 		gState 
//
//*****************************************************************************

int XRAllocateGState(void)
{
XRGState* gs;

	gs = [XRGState new];
	gs->font = gstate->font;

	return 0;
}
