/*
   XRFont.m

   NSFont for GNUstep GUI X/RAW Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Scott Christley
   Author: Ovidiu Predescu <ovidiu@bx.logicnet.ro>
   Date: February 1997
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May, October 1998
   Author:  Michael Hanni <mhanni@sprintmail.com>
   Date: August 1998

   This file is part of the GNUstep GUI X/RAW Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <Foundation/NSDictionary.h>
#include <Foundation/NSPathUtilities.h>

#include <AppKit/NSFontManager.h>

#include <gnustep/xraw/XRFont.h>
#include <gnustep/xraw/XRContext.h>


extern void XRSetCurrentFont(NSFont *aFont);
				// set drawing 

															// engine's font
//
// Backend structure for XRFont
//
typedef struct 												// Font info struct
{												 
	XFontStruct *font_info;									// ptr to xfont 
	float tabWidth;

} XRFont_struct;

#define XFONT (((XRFont_struct *)be_font_reserved)->font_info)
#define TABWIDTH (((XRFont_struct *)be_font_reserved)->tabWidth)



@implementation XRFont

															// class global 
static NSMutableDictionary* fontDictionary = nil;
			// dictionary of

															// existing fonts
- (XFontStruct *)xFontStruct
		{
return XFONT;
}
		

															
- (void)_initWithXFont:(XFontStruct *)xFontStruct
				
{
															// private XRAW 
	XFONT = xFontStruct;
									// init method not
	TABWIDTH = 0;
											// part of OS spec
	TABWIDTH = [self widthOfString:@"\t"];
}

+ (void)initialize
{
static BOOL initialized = NO;

	if (!initialized) 
		{
		initialized = YES;
		fontDictionary = [NSMutableDictionary new];
		}
}

+ (NSFont*)fontWithName:(NSString*)name matrix:(const float*)fontMatrix
{
Display *xDisplay = [XRContext currentXDisplay];
XRFont* font;
NSString* xfontname;
char *xfontName = "helvetica"; 								// font family
char *xfontWeight = "medium"; 								// font weight 
XFontStruct *xFont;

	if ([name caseInsensitiveCompare:@"Helvetica"] == NSOrderedSame)
		xfontName = "helvetica";
	else
		{ 
		if (([name caseInsensitiveCompare:@"Helvetica-Bold"] 
					== NSOrderedSame))
			{
			xfontName = "helvetica";
			xfontWeight = "bold";
			}
		else 
			{
			if (([name caseInsensitiveCompare: [NSString 
					   stringWithCString:"Courier"]] == NSOrderedSame))
                xfontName = "courier";
			}
		}

	xfontname = [NSString stringWithFormat: 
							@"-*-%s-%s-r-normal-*-%d-*-*-*-*-*-*-*",
							xfontName, xfontWeight, (int)fontMatrix[0]];

	if ((font = [fontDictionary objectForKey:xfontname]))
	// if font exists	
		return font;
										// return it

	font = [[self new] autorelease];
						// else create new
	[fontDictionary setObject:font forKey:xfontname];
		// and add to dict

	font->fontName = [name copy];							// set font attr's
	memcpy (font->matrix, fontMatrix, sizeof (font->matrix));
													// alloc back-end structure
	font->be_font_reserved = calloc(1, sizeof(XRFont_struct));	

	if(xDisplay)
										// Load X font and get
		{
												// font info structure.
    	if ((xFont = XLoadQueryFont(xDisplay, [xfontname cString])) == NULL)
			{
			(void) fprintf( stderr, "Selected font: %s is not available.\n",
							[xfontname cString]);
			(void) fprintf( stderr, "Using system fixed font instead\n");

														// try default font
			if ((xFont = XLoadQueryFont(xDisplay, "9x15")) == NULL)
				(void) fprintf( stderr, "Unable to open fixed font\n");
			}
		else
			(void)fprintf(stderr, "Loaded font: %s \n", [xfontname cString]);

		[font _initWithXFont:xFont];
		}

	return font;
}

- (void)dealloc
{
	XUnloadFont([XRContext currentXDisplay], XFONT->fid);
	free(be_font_reserved);									// free the backend 									
															// structure
	[super dealloc];
}

- (void)set
{															// set the drawing
	XRSetCurrentFont(self);									// engine's font
}

- (float)widthOfString:(NSString*)string
					// calc the size of
{															// string in font
int strLength;
const char *cStr = [string cString];

	if((strLength = [string length]) == 1)					// optimize calc of
		{													// tab width
		if(*cStr == '\t')
			return TABWIDTH;
		}

	return XTextWidth(XFONT, cStr, strLength);
}

- (NSFont*)printerFont			{ return self; }
- (NSFont*)screenFont			{ return self; }

- (NSString*)familyName
{
	return fontName;
}

- (NSRect)boundingRectForFont
{
	return NSZeroRect;
}

- (NSString *)encodingScheme
{
	return nil;
}

- (BOOL)isFixedPitch
{
	return NO;
}

- (BOOL)isBaseFont
{
	return NO;
}

- (float)ascender				{ return XFONT->max_bounds.ascent; }
- (float)descender				{ return XFONT->max_bounds.descent; }

- (float)capHeight
{
	return 0;
}

- (float)italicAngle
{
	return 0;
}

- (NSSize)maximumAdvancement
{
	return NSMakeSize(XFONT->max_bounds.width, 
					  (XFONT->max_bounds.ascent +
XFONT->max_bounds.descent)); }

- (NSSize)minimumAdvancement
{
	return NSMakeSize(0,0);
}

- (float)underlinePosition
{
	return 0;
}

- (float)underlineThickness
{
	return 0;
}

- (float*)widths
{
	return 0;
}

- (NSSize)advancementForGlyph:(NSGlyph)glyph
{
														// glyph is an ascii 
NSSize advancement = { 0, 0 };
							// char value

	if(XFONT->per_char)				// xfont helv seems to have info at -32?
		advancement.width = (float)XFONT->per_char[glyph - 32].width;
	else							// if per_char is NULL assume max bounds
		advancement.width = (float)XFONT->max_bounds.width;

	return advancement;
}

- (NSRect)boundingRectForGlyph:(NSGlyph)glyph
{
	return NSZeroRect;
}

- (BOOL)glyphIsEncoded:(NSGlyph)glyph
{
	return NO;
}

- (NSGlyph)glyphWithName:(NSString*)glyphName
{
	return 0;
}

- (NSPoint)positionOfGlyph:(NSGlyph)curGlyph
		   precededByGlyph:(NSGlyph)prevGlyph
		   isNominal:(BOOL *)nominal
{
	return NSZeroPoint;
}

- (float)pointSize
{
	return (XFONT->max_bounds.ascent +
XFONT->max_bounds.descent);
}

- (Class) classForCoder: aCoder
{
  if ([self class] == [XRFont class])
    return [super class];
  return [self class];
}

@end
