/* -*- C++ -*-
   XGContext - Drawing context using the XGPS Library.

   Copyright (C) 1998,1999 Free Software Foundation, Inc.

   Written by:  Adam Fedor <fedor@boulder.colorado.edu>
   Date: Nov 1998
   
   This file is part of the GNU Objective C User Interface Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   */

#include "gnustep/xgps/XGContext.h"
#include "gnustep/xgps/XGDrawObject.h"
#include "gnustep/xgps/XGGState.h"
#include "wraster/xrtools.h"
#include <AppKit/AppKitExceptions.h>
#include <AppKit/NSAffineTransform.h>
#include <AppKit/NSView.h>
#include <AppKit/NSWindow.h>
#include <Foundation/NSException.h>
#include <Foundation/NSArray.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSData.h>
#include <Foundation/NSValue.h>
#include <Foundation/NSString.h>
#include <Foundation/NSUserDefaults.h>
#include <Foundation/NSDebug.h>

#include "gnustep/xgps/XGContextPrivate.h"
#include "XGDrawingEngine.h"

#include "math.h"
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>


#if LIB_FOUNDATION_LIBRARY
# include <Foundation/NSPosixFileDescriptor.h>
#elif defined(NeXT_PDO)
# include <Foundation/NSFileHandle.h>
# include <Foundation/NSNotification.h>
#endif

#define GSI_ARRAY_TYPES       GSUNION_OBJ

#if     GS_WITH_GC == 0
#define GSI_ARRAY_RELEASE(X)    [(X).obj release]
#define GSI_ARRAY_RETAIN(X)     [(X).obj retain]
#else
#define GSI_ARRAY_RELEASE(X)
#define GSI_ARRAY_RETAIN(X)     (X).obj
#endif

#ifdef GSIArray
#undef GSIArray
#endif
#include <base/GSIArray.h>

static NSMutableDictionary	*globalfontdir = nil;
static Atom			WM_STATE;
static BOOL                     xDndInitialized = NO;
static DndClass                 dnd;

DndClass xdnd(void)
{
  return dnd;			// FIX ME rename with private desig
}

Atom 
GSActionForDragOperation(NSDragOperation op)
{
  Atom xaction;
  switch (op)
    {
    case NSDragOperationNone: xaction = None; break;
    case NSDragOperationCopy: xaction = dnd.XdndActionCopy; break;
    case NSDragOperationLink: xaction = dnd.XdndActionLink; break;
    case NSDragOperationGeneric: xaction = dnd.XdndActionCopy; break;
    case NSDragOperationPrivate: xaction = dnd.XdndActionPrivate; break;
    case NSDragOperationAll: xaction = dnd.XdndActionPrivate; break;
    }
  return xaction;
}

NSDragOperation
GSDragOperationForAction(Atom xaction)
{
  NSDragOperation action;
  if (xaction == dnd.XdndActionCopy)
    action = NSDragOperationCopy;
  else if (xaction == dnd.XdndActionMove)
    action = NSDragOperationCopy;
  else if (xaction == dnd.XdndActionLink) 
    action = NSDragOperationLink;
  else if (xaction == dnd.XdndActionAsk) 
    action = NSDragOperationGeneric;
  else if (xaction == dnd.XdndActionPrivate) 
    action = NSDragOperationPrivate;
  else
    action = NSDragOperationNone;
  return action;
}

int
XGErrorHandler(Display *display, XErrorEvent *err)
{
  int length = 1024;
  char buffer[length+1];

  XGetErrorText(display, err->error_code, buffer, length),
  [NSException raise: NSWindowServerCommunicationException
    format: @"X-Windows error - %s\n\
          on display: %s\n\
    		type: %d\n\
       serial number: %d\n\
	request code: %d\n",
	buffer,
    	XDisplayName(DisplayString(display)),
	err->type, err->serial, err->request_code];
  return 0;
}

static NSEvent*
process_key_event(XEvent* xEvent, XGContext* ctxt, NSEventType eventType);

static unsigned short
process_key_code(XEvent *xEvent, KeySym keysym, unsigned eventModifierFlags);

static unsigned
process_modifier_flags(unsigned int state);

@interface NSEvent (WindowHack)
- (void) _patchLocation: (NSPoint)loc;
@end

@implementation NSEvent (WindowHack)
- (void) _patchLocation: (NSPoint)loc
{
  location_point = loc;
}
@end


@interface XGContext (Private)
- (void) _checkFontDir;
- (void) _findXFont: (font_t *)font;
- (void) receivedEvent: (void*)data
                  type: (RunLoopEventType)type
                 extra: (void*)extra
               forMode: (NSString*)mode;
- (void) setupRunLoopInputSourcesForMode: (NSString*)mode; 
- (NSDate*) timedOutEvent: (void*)data
                     type: (RunLoopEventType)type
                  forMode: (NSString*)mode;
@end

#define XDPY (context->dpy)
#define XSCR (context->screen_number)

@implementation XGContext 

+ (Display*) currentXDisplay
{
  return [(XGContext*)[self currentContext] xDisplay];
}

- (RContextAttributes *) _getXDefaults
{
  int dummy;
  RContextAttributes *attribs;

  NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
  attribs = (RContextAttributes *)malloc(sizeof(RContextAttributes));

  attribs->flags = 0;
  if ([defaults boolForKey: @"NSDefaultVisual"])
    attribs->flags |= RC_DefaultVisual;
  if ((dummy = [defaults integerForKey: @"NSDefaultVisual"]))
    {
      attribs->flags |= RC_VisualID;
      attribs->visualid = dummy;
    }
  if ((dummy = [defaults integerForKey: @"NSColorsPerChannel"]))
    {
      attribs->flags |= RC_ColorsPerChannel;
      attribs->colors_per_channel = dummy;
    }
  /*
  if ((str = [defaults stringForKey: @"NSColormapStyle"]))
    {
      attribs->flags |= RC_ColormapStyle;
      if ([str isEqual: @"none"])
        attribs->colormap_style = no_colormap;
      else if ([str isEqual: @"four"])
        attribs->colormap_style = four_colormap;
      else if ([str isEqual: @"gray"])
        attribs->colormap_style = gray_colormap;
      else
        attribs->colormap_style = rgb_colormap;
    }
  if ([defaults boolForKey: @"NSPrivateColormap"])
    attribs->flags |= RC_PrivateColormap;
  */

  return attribs;
}

- _initXContext
{
  Display		*dpy;
  int			screen_number;
  NSString		*display_name;
  RContextAttributes	*attribs;
  XColor		testColor;
  unsigned char		r, g, b;
  
  display_name = [context_info objectForKey: @"DisplayName"];
  if (display_name)
    dpy = XOpenDisplay([display_name cString]);
  else
    dpy = XOpenDisplay(NULL);

  /* Use the fact that the screen number is specified like an extension
     e.g. hostname:0.1 */
  screen_number = [[display_name pathExtension] intValue];

  if (dpy == NULL)
    {
      char *dname = XDisplayName([display_name cString]);
      DPS_FATAL(DPSconfigurationerror, @"Unable to connect to X Server %s", 
		dname);
    }

  /* Get the visual information */
  attribs = NULL;
  //attribs = [self _getXDefaults];
  context = RCreateContext(dpy, screen_number, attribs);

  /*
   * If we have shared memory available, only use it when the XGPS-Shm
   * default is set to YES
   */
  if (context->attribs->use_shared_memory == True
    && [[NSUserDefaults standardUserDefaults] boolForKey: @"XGPS-Shm"] != YES)
    context->attribs->use_shared_memory = False;

  /* Create a window for initial drawing */
  /* This window is never mapped */
  appRootWindow = XCreateSimpleWindow(XDPY, 
				      RootWindow(XDPY, XSCR),
				      0, 0, 100, 100, 1, 1,
				      context->black);

  /*
   *	Crude tests to see if we can accelerate creation of pixels from
   *	8-bit red, green and blue color values.
   */
  if (context->depth == 16)
    {
      drawMechanism = XGDM_FAST16;
      r = 8;
      g = 9;
      b = 7;
      testColor.pixel = (((r << 5) + g) << 6) + b;
      XQueryColor(context->dpy, context->cmap, &testColor);
      if (((testColor.red >> 11) != r)
	|| ((testColor.green >> 11) != g)
	|| ((testColor.blue >> 11) != b))
	{
	  NSLog(@"WARNING - XGContext is unable to use the "
	    @"fast algorithm for writing to a 16-bit display on "
	    @"this host - perhaps you'd like to adjust the code "
	    @"to work ... and submit a patch.");
	  drawMechanism = XGDM_PORTABLE;
	}
    }
  else if (context->depth == 32)
    {
      drawMechanism = XGDM_FAST32;
      r = 32;
      g = 33;
      b = 31;
      testColor.pixel = (((r << 16) + g) << 8) + b;
      XQueryColor(context->dpy, context->cmap, &testColor);
      if (((testColor.red >> 8) != r)
	|| ((testColor.green >> 8) != g)
	|| ((testColor.blue >> 8) != b))
	{
	  NSLog(@"WARNING - XGContext is unable to use the "
	    @"fast algorithm for writing to a 32-bit display on "
	    @"this host - perhaps you'd like to adjust the code "
	    @"to work ... and submit a patch.");
	  drawMechanism = XGDM_PORTABLE;
	}
    }
  else
    {
      drawMechanism = XGDM_PORTABLE;
    }

  XSetErrorHandler(XGErrorHandler);

  return self;
}

- (id) initWithContextInfo: (NSDictionary *)info
{
  NSZone	*z = [self zone];

  alloc_mode = alloc_local;
  /* Initialize lists and stacks */
  windowList = [[NSMutableArray allocWithZone: z] initWithCapacity: 8];
  opstack =  NSZoneMalloc(z, sizeof(GSIArray_t));
  GSIArrayInitWithZoneAndCapacity((GSIArray)opstack, z, 2);
  gstack =  NSZoneMalloc(z, sizeof(GSIArray_t));
  GSIArrayInitWithZoneAndCapacity((GSIArray)gstack, z, 2);
  ulist  = [[NSMutableArray allocWithZone: z] initWithCapacity: 32];
  
  [self _initXContext];

  /* Create a default gstate */
  gstate = [CTXT_OBJ_ALLOC(XGGState) initWithDrawContext: self];

  [super initWithContextInfo: info];

  /*
   * Make self the current context 'cos some of our initialization needs it.
   */
  [isa setCurrentContext: self];

  [self setupRunLoopInputSourcesForMode: NSDefaultRunLoopMode]; 
  [self setupRunLoopInputSourcesForMode: NSConnectionReplyMode]; 
  [self setupRunLoopInputSourcesForMode: NSModalPanelRunLoopMode]; 
  [self setupRunLoopInputSourcesForMode: NSEventTrackingRunLoopMode]; 

  WM_STATE = XInternAtom(context->dpy, "WM_STATE", False);
  return self;
}

- (void) dealloc
{
  GSIArrayEmpty((GSIArray)opstack);
  NSZoneFree([self zone], opstack);
  GSIArrayEmpty((GSIArray)gstack);
  NSZoneFree([self zone], gstack);
  RELEASE(windowList);
 RELEASE(ulist);
  XDestroyWindow(XDPY, appRootWindow);
  XCloseDisplay(XDPY);
  [super dealloc];
}

- (void) setupRunLoopInputSourcesForMode: (NSString*)mode
{
  int		xEventQueueFd = XConnectionNumber(context->dpy);
  NSRunLoop	*currentRunLoop = [NSRunLoop currentRunLoop];

#if defined(LIB_FOUNDATION_LIBRARY)
  {
    id fileDescriptor = [[[NSPosixFileDescriptor alloc]
	initWithFileDescriptor: xEventQueueFd]
	autorelease];

    // Invoke limitDateForMode: to setup the current
    // mode of the run loop (the doc says that this
    // method and acceptInputForMode: beforeDate: are
    // the only ones that setup the current mode).

    [currentRunLoop limitDateForMode: mode];

    [fileDescriptor setDelegate: self];
    [fileDescriptor monitorFileActivity: NSPosixReadableActivity];
  }
#elif defined(NeXT_PDO)
  {
    id fileDescriptor = [[[NSFileHandle alloc]
	initWithFileDescriptor: xEventQueueFd]
	autorelease];

    [[NSNotificationCenter defaultCenter] addObserver: self
	selector: @selector(activityOnFileHandle: )
	name: NSFileHandleDataAvailableNotification
	object: fileDescriptor];
    [fileDescriptor waitForDataInBackgroundAndNotifyForModes:
	[NSArray arrayWithObject: mode]];
  }
#else
  [currentRunLoop addEvent: (void*)(gsaddr)xEventQueueFd
		      type: ET_RDESC
		   watcher: (id<RunLoopEvents>)self
		   forMode: mode];
#endif
}

#if LIB_FOUNDATION_LIBRARY
- (void) activity: (NSPosixFileActivities)activity
		posixFileDescriptor: (NSPosixFileDescriptor*)fileDescriptor
{
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];
}
#elif defined(NeXT_PDO)
- (void) activityOnFileHandle: (NSNotification*)notification
{
  id fileDescriptor = [notification object];
  id runLoopMode = [[NSRunLoop currentRunLoop] currentMode];

  [fileDescriptor waitForDataInBackgroundAndNotifyForModes:
	[NSArray arrayWithObject: runLoopMode]];
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];
}
#endif

- (void) receivedEvent: (void*)data
                  type: (RunLoopEventType)type
                 extra: (void*)extra
               forMode: (NSString*)mode
{
  static NSRect		xFrame;
  static Time		timeOfLastClick = 0;
  static unsigned long	timeOfLastMotion = 0;
  static BOOL		switchingFocusBetweenOurWindows = NO;
  static int		clickCount = 1;
  static unsigned int	eventFlags;
  static unsigned int	last_XEvent_state = -1;
  static XGWindow	*window = nil;
  static Window		lastXWin = None;
  static int		windowNumber;
  NSEvent		*e = nil;
  XEvent		xEvent;
  NSRect		new_frame;
  NSPoint		eventLocation;
  Window		xWin;
  XGWindow		*w;
  NSEventType		eventType;
  int			count;
  BOOL			inTrackingLoop = (mode == NSEventTrackingRunLoopMode);

  while ((count = XPending(context->dpy)) > 0)
    {
      // loop and grab all of the events from the X queue
      while (count-- > 0)
	{
	  NSDebugLog(@"Get next XWindows event\n");
	  XNextEvent(context->dpy, &xEvent);
	  switch (xEvent.type)
	    {
	      // mouse button events
	      case ButtonPress:
		NSDebugLog(@"ButtonPress: \
			    xEvent.xbutton.time %u timeOfLastClick %u \n",
			    xEvent.xbutton.time, timeOfLastClick);
	      // hardwired test for a double click
	      // default of 300 should be user set;
	      // under NS the windowserver does this
	      if (xEvent.xbutton.time < (unsigned long)(timeOfLastClick +300))
		clickCount++;
	      else
		clickCount = 1;		// reset the click count
	      timeOfLastClick = xEvent.xbutton.time;

	      eventType = (xEvent.xbutton.button == Button1 ?
			    NSLeftMouseDown : NSRightMouseDown);
	      if (xEvent.xbutton.state != last_XEvent_state)
		{
		  eventFlags = process_modifier_flags(xEvent.xbutton.state);
		  last_XEvent_state = xEvent.xbutton.state;
		}
	      // if pointer is grabbed use grab window
	      xWin = (grabWindow == 0) ? xEvent.xbutton.window : grabWindow;
	      if (xWin != lastXWin)
		{
		  window = [XGWindow _windowForXWindow: xWin];
		  windowNumber = [window windowNumber];
		  lastXWin = xWin;
		}
	      xFrame = [window xFrame];
	      eventLocation.x = xEvent.xbutton.x;
	      eventLocation.y = xFrame.size.height-xEvent.xbutton.y;

	      // create NSEvent
	      e = [NSEvent mouseEventWithType: eventType
				     location: eventLocation
				modifierFlags: eventFlags
				    timestamp: (NSTimeInterval)xEvent.xbutton.time
				 windowNumber: windowNumber
				      context: self
				  eventNumber: xEvent.xbutton.serial
				   clickCount: clickCount
				     pressure: 1.0];
	      break;

	    case ButtonRelease:
	      eventType = (xEvent.xbutton.button == Button1 ?
			    NSLeftMouseUp : NSRightMouseUp);
	      if (xEvent.xbutton.state != last_XEvent_state)
		{
		  eventFlags = process_modifier_flags(xEvent.xbutton.state);
		  last_XEvent_state = xEvent.xbutton.state;
		}
	      // if pointer is grabbed use grab window
	      xWin = (grabWindow == 0) ? xEvent.xbutton.window : grabWindow;
	      if (xWin != lastXWin)
		{
		  window = [XGWindow _windowForXWindow: xWin];
		  windowNumber = [window windowNumber];
		  lastXWin = xWin;
		}
	      xFrame = [window xFrame];
	      eventLocation.x = xEvent.xbutton.x;
	      eventLocation.y = xFrame.size.height- xEvent.xbutton.y;

	      e = [NSEvent mouseEventWithType: eventType
				     location: eventLocation
				modifierFlags: eventFlags
				    timestamp: (NSTimeInterval)xEvent.xbutton.time
				 windowNumber: windowNumber
				      context: self
				  eventNumber: xEvent.xbutton.serial
				   clickCount: clickCount
				     pressure: 1.0];
	      break;

	    case CirculateNotify:
	      // change to the stacking order
	      NSDebugLog(@"CirculateNotify\n");
	      break;

	    case CirculateRequest:
	      NSDebugLog(@"CirculateRequest\n");
	      break;

	    case ClientMessage:
	      {
		NSTimeInterval time;
		Atom protocols_atom;

		NSDebugLog(@"ClientMessage\n");
		w =[XGWindow _windowForXWindow: xEvent.xclient.window];

		protocols_atom = XInternAtom(context->dpy,
		  "WM_PROTOCOLS", False);
		if (xEvent.xclient.message_type == protocols_atom)
		  {
		    // check if WM is asking us to close a window
		    Atom delete_win_atom;
		    delete_win_atom = XInternAtom(context->dpy,
						  "WM_DELETE_WINDOW", False);
		    if (xEvent.xclient.data.l[0] == delete_win_atom)
		      {
			[w performClose: NSApp];
		      }
		  }
		else if (xEvent.xclient.message_type == dnd.XdndEnter)
		  {
		    Window source;
		    NSDebugLLog(@"NSDragging", @"  XdndEnter message\n");
		    source = XDND_ENTER_SOURCE_WIN(&xEvent);
		    eventLocation = NSMakePoint(0,0);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: 0
			        windowNumber: [w windowNumber]
			        context: self
			        subtype: GSAppKitDraggingEnter
			        data1: source
			        data2: 0];
		    /* If this is a non-local drag, set the dragInfo */
		    if ([XGWindow _windowForXWindow: source] == nil)
		      [[XGDragView sharedDragView] setDragInfoFromEvent: e];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndPosition)
		  {
		    Window source;
		    Atom action;
		    NSDebugLLog(@"NSDragging", @"  XdndPosition message\n");
		    source = XDND_POSITION_SOURCE_WIN(&xEvent);
		    xFrame = [window xFrame];
		    eventLocation.x = XDND_POSITION_ROOT_X(&xEvent)
		      - NSMinX(xFrame);
		    eventLocation.y = XDND_POSITION_ROOT_Y(&xEvent)
		      - NSMinY(xFrame);
		    eventLocation.y = xFrame.size.height - eventLocation.y;
		    time = XDND_POSITION_TIME(&xEvent);
		    action = XDND_POSITION_ACTION(&xEvent);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: time
			        windowNumber: [w windowNumber]
			        context: self
			        subtype: GSAppKitDraggingUpdate
			        data1: source
			        data2: GSDragOperationForAction(action)];
		    /* If this is a non-local drag, set the dragInfo */
		    if ([XGWindow _windowForXWindow: source] == nil)
		      [[XGDragView sharedDragView] setDragInfoFromEvent: e];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndStatus)
		  {
		    Window target;
		    Atom action;
		    NSDebugLLog(@"NSDragging", @"  XdndStatus message\n");
		    target = XDND_STATUS_TARGET_WIN(&xEvent);
		    eventLocation = NSMakePoint(0, 0);
		    action = XDND_STATUS_ACTION(&xEvent);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: 0
			        windowNumber: [w windowNumber]
			        context: self
			        subtype: GSAppKitDraggingStatus
			        data1: target
			        data2: GSDragOperationForAction(action)];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndLeave)
		  {
		    Window source;
		    NSDebugLLog(@"NSDragging", @"  XdndLeave message\n");
		    source = XDND_LEAVE_SOURCE_WIN(&xEvent);
		    eventLocation = NSMakePoint(0, 0);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: 0
			        windowNumber: [w windowNumber]
			        context: self
			        subtype: GSAppKitDraggingExit
			        data1: 0
			        data2: 0];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndDrop)
		  {
		    Window source;
		    NSDebugLLog(@"NSDragging", @"  XdndDrop message\n");
		    source = XDND_DROP_SOURCE_WIN(&xEvent);
		    eventLocation = NSMakePoint(0, 0);
		    time = XDND_DROP_TIME(&xEvent);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: time
			        windowNumber: [w windowNumber]
			        context: self
			        subtype: GSAppKitDraggingDrop
			        data1: source
			        data2: 0];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndFinished)
		  {
		    Window target;
		    NSDebugLLog(@"NSDragging", @"  XdndFinished message\n");
		    target = XDND_FINISHED_TARGET_WIN(&xEvent);
		    eventLocation = NSMakePoint(0, 0);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: 0
			        windowNumber: [w windowNumber]
			        context: self
			        subtype: GSAppKitDraggingFinished
			        data1: target
			        data2: 0];
		  }
	
	      }
	      break;

	      case ColormapNotify:
		// colormap attribute
		NSDebugLog(@"ColormapNotify\n");
		break;

	      // the window has been resized, change the width and height
	      // and update the window so the changes get displayed
	      case ConfigureNotify:
		NSDebugLog(@"ConfigureNotify\n");
		new_frame.origin.x = (float)xEvent.xconfigure.x;
		new_frame.origin.y = (float)xEvent.xconfigure.y;
		new_frame.size.width = (float)xEvent.xconfigure.width;
		new_frame.size.height = (float)xEvent.xconfigure.height;
		NSDebugLog(@"New frame %f %f %f %f\n",
			    new_frame.origin.x, new_frame.origin.y,
			    new_frame.size.width, new_frame.size.height);
		NSDebugLog(@"border_width %d\n",
			    xEvent.xconfigure.border_width);
		w = [XGWindow _windowForXWindow: xEvent.xconfigure.window];
		if (w != nil)
		  {
		    NSTimeInterval	t = (NSTimeInterval)timeOfLastMotion;
		    NSRect		xFrame = [w xFrame];
		    BOOL		resized = NO;
		    int			width = xEvent.xconfigure.width;
		    int			height = xEvent.xconfigure.height;

		    [w xSetFrameFromXFrame: new_frame];

		    if (xFrame.size.width != new_frame.size.width
		      || xFrame.size.height != new_frame.size.height)
		      {
			e = [NSEvent otherEventWithType: NSAppKitDefined
					       location: eventLocation
					  modifierFlags: eventFlags
					      timestamp: t
					   windowNumber: [w windowNumber]
						context: self
						subtype: GSAppKitWindowResized
						  data1: width
						  data2: height];
			resized = YES;
			[w sendEvent: e];
			e = nil;
		      }
		    if (xFrame.origin.x != new_frame.origin.x
		      || xFrame.origin.y != new_frame.origin.y)
		      {
			int	xScreen;
			int	h;
			int	x = xEvent.xconfigure.x;
			int	y = xEvent.xconfigure.y;
			Window	r;
			Window	qParent;
			Window	*qChildren;
			unsigned	qNum;

			xScreen = [(XGScreen *)[NSScreen mainScreen] xScreen];
			h = DisplayHeight(XDPY, xScreen);

			XQueryTree(XDPY, [w xWindow],
			  &r, &qParent, &qChildren, &qNum);
			XFree(qChildren);

			if (qParent != r)
			  {
			    int		X;
			    int		Y;
			    int		b;
			    int		d;

			    /*
			     * get info relative to parent window in case the
			     * window manager has messed things up.
			     */
			    XGetGeometry(XDPY, [w xWindow], &r,
			      &x, &y, &width, &height, &b, &d);
			    /*
			     * Window has a title bar so have X server
			     * translate to display coordinates first,
			     * then convert to local.
			     */
			    XTranslateCoordinates(XDPY, qParent, 
			      appRootWindow, x, y, &X, &Y, &r);
			    x = X;
			    y = Y;
			  }

			y = h - (y + height);

			if (e != nil)
			  [event_queue addObject: e];
			e = [NSEvent otherEventWithType: NSAppKitDefined
					       location: eventLocation
					  modifierFlags: eventFlags
					      timestamp: t
					   windowNumber: [w windowNumber]
						context: self
						subtype: GSAppKitWindowMoved
						  data1: x
						  data2: y];
			[w sendEvent: e];
			e = nil;
		      }
                    if (resized)
                      [w update];
		  }
		NSDebugLog(@"after ConfigureNotify\n");
		break;

	      // same as ConfigureNotify but we get this event
	      // before the change has actually occurred
	      case ConfigureRequest:
		NSDebugLog(@"ConfigureRequest\n");
		break;

	      // a window has been created
	      case CreateNotify:
		NSDebugLog(@"CreateNotify\n");
		break;

	      // a window has been destroyed
	      case DestroyNotify:
		NSDebugLog(@"DestroyNotify\n");
		break;

	      // when the pointer enters a window
	      case EnterNotify:
		NSDebugLog(@"EnterNotify\n");
		break;

	      // when the pointer leaves a window
	      case LeaveNotify:
		NSDebugLog(@"LeaveNotify\n");
		break;

	      // the visibility of a window has changed
	      case VisibilityNotify:
		NSDebugLog(@"VisibilityNotify\n");
		break;

	      // a portion of the window has become visible and
	      // we must redisplay it
	      case Expose:
		{
		  XRectangle rectangle;
		  static XGWindow *exposeWin;
		  static Window lastExposeXWin = None;

		  NSDebugLog(@"Expose\n");

		  if (xEvent.xbutton.window != lastExposeXWin)
		    {
		      lastExposeXWin = xEvent.xbutton.window;
		      exposeWin = [XGWindow _windowForXWindow: lastExposeXWin];
		    }
		  rectangle.x = xEvent.xexpose.x;
		  rectangle.y = xEvent.xexpose.y;
		  rectangle.width = xEvent.xexpose.width;
		  rectangle.height = xEvent.xexpose.height;
		  [exposeWin _addExposedRectangle: rectangle];

		  if (xEvent.xexpose.count == 0)
		    [exposeWin _processExposedRectangles];
		  break;
		}

	      // keyboard focus entered a window
	      case FocusIn:
		NSDebugLog(@"FocusIn\n");
		[NSApp activateIgnoringOtherApps: YES];
		w = [XGWindow _windowForXWindow: xEvent.xfocus.window];
		[w makeKeyWindow];	
		break;

	      // keyboard focus left a window
	      case FocusOut:
		NSDebugLog(@"FocusOut\n");
		if (1)
		  {
		    Window	fw;
		    int		rev;

		    /*
		     * See where the focus has moved to -
		     * If it has gone to 'none' or 'PointerRoot' then 
		     * it's not one of ours.
		     * If it has gone to our root window - use the icon window.
		     * If it has gone to a window - we see if it is one of ours.
		     */
		    XGetInputFocus(xEvent.xfocus.display, &fw, &rev);
		    if (fw == None || fw == PointerRoot)
		      {
			w = nil;
		      }
		    else if (fw == appRootWindow)
		      {
			w = iconWindow;
			XSetInputFocus(context->dpy, [w xWindow],
			  RevertToParent, CurrentTime);
		      }
		    else
		      {
			w = [XGWindow _windowForXWindow: fw];
		      }
#if 0
		    fprintf(stderr, "Focus win:%x,%x,%x mode:%x detail:%x\n",
			    xEvent.xfocus.window, fw, w,
			    xEvent.xfocus.mode,
			    xEvent.xfocus.detail);
#endif
		    if (w == nil)
		      [NSApp deactivate]; 
		  }
		break;

	      case GraphicsExpose:
		NSDebugLog(@"GraphicsExpose\n");
		break;

	      case NoExpose:
		NSDebugLog(@"NoExpose\n");
		break;

	      // window is moved because of a change in the size of its parent
	      case GravityNotify:
		NSDebugLog(@"GravityNotify\n");
		break;

	      // a key has been pressed
	      case KeyPress:
		e = process_key_event (&xEvent, self, NSKeyDown);
		break;

	      // a key has been released
	      case KeyRelease:
		e = process_key_event (&xEvent, self, NSKeyUp);
		break;

	      // reports the state of the keyboard when pointer or
	      // focus enters a window
	      case KeymapNotify:
		NSDebugLog(@"KeymapNotify\n");
		break;

	      // when a window changes state from ummapped to
	      // mapped or vice versa
	      case MapNotify:
		NSDebugLog(@"MapNotify\n");
		w = [XGWindow _windowForXWindow: xEvent.xunmap.window];
		[(XGWindow *)w _setVisible: YES];
		break;

	      // Window is no longer visible.
	      case UnmapNotify:
		NSDebugLog(@"UnmapNotify\n");
		w = [XGWindow _windowForXWindow: xEvent.xunmap.window];
		[(XGWindow *)w _setVisible: NO];
		break;

	      // like MapNotify but occurs before the request is carried out
	      case MapRequest:
		NSDebugLog(@"MapRequest\n");
		break;

	      // keyboard or mouse mapping has been changed by another client
	      case MappingNotify:
		NSDebugLog(@"MappingNotify\n");
		break;

	      case MotionNotify:
		{
		  BOOL	shouldMakeEvent = NO;

		  /*
		   * the mouse has moved regulate the translation of X motion
		   * events in order to avoid saturating the AppKit with
		   * unneeded NSEvents - needs fixing!
		   */
		  if ((inTrackingLoop && count == 1)
		    || (xEvent.xmotion.time > (unsigned)(timeOfLastMotion+10)))
		    {
		      shouldMakeEvent = YES;
		    }
		      shouldMakeEvent = YES;

		  timeOfLastMotion = xEvent.xmotion.time;
		  if (shouldMakeEvent)
		    {
		      unsigned int	state = xEvent.xmotion.state;

		      if (state & Button1Mask)
			{
			  eventType = NSLeftMouseDragged;
			}
		      else if (state & Button2Mask)
			{
			  eventType = NSRightMouseDragged;
			}
		      else
			{
			  eventType = NSMouseMoved;
			}

		      if (state != last_XEvent_state)
			{
			  eventFlags = process_modifier_flags(state);
			  last_XEvent_state = state;
			}

		      // if pointer is grabbed use grab window instead
		      xWin = (grabWindow == 0)
			? xEvent.xmotion.window : grabWindow;
		      if (xWin != lastXWin)
			{
			  window = [XGWindow _windowForXWindow: xWin];
			  windowNumber = [window windowNumber];
			  lastXWin = xWin;
			  xFrame = [window xFrame];
			}

		      eventLocation = NSMakePoint(xEvent.xmotion.x,
				    xFrame.size.height - xEvent.xmotion.y);

		      e = [NSEvent mouseEventWithType: eventType
					     location: eventLocation
					modifierFlags: eventFlags
					    timestamp: (NSTimeInterval)timeOfLastMotion
					 windowNumber: windowNumber
					      context: self
					  eventNumber: xEvent.xmotion.serial
					   clickCount: 1
					     pressure: 1.0];
		    }

		  NSDebugLog(@"MotionNotify\n");
		  break;
		}

	      // a window property has changed or been deleted
	      case PropertyNotify:
		NSDebugLog(@"PropertyNotify\n");
                if (WM_STATE == xEvent.xproperty.atom)
		  {
                    switchingFocusBetweenOurWindows = YES;
		  }
		break;

	      // a client successfully reparents a window
	      case ReparentNotify:
		NSDebugLog(@"ReparentNotify\n");
		NSDebugLog(@"parent offset %f %f\n", xEvent.xreparent.x,
					      xEvent.xreparent.y);
		break;

	      // another client attempts to change the size of a window
	      case ResizeRequest:
		NSDebugLog(@"ResizeRequest\n");
		break;

	      // events dealing with the selection
	      case SelectionClear:
	      case SelectionNotify:
	      case SelectionRequest:
		NSDebugLog(@"Selection*\n");
		break;

	      // We shouldn't get here unless we forgot to trap an event above
	      default:
		NSLog(@"Received an untrapped event\n");
		break;
	    }
	  if (e)
	    [event_queue addObject: e];
	  e = nil;
	}
    }
}

	// TODO: correctly map key events to
	// NSEvents. This can be made when 16-bit
	// character strings will be available.
static NSEvent*
process_key_event (XEvent* xEvent, XGContext* context, NSEventType eventType)
{
  char	buf[256];
  int	count;
  XComposeStatus compose;
  NSString	*keys, *ukeys;
  KeySym	keysym;
  NSPoint	eventLocation;
  unsigned short keyCode;
  unsigned int	eventFlags;
  NSEvent	*event = nil;
  NSApplication *theApp = [NSApplication sharedApplication];
  XGWindow	*window;
  NSRect xFrame = NSZeroRect;

  NSDebugLog(@"Process key event");

  eventFlags = process_modifier_flags(xEvent->xkey.state);

  count = XLookupString((XKeyEvent *)xEvent, buf, 256, &keysym, &compose);

  // Make sure that the string is properly terminated
  if (count > 255)
    buf[255] = '\0';
  else
    {
      if (count < 1)
	buf[0] = '\0';
      else
	buf[count] = '\0';
    }

  window = (XGWindow*)[theApp keyWindow];
  if (window)
    xFrame = [window xFrame];

  eventLocation.x = xEvent->xbutton.x;
  eventLocation.y = xFrame.size.height - xEvent->xbutton.y;
  NSDebugLog (@"keysym=%d, xLocation = (%d, %d), userLocation = (%f, %f)",
	      keysym, xEvent->xbutton.x, (int)(xFrame.size.height -
	      xEvent->xbutton.y), eventLocation.x, eventLocation.y);

  keyCode = process_key_code(xEvent, keysym, eventFlags);

  ukeys = [NSString stringWithCString: buf];
  keys = ukeys;		/* Stupid implementation (to be replaced) */

  if (keyCode == 0 && *buf == '\0' && context->lastKeyFlags != eventFlags)
    eventType = NSFlagsChanged;
  context->lastKeyFlags = eventFlags;

  event = [NSEvent keyEventWithType: eventType
			   location: eventLocation
		      modifierFlags: eventFlags
			  timestamp: (NSTimeInterval)xEvent->xkey.time
		       windowNumber: [window windowNumber]
			    context: context
			 characters: keys
	charactersIgnoringModifiers: ukeys
			  isARepeat: NO
			    keyCode: keyCode];

  return event;
}

static unsigned short
process_key_code(XEvent *xEvent, KeySym keysym, unsigned int eventModifierFlags)
{
  unsigned short keyCode = 0;

  if ((keysym == XK_Return) || (keysym == XK_KP_Enter)
      || (keysym == XK_Linefeed))
    {				// do nothing for now
      keyCode = 0x0d;
    }
  else if ((keysym >= XK_Shift_L) && (keysym <= XK_Hyper_R))
    {
      // The shift button is depressed.  This should have
      // already been detected in the process_modifier_flags
      // function.  Therefore, nothing is done here...
    }
  if ((keysym >= XK_F1) && (keysym <= XK_F35))
    {
      // if a function key was pressed
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;

      switch (xEvent->xkey.keycode)
	{
	  case XK_F1: keyCode = NSF1FunctionKey; break;
	  case XK_F2: keyCode = NSF2FunctionKey; break;
	  case XK_F3: keyCode = NSF3FunctionKey; break;
	  case XK_F4: keyCode = NSF4FunctionKey; break;
	  case XK_F5: keyCode = NSF5FunctionKey; break;
	  case XK_F6: keyCode = NSF6FunctionKey; break;
	  case XK_F7: keyCode = NSF7FunctionKey; break;
	  case XK_F8: keyCode = NSF8FunctionKey; break;
	  case XK_F9: keyCode = NSF9FunctionKey; break;
	  case XK_F10: keyCode = NSF10FunctionKey; break;
	  case XK_F11: keyCode = NSF11FunctionKey; break;
	  case XK_F12: keyCode = NSF12FunctionKey; break;
	  case XK_F13: keyCode = NSF13FunctionKey; break;
	  case XK_F14: keyCode = NSF14FunctionKey; break;
	  case XK_F15: keyCode = NSF15FunctionKey; break;
	  case XK_F16: keyCode = NSF16FunctionKey; break;
	  case XK_F17: keyCode = NSF17FunctionKey; break;
	  case XK_F18: keyCode = NSF18FunctionKey; break;
	  case XK_F19: keyCode = NSF19FunctionKey; break;
	  case XK_F20: keyCode = NSF20FunctionKey; break;
	  case XK_F21: keyCode = NSF21FunctionKey; break;
	  case XK_F22: keyCode = NSF22FunctionKey; break;
	  case XK_F23: keyCode = NSF23FunctionKey; break;
	  case XK_F24: keyCode = NSF24FunctionKey; break;
	  case XK_F25: keyCode = NSF25FunctionKey; break;
	  case XK_F26: keyCode = NSF26FunctionKey; break;
	  case XK_F27: keyCode = NSF27FunctionKey; break;
	  case XK_F28: keyCode = NSF28FunctionKey; break;
	  case XK_F29: keyCode = NSF29FunctionKey; break;
	  case XK_F30: keyCode = NSF30FunctionKey; break;
	  case XK_F31: keyCode = NSF31FunctionKey; break;
	  case XK_F32: keyCode = NSF32FunctionKey; break;
	  case XK_F33: keyCode = NSF33FunctionKey; break;
	  case XK_F34: keyCode = NSF34FunctionKey; break;
	  case XK_F35: keyCode = NSF35FunctionKey; break;

	  default: 			// do nothing
	}
    }
  else
    {
      if (keysym == XK_BackSpace)
	{
	  keyCode = NSBackspaceKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Delete)
	{
	  keyCode = NSDeleteFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Home)
	{
	  keyCode = NSHomeFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Left)
	{
	  keyCode = NSLeftArrowFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Up)
	{
	  keyCode = NSUpArrowFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Right)
	{
	  keyCode = NSRightArrowFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Down)
	{
	  keyCode = NSDownArrowFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Prior)
	{
	  keyCode = NSPrevFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#ifndef NeXT
	}
      else if (keysym == XK_Page_Up)
	{
	  keyCode = NSPageUpFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#endif
	}
      else if (keysym == XK_Next)
	{
	  keyCode = NSNextFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#ifndef NeXT
	}
      else if (keysym == XK_Page_Down)
	{
	  keyCode = NSPageDownFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#endif
	}
      else if (keysym == XK_End)
	{
	  keyCode = NSEndFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Begin)
	{
	  keyCode = NSBeginFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Select)
	{
	  keyCode = NSSelectFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Print)
	{
	  keyCode = NSPrintFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Execute)
	{
	  keyCode = NSExecuteFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Insert)
	{
	  keyCode = NSInsertFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Undo)
	{
	  keyCode = NSUndoFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Redo)
	{
	  keyCode = NSRedoFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Menu)
	{
	  keyCode = NSMenuFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Find)
	{
	  keyCode = NSFindFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Help)
	{
	  keyCode = NSHelpFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Break)
	{
	  keyCode = NSBreakFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Mode_switch)
	{
	  keyCode = NSModeSwitchFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#ifndef NeXT
	}
      else if (keysym == XK_Sys_Req)
	{
	  keyCode = NSSysReqFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#endif
	}
      else if (keysym == XK_Scroll_Lock)
	{
	  keyCode = NSScrollLockFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Pause)
	{
	  keyCode = NSPauseFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Clear)
	{
	  keyCode = NSClearDisplayFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if ((keysym == XK_Shift_L) || (keysym == XK_Shift_R))
	{
	  eventModifierFlags = eventModifierFlags
	    | NSFunctionKeyMask | NSShiftKeyMask;
	}
      else if ((keysym == XK_Control_L) || (keysym == XK_Control_R))
	{
	  eventModifierFlags = eventModifierFlags
	    | NSFunctionKeyMask | NSControlKeyMask;
	}
      else if (keysym == XK_Alt_R)
	{
	  eventModifierFlags = eventModifierFlags
	    | NSFunctionKeyMask | NSAlternateKeyMask;
	}
      else if (keysym == XK_Alt_L)
	{
	  eventModifierFlags = eventModifierFlags
	    | NSFunctionKeyMask | NSCommandKeyMask;
	}
      else if (keysym == XK_Tab)
	{
	  keyCode = 0x09;
	}
    }
  // If the key press originated from the key pad
  if ((keysym >= XK_KP_Space) && (keysym <= XK_KP_9))
    {
      eventModifierFlags = eventModifierFlags | NSNumericPadKeyMask;

      switch (keysym)
	{
	  case XK_KP_F1:        keyCode = NSF1FunctionKey;         break;
	  case XK_KP_F2:        keyCode = NSF2FunctionKey;         break;
	  case XK_KP_F3:        keyCode = NSF3FunctionKey;         break;
	  case XK_KP_F4:        keyCode = NSF4FunctionKey;         break;
#ifndef NeXT
	  case XK_KP_Home:      keyCode = NSHomeFunctionKey;       break;
	  case XK_KP_Left:      keyCode = NSLeftArrowFunctionKey;  break;
	  case XK_KP_Up:        keyCode = NSUpArrowFunctionKey;    break;
	  case XK_KP_Right:     keyCode = NSRightArrowFunctionKey; break;
	  case XK_KP_Down:      keyCode = NSDownArrowFunctionKey;  break;
	  case XK_KP_Prior:     keyCode = NSPrevFunctionKey;       break;
//		case XK_KP_Page_Up:   keyCode = NSPageUpFunctionKey;     break;
	  case XK_KP_Next:      keyCode = NSNextFunctionKey;       break;
//		case XK_KP_Page_Down: keyCode = NSPageDownFunctionKey;   break;
	  case XK_KP_End:       keyCode = NSEndFunctionKey;        break;
	  case XK_KP_Begin:     keyCode = NSBeginFunctionKey;      break;
	  case XK_KP_Insert:    keyCode = NSInsertFunctionKey;     break;
	  case XK_KP_Delete:    keyCode = NSDeleteFunctionKey;     break;
#endif
	  default:  break;  /* Nothing to do */
	}
    }

  if (((keysym >= XK_KP_Space) && (keysym <= XK_KP_9))
    || ((keysym >= XK_space) && (keysym <= XK_asciitilde)))
    {
    }					// Not processed

  return keyCode;
}

// process_modifier_flags() determines which modifier keys (Command, Control,
// Shift,  and so forth) were held down while the event occured.
static unsigned int
process_modifier_flags(unsigned int state)
{
  unsigned int eventModifierFlags = 0;

  if (state & ControlMask)
    eventModifierFlags = eventModifierFlags | NSControlKeyMask;

  if (state & ShiftMask)
    eventModifierFlags = eventModifierFlags | NSShiftKeyMask;

  if (state & Mod1Mask)
    {
      eventModifierFlags = eventModifierFlags | NSAlternateKeyMask;
      NSDebugLog (@"setButtonModifierFlags(): Mod1Mask\n");
    }

  if (state & Mod2Mask)
    {
      eventModifierFlags = eventModifierFlags | NSCommandKeyMask;
      NSDebugLog (@"setButtonModifierFlags(): Mod2Mask\n");
    }

  if (state & Mod3Mask)
    {
      eventModifierFlags = eventModifierFlags | NSAlphaShiftKeyMask;
      NSDebugLog (@"setButtonModifierFlags(): Mod3Mask\n");
    }

  if (state & Mod4Mask)
    {
      eventModifierFlags = eventModifierFlags | NSHelpKeyMask;
      NSDebugLog (@"setButtonModifierFlags(): Mod4Mask\n");
    }

  if (state & Mod5Mask)
    {
      eventModifierFlags = eventModifierFlags | NSAlternateKeyMask;
      NSDebugLog (@"setButtonModifierFlags(): Mod5Mask\n");
    }

  return eventModifierFlags;
}

- (NSDate*) timedOutEvent: (void*)data
                     type: (RunLoopEventType)type
                  forMode: (NSString*)mode
{
  return nil;
}

- (XGDrawMechanism) drawMechanism
{
  return drawMechanism;
}

- (BOOL)isDrawingToScreen
{
  return YES;
}

- (RContext *) xrContext
{
  return context;
}

- (XGGState *) xrCurrentGState
{
  return gstate;
}

- (XGGState *) xrGStateWithIdent: (int)gst
{
  XGGState *g;
  [self DPSexecuserobject: gst];
  ctxt_pop(g, opstack, XGGState);
  return g;
}

- (Region)viewclipRegion
{
  return context->viewclip;
}

- (void) setViewclipRegion: (Region)region;
{
  if (context->viewclip)
    XDestroyRegion(context->viewclip);
  context->viewclip = region;
}

- (Display *) xDisplay
{
  return context->dpy;
}

- (void) setXDisplay: (Display *)xdisplay
{
  NSLog(@"Warning: resetting X display\n");
  XDPY = xdisplay;
  /* FIXME: Should update visualinfo, colormaps, etc? */
}

- (Window) xDisplayRootWindow
{
  return RootWindow(XDPY, XSCR);
}

- (Window) xAppRootWindow
{
  return appRootWindow;
}

- (void) flush
{
  XFlush(XDPY);
}

- (BOOL) _setFrame: (NSRect)rect forWindow: (int)winNum
{
  int			xScreen;
  NSAutoreleasePool	*arp;
  NSRect		xVal;
  NSRect		last;
  NSEvent		*event;
  NSDate		*limit;
  NSMutableArray	*tmpQueue;
  unsigned		pos;
  float			xdiff;
  float			ydiff;
  XGWindow		*window;
  XSizeHints            siz_hints;
  NSRect		frame;
  BOOL			resize = NO;
  BOOL			move = NO;

  window = (XGWindow*)[XGWindow _windowWithTag: winNum];
  frame = [window frame];
  if (NSEqualRects(rect, frame) == YES)
    {
      return (NO);
    }
  if (NSEqualSizes(rect.size, frame.size) == NO)
    {
      resize = YES;
    }
  if (NSEqualPoints(rect.origin, frame.origin) == NO)
    {
      move = YES;
    }
  xdiff = rect.origin.x - frame.origin.x;
  ydiff = rect.origin.y - frame.origin.y;

  xScreen = [(XGScreen *)[NSScreen mainScreen] xScreen];
  xVal.origin.x = rect.origin.x;
  xVal.origin.y = rect.origin.y + rect.size.height;
  xVal.origin.y = (float)DisplayHeight(XDPY, xScreen) - xVal.origin.y;
  xVal.size.width = rect.size.width;
  xVal.size.height = rect.size.height;

  last = [window xFrame];

  XMoveResizeWindow (XDPY, [window xWindow],
    (int)xVal.origin.x, (int)xVal.origin.y,
    (int)xVal.size.width, (int)xVal.size.height);
  siz_hints.x = xVal.origin.x;
  siz_hints.y = xVal.origin.y;
  XSetNormalHints(XDPY, [window xWindow], &siz_hints);

  /*
   * Having told the X system to move the window, we must now
   * wait for it to actually do the job.
   */
  XSync(XDPY, 0);

  /*
   * Soak up any X events that have arrived.
   */
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];

  /*
   * Now massage all the events currently in the queue to make sure
   * mouse locations in our window are adjusted as necessary.
   */
  arp = [NSAutoreleasePool new];
  limit = [NSDate distantPast];	/* Don't wait for new events.	*/
  tmpQueue = [NSMutableArray arrayWithCapacity: 8];
  for (;;)
    {
      NSEventType	type;

      event = DPSGetEvent(self, NSAnyEventMask, limit,
	NSEventTrackingRunLoopMode);
      if (event == nil)
	break;
      type = [event type];
      if (type == NSAppKitDefined && [event windowNumber] == winNum)
	{
	  GSAppKitSubtype	sub = [event subtype];

	  /*
	   *	Window movement or resize events for the window we are
	   *	watching are posted immediately, so they can take effect.
	   */
	  if (sub == GSAppKitWindowMoved || sub == GSAppKitWindowResized)
	    {
	      [window sendEvent: event];
              if (sub == GSAppKitWindowResized)
                [window update];
	    }
	  else
	    {
	      [tmpQueue addObject: event];
	    }
	}
      else if (type != NSPeriodic && type != NSLeftMouseDragged
	&& type != NSRightMouseDragged && type != NSMouseMoved)
	{
	  /*
	   * Save any events that arrive before our window is moved - excepting
	   * periodic events (which we can assume will be outdated) and mouse
	   * movement events (which might flood us).
	   */
	  [tmpQueue addObject: event];
	}
      if (NSEqualRects(rect, [window frame]) == YES)
	{
	  break;
	}
      if (NSEqualRects(last, [window xFrame]) == NO)
	{
	  NSDebugLog(@"From: %@\nWant %@\nGot %@",
	    NSStringFromRect(last),
	    NSStringFromRect(xVal),
	    NSStringFromRect([window xFrame]));
	  last = [window xFrame];
	}
    }
  /*
   * If we got any events while waiting for the window movement, we
   * may need to adjust their locations to match the new window position.
   */
  pos = [tmpQueue count];
  while (pos-- > 0) 
    {
      event = [tmpQueue objectAtIndex: pos];
      if ([event windowNumber] == winNum)
	{
	  NSPoint	loc = [event locationInWindow];

	  loc.x -= xdiff;
	  loc.y -= ydiff;
	  [event _patchLocation: loc];
	}
      DPSPostEvent(self, event, YES);
    }
  [arp release];
  /*
   * Failsafe - if X hasn't told us it has moved/resized the window, we
   * fake the notification.
   */
  if (NSEqualRects([window frame], rect) == NO)
    {
      NSEvent	*e;
      int	width = xVal.size.width;
      int	height = xVal.size.height;
      int	x = xVal.origin.x;
      int	y = xVal.origin.y;
      int	xScreen;
      int	h;
      Window	r;
      Window	qParent;
      Window	*qChildren;
      unsigned	qNum;

      xScreen = [(XGScreen *)[NSScreen mainScreen] xScreen];
      h = DisplayHeight(XDPY, xScreen);

      XQueryTree(XDPY, [window xWindow],
	&r, &qParent, &qChildren, &qNum);

      xScreen = [(XGScreen *)[NSScreen mainScreen] xScreen];
      h = DisplayHeight(XDPY, xScreen);

      if (qParent != r)
	{
	  int		X;
	  int		Y;
	  /*
	   * window has a title bar so have X server translate from display
	   * coordinates to parent window coordinates.
	   */
	  XTranslateCoordinates(XDPY, RootWindow(XDPY, xScreen),
	    [window xWindow], x, y, &X, &Y, &r);
	  xVal.origin.x = X;
	  xVal.origin.y = Y;
	}
      [window xSetFrameFromXFrame: xVal];
      if (resize)
	{
	  e = [NSEvent otherEventWithType: NSAppKitDefined
				 location: NSZeroPoint
			    modifierFlags: 0
				timestamp: 0
			     windowNumber: winNum
				  context: self
				  subtype: GSAppKitWindowResized
				    data1: width
				    data2: height];
	  [window sendEvent: e];
	}
      if (move)
	{
	  y = h - (y + height);

	  e = [NSEvent otherEventWithType: NSAppKitDefined
				 location: NSZeroPoint
			    modifierFlags: 0
				timestamp: 0
			     windowNumber: winNum
				  context: self
				  subtype: GSAppKitWindowMoved
				    data1: x
				    data2: y];
	  [window sendEvent: e];
	}
    }
  if (resize)
    {
      [window update];
    }
  return YES;
}

- (void) _orderWindow: (NSWindowOrderingMode)place
           relativeTo: (int)otherWin
            forWindow: (int)winNum
{
  XGWindow	*window;
  Window	xWin;
  Display	*xDpy;
  XSizeHints	hints;
  NSRect	frame;
  int		xScrn;
  int		level;

  window = (XGWindow*)[XGWindow _windowWithTag: winNum];
  if (window == nil)
    return;

  xWin = [window xWindow];
  xDpy = [self xDisplay];
  frame = [window frame];
  level = [window level];
  xScrn = [(XGScreen *)[NSScreen mainScreen] xScreen];

  hints.flags = PPosition | USPosition;
  hints.x = frame.origin.x;
  hints.y = DisplayHeight(xDpy, xScrn) - (frame.origin.y + frame.size.height);

  if (place != NSWindowOut)
    {
      /*
       * At this point, we should make sure that the current window-level
       * is set in the X server, so it can stack windows correctly.
       * Unfortunately, I don't know how to do this.
       */
    }

  switch (place)
    {
      case NSWindowBelow:
	XLowerWindow(xDpy, xWin);
	XMapWindow(xDpy, xWin);
	XMoveWindow(xDpy, xWin, hints.x, hints.y);
	XSetNormalHints(xDpy, xWin, &hints);
	break;

      case NSWindowAbove:
	XMapRaised(xDpy, xWin);
	XMoveWindow(xDpy, xWin, hints.x, hints.y);
	XSetNormalHints(xDpy, xWin, &hints);
	break;

      case NSWindowOut:
	XMoveWindow(xDpy, xWin, hints.x, hints.y);
	XSetNormalHints(xDpy, xWin, &hints);
	XUnmapWindow(xDpy, xWin);
	break;
    }
  XSync(XDPY, 0);

  /*
   * Now do general window management stuff.
   */
  if (place == NSWindowOut)
    {
      if ([NSApp isActive])
	{
	  if ([window isKeyWindow])
	    {
	      [window resignKeyWindow];
	    }
	  if ([window isMainWindow])
	    {
	      [window resignMainWindow];
	    }
	}
    }
  else
    {
      [[window contentView] setNeedsDisplay: YES];
      [window update];
    }
  if ([windowList indexOfObjectIdenticalTo: window] == NSNotFound)
    {
      [windowList addObject: window];
    }
}

/* Drag and Drop */
- (void) _resetDragTypes: (NSArray*)types toWindow: (int)winNum
{
  int count, i;
  Atom *typelist;
  XGWindow  *window;
  NSString *type;
  NSCountedSet *drag_set = [self _dragTypesForWindow: winNum];
  NSEnumerator *drag_enum = [drag_set objectEnumerator];

  window = (XGWindow*)[XGWindow _windowWithTag: winNum];
  count = [drag_set count];
  typelist = NSZoneMalloc([self zone], (count+1) * sizeof(Atom));
  i = 0;
  while ((type = [drag_enum nextObject]))
    {
      NSString *mime = [NSPasteboard mimeTypeForPasteboardType: type];
      NSDebugLog(@"  registering %@ on window\n", mime);
      typelist[i] = XInternAtom(XDPY, [mime cString], False);
      i++;
    }
  typelist[i] = 0;
  xdnd_set_dnd_aware(&dnd, [(XGWindow*)window xWindow], typelist);
  NSZoneFree([self zone], typelist);
}

- (BOOL) _addDragTypes: (NSArray*)types toWindow: (int)winNum
{
  BOOL did_add;
  XGWindow  *window;

  window = (XGWindow*)[XGWindow _windowWithTag: winNum];
  if (!xDndInitialized && window)
    {
      xDndInitialized = YES;
      xdnd_init(&dnd, XDPY);
      xdnd_set_dnd_aware(&dnd, [(XGWindow*)window xWindow], NULL);
    }
  did_add = [super _addDragTypes: types toWindow: winNum];
  if (did_add)
    [self _resetDragTypes: types toWindow: winNum];
  return did_add;
}

- (BOOL) _removeDragTypes: (NSArray*)types fromWindow: (int)winNum
{
  BOOL did_add;

  did_add = [super _addDragTypes: types toWindow: winNum];
  if (did_add)
    [self _resetDragTypes: types toWindow: winNum];
  return did_add;
}

- (id <NSDraggingInfo>)_dragInfo
{
  return [XGDragView sharedDragView];
}

- (void) _postExternalEvent: (NSEvent *)theEvent
{
  if ([theEvent subtype] == GSAppKitDraggingStatus)
    {
      Atom xaction;
      NSDragOperation action = [theEvent data2];
      xaction = GSActionForDragOperation(action);
      xdnd_send_status(&dnd, 
		       [theEvent data1],
		       [(XGWindow *)[theEvent window] xWindow], 
		       (action != NSDragOperationNone),
		       0,
		       0, 0, 0, 0,
		       xaction);
    }
  else if ([theEvent subtype] == GSAppKitDraggingFinished)
    {
      xdnd_send_finished(&dnd, 
			 [theEvent data1],
			 [(XGWindow *)[theEvent window] xWindow], 
			 0);
    }
  else
    {
      NSLog(@"Internal: unhandled post external event\n");
    }

}

@end

@implementation XGContext (Ops)

/* ----------------------------------------------------------------------- */
/* Color operations */
/* ----------------------------------------------------------------------- */
- (void)DPScurrentcmykcolor: (float *)c : (float *)m : (float *)y : (float *)k 
{
  [gstate DPScurrentcmykcolor:c :m :y :k];
}

- (void)DPSsetcmykcolor: (float)c : (float)m : (float)y : (float)k 
{
  [gstate DPSsetcmykcolor:c :m :y :k];
}

/* ----------------------------------------------------------------------- */
/* Data operations */
/* ----------------------------------------------------------------------- */
- (void)DPSclear 
{
  GSIArrayEmpty((GSIArray)opstack);
  GSIArrayInitWithZoneAndCapacity((GSIArray)opstack, [self zone], 2);
}

- (void)DPScleartomark 
{
  unsigned count = GSIArrayCount((GSIArray)opstack);
  unsigned pos = count;

  while (pos > 0)
    {
      NSObject *obj = (GSIArrayItemAtIndex((GSIArray)opstack, --pos)).obj;

      if ([obj isKindOfClass: [XGDrawObject class]]
	  && [(XGDrawObject *)obj drawType] == ctxt_mark)
	{
	  int i;
	  for (i = count; i >=pos; i--)
	    GSIArrayRemoveItemAtIndex((GSIArray)opstack, i);
	  return;
	}
    }
}

- (void)DPScopy: (int)n 
{
}

- (void)DPScount: (int *)n 
{
  CHECK_NULL_OUTPUT(n);
  *n = GSIArrayCount((GSIArray)opstack);
}

- (void)DPScounttomark: (int *)n 
{
}

- (void)DPSdup 
{
}

- (void)DPSexch 
{
  int count = GSIArrayCount((GSIArray)opstack);
  if (count < 2)
    DPS_ERROR(DPSstackunderflow, @"Attempt to exch in empty stack");
  GSIArrayInsertItem((GSIArray)opstack, 
		 GSIArrayLastItem((GSIArray)opstack), count-2);
  GSIArrayRemoveLastItem((GSIArray)opstack);
}

- (void)DPSexecstack 
{
}

- (void)DPSget 
{
}

- (void)DPSindex: (int)i 
{
}

- (void)DPSmark 
{
}

- (void)DPSmatrix 
{
  ctxt_push([NSAffineTransform transform], opstack);
}

- (void)DPSnull 
{
}

- (void)DPSpop 
{
  id obj;
  ctxt_pop(obj, opstack, NSObject);
}

- (void)DPSput 
{
}

- (void)DPSroll: (int)n : (int)j 
{
}

/* ----------------------------------------------------------------------- */
/* Font operations */
/* ----------------------------------------------------------------------- */

/* Takes a font name stored in font.name and tries to find the X11 font
   information for that font. If the the font name does not begin with a
   '-', then it is assumed to be a standard PostScript name that should
   be converted to the X11 font name format. */
- (void) _findXFont: (font_t *)font
{
  int i, len;
  char   buf[1024];

  /* Try to convert the font name into a X11 font name */
  if (font->name[0] != '-')
    {
      char *s;
      len = strlen(font->name);
      font->internal_name = copy_string_buffer(font->name);
      for (i=0; i < len; i++)
	font->internal_name[i] = tolower(font->internal_name[i]);
      if (font->weight == 0)
	font->weight = "medium";
      if (font->slant == 0)
	font->slant  = 'r';
      if (font->size == 0)
	font->size   = 120;
      s = strchr(font->internal_name, '-');
      if (s)
	{
	  switch(*s) 
	    {
	    case 'b': font->weight = "bold"; break;
	    case 'd': font->weight = "demibold"; break;
	    case 'o': font->slant  = 'o'; break;
	    case 'i': font->slant  = 'i'; break;
	    default:
	    }
	}
      sprintf(buf, "-*-%s-%s-%c-*-%d-*", font->internal_name,
	      font->weight, font->slant, (int)font->size);
      free(font->internal_name);
      font->internal_name = NULL;
    }
  else
    strcpy(buf, font->name);

  font->info = XLoadQueryFont(XDPY, buf);
  if (font->info)
    {
      font->internal_name = copy_string_buffer(buf);
      font->id   = font->info->fid;
    }
}

- (void) _checkFontDir
{
  if (globalfontdir == nil)
    globalfontdir = [CTXT_OBJ_ALLOC(NSMutableDictionary) init];
  if (fontid == nil)
    fontid = [NSMutableArray arrayWithCapacity:32];
}

- (void)DPSdefineresource: (const char *)category 
{
  id obj;
  NSString *key;

  ctxt_pop(obj, opstack, NSNumber);
  ctxt_pop(key, opstack, NSNumber);
  if (strcmp(category, "Font") == 0)
    [self _checkFontDir];
  else
    DPS_ERROR(DPSundefined, [NSString stringWithCString: category]);

  [globalfontdir setObject: obj forKey: key];
  ctxt_push(obj, opstack);
}

- (void)DPSfindresource: (const char *)key : (const char *)category 
{
  id obj;

  if (strcmp(category, "Font") == 0)
    [self _checkFontDir];
  else
    DPS_ERROR(DPSundefined, @"Could not find key %s", key);

  obj = [globalfontdir objectForKey: [NSString stringWithCString:key]];

  if (obj == nil)
    {
      if (strcmp(category, "Font") == 0)
	{
	  int ident;
	  font_t font;
	  font.name = copy_string_buffer(key);
	  font.weight = 0;
	  font.slant = 0;
	  font.size = 0;
	  [self _findXFont: &font];
	  if (font.internal_name != NULL)
	    {
	      ident = [fontid count];
	      obj = [NSValue value: &font withObjCType: @encode(font_t)];
	      [fontid addObject: obj];
	      obj = [NSNumber numberWithLong: ident];
	    }
	}
      if (obj == nil)
	{
	  DPS_ERROR(DPSundefinedresource, @"Could not find resource %s", key);
	}
    }

  ctxt_push(obj, opstack);
}

- (void)DPSFontDirectory 
{
}

- (void)DPSISOLatin1Encoding 
{
}

- (void)DPSSharedFontDirectory 
{
}

- (void)DPSStandardEncoding 
{
}

- (void)DPScurrentcacheparams 
{
}

- (void)DPScurrentfont 
{
  int ident;
  //[self _checkFontDir];
  //ident = [fontid indexOfObject: fontobj];
  ctxt_push([NSNumber numberWithLong: ident], opstack);
}

- (void)DPSdefinefont 
{
  [self DPSdefineresource: "Font"];
}

- (void)DPSfindfont: (const char *)name 
{
  [self DPSfindresource: name : "Font"];
}

- (void)DPSmakefont 
{
  id obj, fontobj;
  long ident;
  font_t font;

  ctxt_pop(obj, opstack, NSNumber);
  ident = [obj longValue];
  CHECK_INVALID_FONT(ident);
  fontobj = [fontid objectAtIndex: ident];
  [fontobj getValue:&font];
  [gstate setFontStruct: font.info];
}

- (void)DPSscalefont: (float)size 
{
  id obj;
  long ident;
  font_t font;

  ctxt_pop(obj, opstack, NSNumber);
  ident = [obj longValue];
  CHECK_INVALID_FONT(ident);

  obj = [fontid objectAtIndex:ident];
  [obj getValue:&font];
  font.size = size * 10;

  /* Try to find a X font that matches this size */
  [self _findXFont: &font];
  if (font.internal_name != NULL)
    {
      obj = [NSValue value: &font withObjCType: @encode(font_t)];
      ident = [fontid count];
      [fontid addObject: obj];
    }
  ctxt_push([NSNumber numberWithLong: ident], opstack);
}

- (void)DPSselectfont: (const char *)name : (float)scale 
{
}

- (void)DPSsetfont: (int)f 
{
  id fontobj;
  font_t font;
  CHECK_INVALID_FONT(f);
  fontobj = [fontid objectAtIndex: f];
  [fontobj getValue:&font];
  [gstate setFontStruct: font.info];
}

- (void)DPSundefinefont: (const char *)name 
{
}

/* ----------------------------------------------------------------------- */
/* Gstate operations */
/* ----------------------------------------------------------------------- */

- (void)DPSconcat: (const float *)m 
{
  [gstate DPSconcat: m];
}

- (void)DPScurrentdash 
{
}

- (void)DPScurrentflat: (float *)flatness 
{
}

- (void)DPScurrentgray: (float *)gray 
{
  CHECK_NULL_OUTPUT(gray);
  [gstate DPScurrentgray: gray];
}

- (void)DPScurrentgstate: (int)gst 
{
  /* Associate/copy current gstate with gst */
  ctxt_push([NSNumber numberWithInt: gst], opstack);
  ctxt_push(gstate, opstack);
  [self DPSdefineuserobject];
  [self DPSexecuserobject: gst];
}

- (void)DPScurrenthalftone 
{
}

- (void)DPScurrenthalftonephase: (float *)x : (float *)y 
{
}

- (void)DPScurrenthsbcolor: (float *)h : (float *)s : (float *)b 
{
  CHECK_NULL_OUTPUT(h);
  CHECK_NULL_OUTPUT(s);
  CHECK_NULL_OUTPUT(b);
  [gstate DPScurrenthsbcolor:h :s :b];
}

- (void)DPScurrentlinecap: (int *)linecap 
{
  [gstate DPScurrentlinecap: linecap];
}

- (void)DPScurrentlinejoin: (int *)linejoin 
{
  [gstate DPScurrentlinejoin: linejoin];
}

- (void)DPScurrentlinewidth: (float *)width 
{
  [gstate DPScurrentlinewidth: width];
}

- (void)DPScurrentmatrix 
{
  float m[6];
  id obj;
  ctxt_pop(obj, opstack, NSAffineTransform);
  [gstate DPScurrentmatrix: m];
  [obj setMatrix: m];
  ctxt_push(obj, opstack);
}

- (void)DPScurrentmiterlimit: (float *)limit 
{
}

- (void)DPScurrentpoint: (float *)x : (float *)y 
{
  CHECK_NULL_OUTPUT(x);
  CHECK_NULL_OUTPUT(y);
  [gstate DPScurrentpoint:x :y];
}

- (void)DPScurrentrgbcolor: (float *)r : (float *)g : (float *)b 
{
  CHECK_NULL_OUTPUT(r);
  CHECK_NULL_OUTPUT(g);
  CHECK_NULL_OUTPUT(b);
  [gstate DPScurrentrgbcolor:r :g :b];
}

- (void)DPScurrentscreen 
{
}

- (void)DPScurrentstrokeadjust: (int *)b 
{
}

- (void)DPScurrenttransfer 
{
}

- (void)DPSdefaultmatrix 
{
}

- (void)DPSgrestore 
{
  RELEASE(gstate);
  ctxt_pop(gstate, gstack, XGGState);
  RETAIN(gstate);
}

- (void)DPSgrestoreall 
{
}

- (void)DPSgsave 
{
  ctxt_push(gstate, gstack);
  AUTORELEASE(gstate);
  gstate = [gstate copy];
}

- (void)DPSgstate 
{
  ctxt_push(AUTORELEASE([gstate copy]), opstack);
}

- (void)DPSinitgraphics 
{
  [gstate DPSinitgraphics];
}

- (void)DPSinitmatrix 
{
  [gstate DPSinitmatrix];
}

- (void)DPSrotate: (float)angle 
{
  [gstate DPSrotate: angle];
}

- (void)DPSscale: (float)x : (float)y 
{
  [gstate DPSscale:x :y];
}

- (void)DPSsetdash: (const float *)pat : (int)size : (float)offset 
{
}

- (void)DPSsetflat: (float)flatness 
{
}

- (void)DPSsetgray: (float)gray 
{
  [gstate DPSsetgray: gray];
}

- (void)DPSsetgstate: (int)gst 
{
  [self DPSexecuserobject: gst];
  RELEASE(gstate);
  ctxt_pop(gstate, opstack, XGGState);
  RETAIN(gstate);
}

- (void)DPSsethalftone 
{
}

- (void)DPSsethalftonephase: (float)x : (float)y 
{
}

- (void)DPSsethsbcolor: (float)h : (float)s : (float)b 
{
  [gstate DPSsethsbcolor:h :s :b];
}

- (void)DPSsetlinecap: (int)linecap 
{
  [gstate DPSsetlinecap: linecap];
}

- (void)DPSsetlinejoin: (int)linejoin 
{
  [gstate DPSsetlinejoin: linejoin];
}

- (void)DPSsetlinewidth: (float)width 
{
  [gstate DPSsetlinewidth: width];
}

- (void)DPSsetmatrix 
{
  float matrix[6];
  NSAffineTransform *obj;

  ctxt_pop(obj, opstack, NSAffineTransform);
  [obj getMatrix: matrix];
  [gstate DPSsetmatrix: matrix];
}

- (void)DPSsetmiterlimit: (float)limit 
{
}

- (void)DPSsetrgbcolor: (float)r : (float)g : (float)b 
{
  [gstate DPSsetrgbcolor:r :g :b];
}

- (void)DPSsetscreen 
{
}

- (void)DPSsetstrokeadjust: (int)b 
{
}

- (void)DPSsettransfer 
{
}

- (void)DPStranslate: (float)x : (float)y 
{
  [gstate DPStranslate:x :y];
}

/* ----------------------------------------------------------------------- */
/* Matrix operations */
/* ----------------------------------------------------------------------- */
- (void)DPSflush
{
}

/* ----------------------------------------------------------------------- */
/* Matrix operations */
/* ----------------------------------------------------------------------- */

- (void)DPSconcatmatrix
{
  NSAffineTransform *obj1, *obj2, *obj3;

  ctxt_pop(obj3, opstack, NSAffineTransform);
  ctxt_pop(obj2, opstack, NSAffineTransform);
  ctxt_pop(obj1, opstack, NSAffineTransform);
  [obj2 concatenateWith: obj1];
  obj3 = [obj2 copy];
  ctxt_push(obj3, opstack);
  RELEASE(obj3);
}

- (void)DPSdtransform: (float)x1 : (float)y1 : (float *)x2 : (float *)y2 
{
  NSAffineTransform *obj;
  NSPoint point;

  ctxt_pop(obj, opstack, Nil);
  if (![obj isKindOfClass: [NSAffineTransform class]])
    {
      ctxt_push(obj, opstack);
      ctxt_push([NSAffineTransform transform], opstack);
      [self DPScurrentmatrix];
      ctxt_pop(obj, opstack, NSAffineTransform);
    }
  point.x = x1; point.y = y1;
  point = [obj pointInMatrixSpace: point];
  *x2 = point.x;
  *y2 = point.y;
}

- (void)DPSidentmatrix 
{
  NSAffineTransform *obj;

  ctxt_pop(obj, opstack, NSAffineTransform);
  obj = [NSAffineTransform transform];
  [obj makeIdentityMatrix];
  ctxt_push(obj, opstack);
}

- (void)DPSidtransform: (float)x1 : (float)y1 : (float *)x2 : (float *)y2 
{
  NSAffineTransform *obj;
  NSPoint point;

  ctxt_pop(obj, opstack, Nil);
  if (![obj isKindOfClass: [NSAffineTransform class]])
    {
      ctxt_push(obj, opstack);
      ctxt_push([NSAffineTransform transform], opstack);
      [self DPScurrentmatrix];
      ctxt_pop(obj, opstack, NSAffineTransform);
    }
  [obj inverse];
  point.x = x1; point.y = y1;
  point = [obj pointInMatrixSpace: point];
  *x2 = point.x;
  *y2 = point.y;
}

- (void)DPSinvertmatrix 
{
  NSAffineTransform *obj, *obj2;

  ctxt_pop(obj, opstack, NSAffineTransform);
  ctxt_pop(obj2, opstack, NSAffineTransform);
  [obj inverse];
  ctxt_push(obj, opstack);
}

- (void)DPSitransform: (float)x1 : (float)y1 : (float *)x2 : (float *)y2 
{
  NSAffineTransform *obj;
  NSPoint point;

  ctxt_pop(obj, opstack, Nil);
  if (![obj isKindOfClass: [NSAffineTransform class]])
    {
      ctxt_push(obj, opstack);
      ctxt_push([NSAffineTransform transform], opstack);
      [self DPScurrentmatrix];
      ctxt_pop(obj, opstack, NSAffineTransform);
    }
  [obj inverse];
  point.x = x1; point.y = y1;
  point = [obj pointInMatrixSpace: point];
  *x2 = point.x;
  *y2 = point.y;
}

- (void)DPStransform: (float)x1 : (float)y1 : (float *)x2 : (float *)y2 
{
  NSAffineTransform *obj;
  NSPoint point;

  ctxt_pop(obj, opstack, Nil);
  if (![obj isKindOfClass: [NSAffineTransform class]])
    {
      ctxt_push(obj, opstack);
      ctxt_push([NSAffineTransform transform], opstack);
      [self DPScurrentmatrix];
      ctxt_pop(obj, opstack, Nil);
    }
  point.x = x1; point.y = y1;
  point = [obj pointInMatrixSpace: point];
  *x2 = point.x;
  *y2 = point.y;
}

/* ----------------------------------------------------------------------- */
/* Opstack operations */
/* ----------------------------------------------------------------------- */

- (void)DPSdefineuserobject
{
  int n;
  id obj;
  NSNumber *number;
  ctxt_pop(obj, opstack, NSObject);
  ctxt_pop(number, opstack, NSNumber);
  n = [number intValue];
  if (n < 0)
    DPS_ERROR(DPSinvalidparam, @"Invalid userobject index");
  else if (n < [ulist count])
    [ulist replaceObjectAtIndex: n withObject: obj];
  else
    {
      int i = n - [ulist count];
      while (i--)
	[ulist addObject: AUTORELEASE([[NSObject alloc] init])];
      [ulist addObject: obj];
    }
}

- (void)DPSexecuserobject: (int)index
{
  if (index < 0 || index >= [ulist count])
    DPS_ERROR(DPSinvalidparam, @"Invalid userobject index");
  ctxt_push([ulist objectAtIndex: index], opstack);
}

- (void)DPSundefineuserobject: (int)index
{
  if (index < 0 || index >= [ulist count])
    DPS_ERROR(DPSinvalidparam, @"Invalid userobject index");
  [ulist replaceObjectAtIndex: index
                   withObject: AUTORELEASE([[NSObject alloc] init])];
}

- (void)DPSgetboolean: (int *)it 
{
  NSNumber *number;
  CHECK_NULL_OUTPUT(it);
  ctxt_pop(number, opstack, NSNumber);
  *it = [number boolValue];
}

- (void)DPSgetchararray: (int)size : (char *)s 
{
  NSMutableData *data;
  CHECK_NULL_OUTPUT(s);
  ctxt_pop(data, opstack, NSMutableData);
  memcpy(s, [data bytes], size*sizeof(char));
}

- (void)DPSgetfloat: (float *)it 
{
  NSNumber *number;
  CHECK_NULL_OUTPUT(it);
  ctxt_pop(number, opstack, NSNumber);
  *it = [number floatValue];
}

- (void)DPSgetfloatarray: (int)size : (float *)a 
{
  NSMutableData *data;
  CHECK_NULL_OUTPUT(a);
  ctxt_pop(data, opstack, NSMutableData);
  memcpy(a, [data bytes], size*sizeof(float));
}

- (void)DPSgetint: (int *)it 
{
  NSNumber *number;
  CHECK_NULL_OUTPUT(it);
  ctxt_pop(number, opstack, NSNumber);
  *it = [number intValue];
}

- (void)DPSgetintarray: (int)size : (int *)a 
{
  NSMutableData *data;
  CHECK_NULL_OUTPUT(a);
  ctxt_pop(data, opstack, NSMutableData);
  memcpy(a, [data bytes], size*sizeof(int));
}

- (void)DPSgetstring: (char *)s 
{
  NSString *str;
  CHECK_NULL_OUTPUT(s);
  ctxt_pop(str, opstack, NSString);
  strcpy(s, [str cString]);
}

- (void)DPSsendboolean: (int)it 
{
  ctxt_push([NSNumber numberWithBool: it], opstack);
}

- (void)DPSsendchararray: (const char *)s : (int)size 
{
  NSMutableData *data;
  data = [NSMutableData dataWithBytes: s length: size*sizeof(char)];
  ctxt_push(data, opstack);
}

- (void)DPSsendfloat: (float)it 
{
  ctxt_push([NSNumber numberWithFloat: it], opstack);
}

- (void)DPSsendfloatarray: (const float *)a : (int)size 
{
  NSMutableData *data;
  data = [NSMutableData dataWithBytes: a length: size*sizeof(float)];
  ctxt_push(data, opstack);
}

- (void)DPSsendint: (int)it 
{
  ctxt_push([NSNumber numberWithInt: it], opstack);
}

- (void)DPSsendintarray: (const int *)a : (int)size 
{
  NSMutableData *data;
  data = [NSMutableData dataWithBytes: a length: size*sizeof(int)];
  ctxt_push(data, opstack);
}

- (void)DPSsendstring: (const char *)s 
{
  ctxt_push([NSString stringWithCString: s], opstack);
}

/* ----------------------------------------------------------------------- */
/* Paint operations */
/* ----------------------------------------------------------------------- */

- (void)DPSashow: (float)x : (float)y : (const char *)s 
{
  [gstate DPSashow: x : y : s];
}

- (void)DPSawidthshow: (float)cx : (float)cy : (int)c : (float)ax : (float)ay : (const char *)s 
{
  [gstate DPSawidthshow: cx : cy : c : ax : ay : s];
}

- (void)DPScopypage 
{
}

- (void)DPSeofill 
{
  [gstate DPSeofill];
}

- (void)DPSerasepage 
{
}

- (void)DPSfill 
{
  [gstate DPSfill];
}

- (void)DPSimage 
{
}

- (void)DPSimagemask 
{
}

- (void)DPSkshow: (const char *)s 
{
}

- (void)DPSrectfill: (float)x : (float)y : (float)w : (float)h 
{
  [gstate DPSrectfill:x :y :w :h];
}

- (void)DPSrectstroke: (float)x : (float)y : (float)w : (float)h 
{
  [gstate DPSrectstroke:x :y :w :h];
}

- (void)DPSshow: (const char *)s 
{
  [gstate DPSshow: s];
}

- (void)DPSshowpage 
{
}

- (void)DPSstroke 
{
  [gstate DPSstroke];
}

- (void)DPSstrokepath 
{
  [gstate DPSstrokepath];
}

- (void)DPSueofill: (const char *)nums : (int)n : (const char *)op : (int)l 
{
}

- (void)DPSufill: (const char *)nums : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSustroke: (const char *)nums   : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSustrokepath: (const char *)nums : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSwidthshow: (float)x : (float)y : (int)c : (const char *)s 
{
  [gstate DPSwidthshow: x : y : c : s];
}

- (void)DPSxshow: (const char *)s : (const float *)numarray : (int)size 
{
  [gstate DPSxshow: s : numarray : size];
}

- (void)DPSxyshow: (const char *)s : (const float *)numarray : (int)size 
{
  [gstate DPSxyshow: s : numarray : size];
}

- (void)DPSyshow: (const char *)s : (const float *)numarray : (int)size 
{
  [gstate DPSyshow: s : numarray : size];
}

/* ----------------------------------------------------------------------- */
/* Path operations */
/* ----------------------------------------------------------------------- */

- (void)DPSarc: (float)x : (float)y : (float)r : (float)angle1 : (float)angle2 
{
  [gstate DPSarc: x : y : r : angle1 : angle2];
}

- (void)DPSarcn: (float)x : (float)y : (float)r : (float)angle1 : (float)angle2 
{
  [gstate DPSarc: x : y : r : angle1 : angle2];
}

- (void)DPSarct: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)r 
{
  [gstate DPSarc: x1 : y1 : x2 : y2 : r];
}

- (void)DPSarcto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)r : (float *)xt1 : (float *)yt1 : (float *)xt2 : (float *)yt2 
{
  [gstate DPSarcto: x1 : y1 : x2 : y2 : r : xt1 : yt1 : xt2 : yt2];
}

- (void)DPScharpath: (const char *)s : (int)b 
{
}

- (void)DPSclip 
{
  [gstate DPSclip];
}

- (void)DPSclippath 
{
  [gstate DPSclippath];
}

- (void)DPSclosepath 
{
  [gstate DPSclosepath];
}

- (void)DPScurveto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)x3 : (float)y3 
{
  [gstate DPScurveto: x1 : y1 : x2 : y2 : x3 : y3];
}

- (void)DPSeoclip 
{
  [gstate DPSeoclip];
}

- (void)DPSeoviewclip 
{
  [gstate DPSeoviewclip];
}

- (void)DPSflattenpath 
{
  [gstate DPSflattenpath];
}

- (void)DPSinitclip 
{
  [gstate DPSinitclip];
}

- (void)DPSinitviewclip 
{
  [gstate DPSinitviewclip];
}

- (void)DPSlineto: (float)x : (float)y 
{
  [gstate DPSlineto: x : y];
}

- (void)DPSmoveto: (float)x : (float)y 
{
  [gstate DPSmoveto: x : y];
}

- (void)DPSnewpath 
{
  [gstate DPSnewpath];
}

- (void)DPSpathbbox: (float *)llx : (float *)lly : (float *)urx : (float *)ury 
{
  [gstate DPSpathbbox: llx : lly : urx : ury];
}

- (void)DPSpathforall 
{
  [gstate DPSpathforall];
}

- (void)DPSrcurveto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)x3 : (float)y3 
{
  [gstate DPSrcurveto: x1 : y1 : x2 : y2 : x3 : y3];
}

- (void)DPSrectclip: (float)x : (float)y : (float)w : (float)h 
{
  [gstate DPSrectclip: x : y : w : h];
}

- (void)DPSrectviewclip: (float)x : (float)y : (float)w : (float)h 
{
  [gstate DPSrectviewclip: x : y : w : h];
}

- (void)DPSreversepath 
{
  [gstate DPSreversepath];
}

- (void)DPSrlineto: (float)x : (float)y 
{
  [gstate DPSrlineto: x : y];
}

- (void)DPSrmoveto: (float)x : (float)y 
{
  [gstate DPSrmoveto: x : y];
}

- (void)DPSsetbbox: (float)llx : (float)lly : (float)urx : (float)ury 
{
  [gstate DPSsetbbox: llx : lly : urx : ury];
}

- (void)DPSviewclip 
{
  [gstate DPSviewclip];
}

- (void)DPSviewclippath 
{
}

/* ----------------------------------------------------------------------- */
/* Window system ops */
/* ----------------------------------------------------------------------- */

- (void) DPScurrentdrawingfunction: (int *)function
{
}

- (void) DPScurrentgcdrawable: (void **)gc : (void **)draw : (int *)x : (int *)y
{
  if (gc)
    *gc = (void *)[gstate graphicContext];
  if (draw)
    *draw = (void *)[gstate drawable];
  if (x && y)
    {
      NSPoint offset = [gstate offset];
      *x = offset.x;
      *y = offset.y;
    }
}

- (void) DPScurrentgcdrawablecolor: (void **)gc : (void **)draw : (int *)x 
				  : (int *)y : (int *)colorInfo
{
  if (gc)
    *gc = (void *)[gstate graphicContext];
  if (draw)
    *draw = (void *)[gstate drawable];
  if (x && y)
    {
      NSPoint offset = [gstate offset];
      *x = offset.x;
      *y = offset.y;
    }
  if (colorInfo)
    *colorInfo = 0;
}

- (void) DPScurrentoffset: (int *)x : (int *)y
{
  if (x && y)
    {
      NSPoint offset = [gstate offset];
      *x = offset.x;
      *y = offset.y;
    }
}

- (void) DPSsetdrawingfunction: (int) function
{
  XGCValues values;
  values.function = function;
  [gstate setGCValues: values withMask: (GCFunction | GCForeground)];
}

- (void) DPSsetgcdrawable: (void *)gc : (void *)draw : (int)x : (int)y
{
  [gstate setGraphicContext: (GC)gc];
  [gstate setDrawable: (Drawable)draw];
  [gstate setOffset: NSMakePoint(x, y)];
}

- (void) DPSsetgcdrawablecolor: (void *)gc : (void *)draw : (int)x : (int)y
				  : (const int *)colorInfo
{
  [gstate setGraphicContext: (GC)gc];
  [gstate setDrawable: (Drawable)draw];
  [gstate setOffset: NSMakePoint(x, y)];
}

- (void) DPSsetoffset: (short int)x : (short int)y
{
  [gstate setOffset: NSMakePoint(x,y)];
}

- (void) DPSsetrgbactual: (double)r : (double)g : (double)b : (int *)success
{
  unsigned long pixel;
  pixel = xrAllocActualRGB(context, r, g, b);
  if (pixel > 0)
    [gstate DPSsetrgbcolor: r : g : b];
  *success = (pixel > 0);
}

/* Add the current gstate to the gstate index (if not there already)
   and return its index */
- (void) DPScapturegstate: (int *)gst
{
  CHECK_NULL_OUTPUT(gst);
  if ([ulist indexOfObject: gstate] == NSNotFound)
    [ulist addObject: gstate];
  *gst = [ulist indexOfObject: gstate];
}

/*-------------------------------------------------------------------------*/
/* Graphics Extension Ops */
/*-------------------------------------------------------------------------*/
- (void) DPScomposite: (float)x : (float)y : (float)w : (float)h : (int)gstateNum : (float)dx : (float)dy : (int)op
{
  NSRect rect;
  NSPoint p;
  Drawable source;
  XGCValues gcv;

  if (gstateNum)
    {
      XGGState *g;
      [self DPSexecuserobject: gstateNum];
      ctxt_pop(g, opstack, XGGState);
      source = [g drawable];
      if (!source) 
        DPS_ERROR(DPSundefined, @"Composite drawable not realized\n");
    }
  else
    source = 0; /* gstate will set it's own drawable to source */

  rect = NSMakeRect(x, y, w, h);
  p    = NSMakePoint(dx, dy);

  /* FIXME: Really need alpha dithering to do this right - combine with
     XGBitmapImageRep code? */
  switch (op)
    {
    case   NSCompositeClear:
      gcv.function = GXclear;
      break;
    case   NSCompositeCopy:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceOver:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceIn:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceOut:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceAtop:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDataOver:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDataIn:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDataOut:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDataAtop:
      gcv.function = GXcopy;
      break;
    case   NSCompositeXOR:
      gcv.function = GXxor;
      break;
    case   NSCompositePlusDarker:
      gcv.function = GXcopy;
      break;
    case   NSCompositeHighlight:
      gcv.function = GXxor;
      break;
    case   NSCompositePlusLighter:
      gcv.function = GXcopy;
      break;
    default:
      gcv.function = GXcopy;
      break;
    }
  if (gcv.function != GXcopy)
    [gstate setGCValues: gcv withMask: GCFunction];
  [gstate copyDrawable: source fromRect: rect toPoint: p];
  if (gcv.function != GXcopy)
    {
      gcv.function = GXcopy;
      [gstate setGCValues: gcv withMask: GCFunction];
    }
}

- (void) DPScompositerect: (float)x : (float)y : (float)w : (float)h : (int)op
{
  float gray;
  XGCValues gcv;

  [gstate DPScurrentgray: &gray];
  if (fabs(gray - 0.667) < 0.005)
    [gstate DPSsetgray: 0.333];
  else    
    [gstate DPSsetrgbcolor: 0.121 : 0.121 : 0];

  /* FIXME: Really need alpha dithering to do this right - combine with
     XGBitmapImageRep code? */
  switch (op)
    {
    case   NSCompositeClear:
      gcv.function = GXclear;
      break;
    case   NSCompositeCopy:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceOver:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceIn:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceOut:
      gcv.function = GXcopy;
      break;
    case   NSCompositeSourceAtop:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDataOver:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDataIn:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDataOut:
      gcv.function = GXcopy;
      break;
    case   NSCompositeDataAtop:
      gcv.function = GXcopy;
      break;
    case   NSCompositeXOR:
      gcv.function = GXcopy;
      break;
    case   NSCompositePlusDarker:
      gcv.function = GXcopy;
      break;
    case   NSCompositeHighlight:
      gcv.function = GXxor;
      break;
    case   NSCompositePlusLighter:
      gcv.function = GXcopy;
      break;
    default:
      gcv.function = GXcopy;
      break;
    }
  [gstate setGCValues: gcv withMask: GCFunction];
  [gstate DPSrectfill: x : y : w : h];

  gcv.function = GXcopy;
  [gstate setGCValues: gcv withMask: GCFunction];
  [gstate DPSsetgray: gray];
}

- (void) DPSdissolve: (float)x : (float)y : (float)w : (float)h : (int)gstateNum
 : (float)dx : (float)dy : (float)delta
{
  return [self DPScomposite: x : y : w : h : gstateNum : dx : dy 
	 : NSCompositeSourceOver];
}

- (void) DPSreadimage
{
}

- (void) DPSsetalpha: (float)a
{
}

- (void) DPScurrentalpha: (float *)alpha
{
}

/*
 * Return mouse location in base coords ignoring the event loop
 */
- (void) DPSmouselocation: (float*)x :(float*)y
{
  Window	rootWin;
  Window	childWin;
  int		currentX;
  int		currentY;
  int		winX;
  int		winY;
  unsigned	mask;
  BOOL		ok;

  ok = XQueryPointer (context->dpy, [self xDisplayRootWindow],
    &rootWin, &childWin, &currentX, &currentY, &winX, &winY, &mask);
  if (ok)
    {
      int	xScreen = [(XGScreen *)[NSScreen mainScreen] xScreen];

      *x = currentX;
      *y = DisplayHeight(context->dpy, xScreen) - currentY;
    }
  else
    {
      *x = 0;
      *y = 0;
    }
}

- (NSEvent*) DPSGetEventMatchingMask: (unsigned)mask
			  beforeDate: (NSDate*)limit
			      inMode: (NSString*)mode
			     dequeue: (BOOL)flag
{
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];
  return [super DPSGetEventMatchingMask: mask
		      beforeDate: limit
			  inMode: mode
			 dequeue: flag];
}

- (void) DPSDiscardEventsMatchingMask: (unsigned)mask
			  beforeEvent: (NSEvent*)limit
{
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];
  [super DPSDiscardEventsMatchingMask: mask
			  beforeEvent: limit];
}

@end



@interface XGRawWindow : XGWindow
@end

@implementation XGRawWindow

- (BOOL) canBecomeMainWindow
{
  return NO;
}

- (BOOL) canBecomeKeyWindow
{
  return NO;
}

- (void) initDefaults
{
  [super initDefaults];
  menu_exclude = YES;           // Don't show in windows menu.
  is_released_when_closed = NO;
}

- (id) initWithContentRect: (NSRect)contentRect
		 styleMask: (unsigned int)aStyle
		   backing: (NSBackingStoreType)bufferingType
		     defer: (BOOL)flag
		    screen: (NSScreen*)aScreen
{
  XSetWindowAttributes winattrs;
  unsigned long valuemask;

  self = [super initWithContentRect: contentRect
			  styleMask: aStyle
			    backing: bufferingType
			      defer: flag
			     screen: aScreen];

  valuemask = (CWSaveUnder|CWOverrideRedirect);
  winattrs.save_under = True;
  winattrs.override_redirect = True;
  XChangeWindowAttributes ([XGContext currentXDisplay], [self xWindow],
                            valuemask, &winattrs);
  [[NSApplication sharedApplication] removeWindowsItem: self];

  return self;
}
@end



@implementation XGDragView

#define	DWZ	48

static	XGDragView	*sharedDragView = nil;

+ (XGDragView*) sharedDragView
{
  if (!sharedDragView)
    {
      NSRect			winRect = {{0, 0}, {DWZ, DWZ}};
      XGRawWindow		*sharedDragWindow = [XGRawWindow alloc];

      [sharedDragWindow initWithContentRect: winRect
				  styleMask: NSBorderlessWindowMask
				    backing: NSBackingStoreNonretained
				      defer: YES];

      sharedDragView = [XGDragView new];
      [sharedDragWindow setContentView: sharedDragView];
      RELEASE(sharedDragView);
    }

  return sharedDragView;
}

- (void) _sendLocalEvent: (GSAppKitSubtype)subtype
		  action: (NSDragOperation)action
	        position: (NSPoint)eventLocation
	       timestamp: (NSTimeInterval)time
	        toWindow: (NSWindow *)dWindow
{
  NSEvent *e;
  NSGraphicsContext *context = GSCurrentContext();
  NSRect xFrame = [(XGWindow *)dWindow xFrame];

  eventLocation = NSMakePoint(eventLocation.x - NSMinX(xFrame),
			      eventLocation.y - NSMinY(xFrame));
  eventLocation.y = xFrame.size.height - eventLocation.y;

  e = [NSEvent otherEventWithType: NSAppKitDefined
	  location: eventLocation
	  modifierFlags: 0
	  timestamp: time
	  windowNumber: [dWindow windowNumber]
	  context: context
	  subtype: subtype
	  data1: [(XGWindow *)window xWindow]
	  data2: action];
  [dragWindow sendEvent: e];
}

- (void) setDragInfoFromEvent: (NSEvent *)event
{
  dragSource = nil;
  dragPasteboard = [[NSPasteboard pasteboardWithName: NSDragPboard] retain];
  dragPoint = [event locationInWindow];
  dragSequence = [event timestamp];
  dragWindow = [event window];
  dragMask = [event data2];
  dragLocal = NO;
}

- (void) drawRect: (NSRect)rect
{
  [dragCell drawWithFrame: rect inView: self];
}

- (void)dragImage:(NSImage *)anImage
               at:(NSPoint)viewLocation
           offset:(NSSize)initialOffset
            event:(NSEvent *)event
       pasteboard:(NSPasteboard *)pboard
           source:(id)sourceObject
        slideBack:(BOOL)slideFlag
{
  int i, count;
  NSArray *types;
  Display *xDisplay = [XGContext currentXDisplay];

  if (!anImage)
    anImage = [NSImage imageNamed: @"common_Close"];

  if (!dragCell)
    {
      dragCell = [[NSCell alloc] initImageCell: anImage];
      [dragCell setBordered: NO];
    }
  [dragCell setImage: anImage];
  DESTROY(dragPasteboard);
  dragPasteboard = [pboard retain];
  dragSource = sourceObject;

  types = [pboard types];
  count = [types count];
  typelist = NSZoneMalloc([self zone], (count+1) * sizeof(Atom));
  for (i = 0; i < count; i++)
    {
      NSString *mime = [NSPasteboard mimeTypeForPasteboardType: 
		       [types objectAtIndex: i]];
      typelist[i] = XInternAtom(xDisplay, [mime cString], False);
    }
  typelist[count] = 0;

  [self mouseDown: event];
  NSZoneFree([self zone], typelist);
  typelist = NULL;
}

- (void)mouseDown: (NSEvent*)theEvent
{
  Display	*xDisplay = [XGContext currentXDisplay];
  int		xScreen = [(XGScreen *)[NSScreen mainScreen] xScreen];
  unsigned int	eventMask = NSLeftMouseDownMask | NSLeftMouseUpMask
			    | NSLeftMouseDraggedMask | NSMouseMovedMask
			    | NSPeriodicMask | NSAppKitDefinedMask;
  NSPoint	point, lastPoint;
  NSDate	*theDistantFuture = [NSDate distantFuture];
  NSEventType	eventType;
  Window	xWindow = [window xWindow];
  Window	rootWindow;
  float		wx;
  float		wy;
  int		init = 0;
  NSRect	dragWindowXFrame;
  XGWindow	*eWindow = [theEvent window];
  BOOL		dnd_aware = NO;
  Window        mouseWindow;
  Window	lastMouseWindow = None;
  Window	eXWindow = [eWindow xWindow];
  NSSize	imageSize;

  XColor	xg = [(XGColor*)[NSColor greenColor] xColor];
  XColor	xb = [(XGColor*)[NSColor blackColor] xColor];
  XColor	xw = [(XGColor*)[NSColor whiteColor] xColor];

  rootWindow = RootWindow(xDisplay, xScreen);

  [self unregisterDraggedTypes];

  /* Need to lockFocus to do this. FIXME when image caching works */
  [self lockFocus];
  XShapeCombineMask(xDisplay, xWindow, ShapeBounding, 0, 0,
    [(XGBitmapImageRep*)[[dragCell image] bestRepresentationForDevice: nil]
    xPixmapMask], ShapeSet);
  [self unlockFocus];

  /* convert point and lastPoint into base coordinates (NOT X Coordinates) */
  lastPoint = [[eWindow contentView] convertPoint: [theEvent locationInWindow]
					 fromView: nil];
  lastPoint = [eWindow convertBaseToScreen: lastPoint];
  imageSize = [[dragCell image] size];
  [window setFrame: NSMakeRect(lastPoint.x-imageSize.width/2, 
		    lastPoint.y-imageSize.height/2, 
		    imageSize.width, imageSize.height)
	   display: YES];
  [window orderFront: nil];
  //[window _captureMouse: self];
  dragWindowXFrame = [window xFrame];	// in X screen coordinates
  wx = dragWindowXFrame.origin.x;
  wy = dragWindowXFrame.origin.y;
  dragSequence = [theEvent timestamp];
  dragLocal = NO;

  [NSEvent startPeriodicEventsAfterDelay: 0.02 withPeriod: 0.03];

  while ((eventType = [theEvent type]) != NSLeftMouseUp)
    {
      if (eventType == NSAppKitDefined)
	{
	  GSAppKitSubtype	sub = [theEvent subtype];
	  switch (sub)
	    {
	    case GSAppKitDraggingStatus:
	      NSDebugLLog(@"NSDragging", @"got GSAppKitDraggingStatus\n");
	      if ([theEvent data1] == lastMouseWindow)
		{
		  /* Check the type of operation the target will perform
		     and change the cursor accordingly */
		  /* FIXME: Implement */
		}
	      break;

	    case GSAppKitDraggingFinished:
	      NSLog(@"Internal: got GSAppKitDraggingFinished out of seq\n");
	      break;

	    default:
	      NSLog(@"Internal: dropped NSAppKitDefined (%d) event\n", sub);
	      break;
	    }
	}
      else if (eventType != NSPeriodic)
	{
	  point = [theEvent locationInWindow];
	  if (init < 1)				// dump the first point
	    {					// because it is in coords
	      init++;				// of the dragging source
	      point = lastPoint;
	    }
	  else
	    {
  	      point = [[eWindow contentView] 
				convertPoint: [theEvent locationInWindow]
				    fromView: nil];
  	      point = [eWindow convertBaseToScreen: point];
	    }
	}
      else
	{
	  if (point.x != lastPoint.x || point.y != lastPoint.y)
	    {
	      BOOL oldDragLocal;
	      NSWindow *oldDragWindow;
	      NSDragOperation action;
	      Atom xAction;

	      /* Move drag window (point/lastPoint are reversed since
		 X Coordinates are flipped) */
	      wx += (point.x - lastPoint.x);
	      wy += (lastPoint.y - point.y);
	      XMoveWindow(xDisplay, xWindow, (int)wx, (int)wy);
	      lastPoint = point;

	      mouseWindow = _findXWindow(xDisplay, rootWindow, rootWindow,
		wx+imageSize.width+1, wy+imageSize.height/2);
	      oldDragWindow = dragWindow;
	      oldDragLocal = dragLocal;
	      dragWindow = [XGWindow _windowForXWindow: mouseWindow];
	      dragLocal = (dragWindow) ? YES : NO;
	      dragPoint = lastPoint;
	      if (dragLocal)
		dragPoint = [dragWindow convertScreenToBase: dragPoint];
	      action = [dragSource 
		       draggingSourceOperationMaskForLocal: dragLocal];
	      xAction = GSActionForDragOperation(action);

	      if (mouseWindow != lastMouseWindow && dnd_aware)
		{
		  if (oldDragLocal)
		    [self _sendLocalEvent: GSAppKitDraggingExit
		                   action: action
		                 position: NSZeroPoint
		                timestamp: dragSequence
		                 toWindow: oldDragWindow];
		  else
		    xdnd_send_leave(&dnd, lastMouseWindow, xWindow);
		  lastMouseWindow = None;
		  dnd_aware = NO;

		  XDefineCursor(xDisplay, xWindow,
		    [(XGCursor*)[NSCursor arrowCursor] xCursor]);
		  XRecolorCursor(xDisplay,
		    [(XGCursor*)[NSCursor arrowCursor] xCursor], &xb,&xw);
		}

	      if (mouseWindow != rootWindow && mouseWindow != xWindow)
		{
		  if (mouseWindow == lastMouseWindow)
		    {
		      NSDebugLLog(@"NSDragging", @"sending dnd pos\n");
		      if (dragLocal)
			[self _sendLocalEvent: GSAppKitDraggingUpdate
		                   action: action
		                 position: NSMakePoint(wx+imageSize.width+1,
						       wy+imageSize.height/2)
		                timestamp: CurrentTime
		                 toWindow: dragWindow];
		      else
			xdnd_send_position(&dnd, mouseWindow, xWindow,
					 xAction, wx+imageSize.width+1, 
					 wy+imageSize.height/2, CurrentTime);
		    }
		  else
		    {
		      dnd_aware = xdnd_is_dnd_aware(&dnd, mouseWindow,
					    &dnd.dragging_version, typelist);
		      if (dnd_aware)
			{
			  XDefineCursor(xDisplay, xWindow,
				[(XGCursor*)[NSCursor arrowCursor] xCursor]);
			  XRecolorCursor(xDisplay,
				[(XGCursor*)[NSCursor arrowCursor] xCursor], &xg,&xb);
			  
			  NSDebugLLog(@"NSDragging", @"sending dnd enter/pos\n");
			  if (dragLocal)
			    [self _sendLocalEvent: GSAppKitDraggingEnter
		                   action: action
		                 position: NSMakePoint(wx+imageSize.width+1,
						       wy+imageSize.height/2)
		                timestamp: CurrentTime
		                 toWindow: dragWindow];
			  else
			    {
			      xdnd_send_enter(&dnd, mouseWindow, xWindow, 
					      typelist);
			      xdnd_send_position(&dnd, mouseWindow, xWindow,
						 xAction, wx+imageSize.width+1,
						 wy+imageSize.height/2, 
						 CurrentTime);
			    }
			  lastMouseWindow = mouseWindow;
			}
		    }
		}
	    }
	}
      theEvent = [NSApp nextEventMatchingMask: eventMask
				    untilDate: theDistantFuture
				       inMode: NSEventTrackingRunLoopMode
				      dequeue: YES];
    }

  [NSEvent stopPeriodicEvents];
  //[window _releaseMouse: self];
  [window orderOut: nil];
  XDefineCursor(xDisplay, xWindow,
    [(XGCursor*)[NSCursor arrowCursor] xCursor]);
  XRecolorCursor(xDisplay, [(XGCursor*)[NSCursor arrowCursor] xCursor],&xb,&xw);

  if (dnd_aware && mouseWindow != (Window) None)
    {
      NSDebugLLog(@"NSDragging", @"sending dnd drop\n");
      XSetSelectionOwner(xDisplay, dnd.XdndSelection, xWindow, CurrentTime);
      if (dragLocal)
	[self _sendLocalEvent: GSAppKitDraggingDrop
		       action: 0
		     position: NSZeroPoint
		    timestamp: CurrentTime
		     toWindow: dragWindow];
      else
	xdnd_send_drop (&dnd, lastMouseWindow, xWindow, CurrentTime);
    }
  else
    {
      if (!(mouseWindow=_findXWindow(xDisplay, rootWindow, rootWindow, wx, wy)))
	return;
      else
	{
	/*
	  id	browser = [NSApp delegate];	// browser, DO NOT rely on this
	  BOOL	exists = NO, is_dir = NO;	// FIX ME s/b moved to Workspac
	  NSFileManager *fm = [NSFileManager defaultManager];
	  BOOL altKeyDown = [theEvent modifierFlags] & NSAlternateKeyMask;
	  NSMutableString *s = AUTORELEASE([[NSMutableString alloc]
				initWithString: [browser path]]);

	  exists = [fm fileExistsAtPath: s isDirectory: &is_dir];
	  // if ALT key was held down user wants to change to dir
	  if ((exists) && (is_dir) && altKeyDown)
	    {
	      [s insertString: @"cd " atIndex: 0];
	      [s appendString: @"\n"];
	    }
	  // send path string to X window under the cursor
	  if (!_sendXString(xDisplay, mouseWindow, [s cString]))
	    NSLog(@"XGDragWindow: error sending string to X window\n");
	  */
	}
    }
}

- (void) setImage: (NSImage *)anImage
{
  [dragCell setImage: anImage];

  [self lockFocus];
  [dragCell drawWithFrame: NSMakeRect(0,0,48,48) inView: self];
  [self unlockFocus];
}

/* NSDraggingInfo protocol */
- (NSWindow *)draggingDestinationWindow
{
  return dragWindow;
}

- (NSPoint)draggingLocation
{
  return dragPoint;
}

- (NSPasteboard *)draggingPasteboard
{
  return dragPasteboard;
}

- (int)draggingSequenceNumber
{
  return dragSequence;
}

- (id)draggingSource
{
  return dragSource;
}

- (NSDragOperation)draggingSourceOperationMask
{
  if (dragSource)
    return [dragSource draggingSourceOperationMaskForLocal: dragLocal];
  else
    return dragMask;
}

- (NSImage *)draggedImage
{
  if (dragSource)
    return [dragCell image];
  else
    return nil;
}

- (NSPoint)draggedImageLocation
{
  NSPoint loc;
  if (dragSource)
    {
      NSSize size;
      size = [[dragCell image] size];
      loc = NSMakePoint(dragPoint.x-size.width/2, dragPoint.y - size.height/2);
    }
  else
    loc = dragPoint;
  return loc;
}

- (void)slideDraggedImageTo:(NSPoint)screenPoint
{
}

@end

