/* 
   NSDPSContext.m

   Encapsulation of Display Postscript contexts

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Scott Christley <scottc@net-community.com>
   Date: 1996
   
   This file is part of the GNUstep GUI Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>
#include <math.h>
#include <Foundation/NSString.h>
#include <Foundation/NSThread.h>
#include <Foundation/NSLock.h>
#include <Foundation/NSData.h>
#include <Foundation/NSDictionary.h>
#include <gnustep/xdps/NSDPSContext.h>
#include <gnustep/xdps/DPSOperators.h>
#include <X11/keysym.h>
#include <X11/keysymdef.h>
#include <X11/extensions/shape.h>

#include <AppKit/NSAffineTransform.h>
#include <AppKit/NSView.h>
#include <AppKit/NSWindow.h>

#define BOOL XWINDOWSBOOL
#include <DPS/dpsXclient.h>
#include <DPS/dpsXshare.h>
#undef BOOL

#include "PXKDrawingEngine.h"
#include "general.h"
#include "extensions.h"
#include "SharedX/xrtools.h"

//
// DPS exceptions
//
NSString *DPSPostscriptErrorException = @"DPSPostscriptErrorException";
NSString *DPSNameTooLongException = @"DPSNameTooLongException";
NSString *DPSResultTagCheckException = @"DPSResultTagCheckException";
NSString *DPSResultTypeCheckException = @"DPSResultTypeCheckException";
NSString *DPSInvalidContextException = @"DPSInvalidContextException";
NSString *DPSSelectException = @"DPSSelectException";
NSString *DPSConnectionClosedException = @"DPSConnectionClosedException";
NSString *DPSReadException = @"DPSReadException";
NSString *DPSWriteException = @"DPSWriteException";
NSString *DPSInvalidFDException = @"DPSInvalidFDException";
NSString *DPSInvalidTEException = @"DPSInvalidTEException";
NSString *DPSInvalidPortException = @"DPSInvalidPortException";
NSString *DPSOutOfMemoryException = @"DPSOutOfMemoryException";
NSString *DPSCantConnectException = @"DPSCantConnectException";

#define XDPY (((RContext *)context)->dpy)
#define XDRW (((RContext *)context)->drawable)
#define XSCR (((RContext *)context)->screen_number)

//
// Class variables
//
static BOOL GNU_CONTEXT_TRACED = NO;
static BOOL GNU_CONTEXT_SYNCHRONIZED = NO;

static NSDPSContext *context_list = nil;

enum {
  A_COEFF = 0,
  B_COEFF,
  C_COEFF,
  D_COEFF,
  TX_CONS,
  TY_CONS
};

Atom	WM_STATE;

extern int XGErrorHandler(Display *display, XErrorEvent *err);

@interface NSDPSContext (Window)
- (void) _setupRootWindow;
@end

@interface NSDPSContext (Private)
- (void) setupRunLoopInputSourcesForMode: (NSString*)mode;
@end

@implementation NSDPSContext

+ (void)initialize
{
  if (self == [NSDPSContext class])
    {
      // Set initial version
      [self setVersion: 1];

     GNU_CONTEXT_TRACED = NO;
     GNU_CONTEXT_SYNCHRONIZED = NO;
    }
}

- _initXContext
{
  Display  *dpy;
  int      screen_number;
  NSString *display_name;
  RContext *rcontext;
  
  display_name = [context_info objectForKey: @"DisplayName"];
  if (display_name)
    dpy = XOpenDisplay([display_name cString]);
  else
    dpy = XOpenDisplay(NULL);

  /* Use the fact that the screen number is specified like an extension
     e.g. hostname:0.1 */
  screen_number = [[display_name pathExtension] intValue];

  if (dpy == NULL)
    {
      char *dname = XDisplayName([display_name cString]);
      NSLog(@"Unable to connect to X Server %s", dname);
      exit(1);
    }

  OBJC_MALLOC(rcontext, RContext, 1);
  context = (void *)rcontext;
  XDPY = dpy;
  XSCR = screen_number;
  rcontext->depth = DefaultDepth(XDPY, XSCR);
  rcontext->black = BlackPixel(XDPY, XSCR);
  rcontext->white = WhitePixel(XDPY, XSCR);
  rcontext->cmap = DefaultColormap(XDPY, XSCR);
  rcontext->visual = DefaultVisual(XDPY, XSCR);

  /* Create a window for initial drawing */
  /* This window is never mapped */
  XDRW = XCreateSimpleWindow(XDPY, RootWindow(XDPY, XSCR),
				      0, 0, 100, 100, 1, 1,
				      rcontext->black);

  XSetErrorHandler(XGErrorHandler);

  [self _setupRootWindow];
  return self;
}

//
// Initializing a Context
//
- init
{
  NSMutableData *data = [NSMutableData data];

  return [self initWithMutableData: data
	       forDebugging: NO
	       languageEncoding: dps_ascii
	       nameEncoding: dps_strings
	       textProc: NULL
	       errorProc: NULL];
}

- (id) initWithContextInfo: (NSDictionary *)info
{
  [super initWithContextInfo: info];

  is_screen_context = YES;
  //error_proc = errorProc;
  //text_proc = tProc;
  chained_parent = nil;
  chained_child = nil;

  [self _initXContext];

  if (!XDPSExtensionPresent(XDPY)) 
    {
#if HAVE_DPS_DPSNXARGS_H    
      /* Make it possible for this client to start a DPS NX agent */
      XDPSNXSetClientArg(XDPSNX_AUTO_LAUNCH, (void *)True);
#else
      NSLog (@"DPS extension not in server!");
      exit (1);
#endif
    }

  // create the DPS context
  [self createDPSContext];

  /*
   * Make self the current context 'cos some of our initialization needs it.
   */
  [isa setCurrentContext: self];

  [self setupRunLoopInputSourcesForMode: NSDefaultRunLoopMode];
  [self setupRunLoopInputSourcesForMode: NSConnectionReplyMode];
  [self setupRunLoopInputSourcesForMode: NSModalPanelRunLoopMode];
  [self setupRunLoopInputSourcesForMode: NSEventTrackingRunLoopMode];

  WM_STATE = XInternAtom(XDPY, "WM_STATE", False);

  /* Add context to global list of contexts */
  next_context = context_list;
  context_list = self;

  return self;
}

- initWithMutableData:(NSMutableData *)data
	 forDebugging:(BOOL)debug
     languageEncoding:(DPSProgramEncoding)langEnc
	 nameEncoding:(DPSNameEncoding)nameEnc
	     textProc:(DPSTextProc)tProc
	    errorProc:(DPSErrorProc)errorProc
{
  ASSIGN(context_data, data);
  [self initWithContextInfo: nil];
  return self;
}

- (void)dealloc
{
  if (chained_child != nil)
    {
      /* This could be ugly, but...
           we need to make sure all our X resources are freed before
	   we close our display, thus destroy everything, then add ourselves
	   back to the autorelease list so we get called again a final time
      */
      DESTROY(chained_child);
      [[self retain] autorelease];
    }
  else 
    {
      DPSDestroySpace(DPSSpaceFromContext(dps_context));
      XFreeGC (XDPY, ((RContext *)context)->copy_gc);
      XDestroyWindow (XDPY, XDRW);
      XCloseDisplay(XDPY);
      /* Remove context from global list of contexts */
      {
        NSDPSContext *ctxt = context_list, *previous=nil;
        
        while(ctxt) 
	  {
	    if(ctxt == self)
	      break;
	    previous = ctxt;
	    ctxt = ctxt->next_context;
	  }
        if(!ctxt)
          NSLog(@"Internal Error: Couldn't find context to delete");
        else 
	  {
	    if(previous)
	      previous->next_context = next_context;
	    else
	      context_list = next_context;
	  }
      }
      [super dealloc];
    }
}

//
// Testing the Drawing Destination
//
- (BOOL)isDrawingToScreen
{
  return is_screen_context;
}

- (NSDPSContext *)DPSContext
{
  return self;
}

//
// Controlling the Context
//
- (void)interruptExecution
{}

//- (void)notifyObjectWhenFinishedExecuting:(id <NSDPSContextNotification>)obj
//{}

- (void)resetCommunication
{}

- (void)wait
{
  DPSWaitContext (dps_context);
}

+ (void) waitAllContexts
{
  NSDPSContext *ctxt;
  ctxt = context_list;
  while(ctxt) 
    {
      [ctxt wait];
      ctxt = ctxt->next_context;
    }
}

//
// Managing Returned Text and Errors
//
+ (NSString *)stringForDPSError:(const DPSBinObjSeqRec *)error
{
  return nil;
}

- (DPSErrorProc)errorProc
{
  return error_proc;
}

- (void)setErrorProc:(DPSErrorProc)proc
{
  error_proc = proc;
}

- (void)setTextProc:(DPSTextProc)proc
{
  text_proc = proc;
}

- (DPSTextProc)textProc
{
  return text_proc;
}

//
// Sending Raw Data
//
- (void)printFormat:(NSString *)format,...
{}

- (void)printFormat:(NSString *)format arguments:(va_list)argList
{}

- (void)writeData:(NSData *)buf
{}

- (void)writePostScriptWithLanguageEncodingConversion:(NSData *)buf
{}

//
// Managing Binary Object Sequences
//
- (void)awaitReturnValues
{}

- (void)writeBOSArray:(const void *)data
		count:(unsigned int)items
	       ofType:(DPSDefinedType)type
{}

- (void)writeBOSNumString:(const void *)data
		   length:(unsigned int)count
		   ofType:(DPSDefinedType)type
		    scale:(int)scale
{}

- (void)writeBOSString:(const void *)data
		length:(unsigned int)bytes
{}

- (void)writeBinaryObjectSequence:(const void *)data
			   length:(unsigned int)bytes
{}

- (void)updateNameMap
{}

//
// Managing Chained Contexts
//
- (void)setParentContext:(NSDPSContext *)parent
{
  chained_parent = parent;
}

- (void)chainChildContext:(NSDPSContext *)child
{
  if (child)
    {
      chained_child = [child retain];
      [child setParentContext: self];
    }
}

- (NSDPSContext *)childContext
{
  return chained_child;
}

- (NSDPSContext *)parentContext
{
  return chained_parent;
}

- (void)unchainContext
{
  if (chained_child)
    {
      [chained_child setParentContext: nil];
      [chained_child release];
      chained_child = nil;
    }
}

//
// Debugging Aids
//
+ (BOOL)areAllContextsOutputTraced
{
  return GNU_CONTEXT_TRACED;
}

+ (BOOL)areAllContextsSynchronized
{
  return GNU_CONTEXT_SYNCHRONIZED;
}

+ (void)setAllContextsOutputTraced:(BOOL)flag
{
  GNU_CONTEXT_TRACED = flag;
}

+ (void)setAllContextsSynchronized:(BOOL)flag
{
  GNU_CONTEXT_SYNCHRONIZED = flag;
}

- (BOOL)isOutputTraced
{
  return is_output_traced;
}

- (BOOL)isSynchronized
{
  return is_synchronized;
}

- (void)setOutputTraced:(BOOL)flag
{
  is_output_traced = flag;
  XDPSChainTextContext(dps_context, flag);
}

- (void)setSynchronized:(BOOL)flag
{
  is_synchronized = flag;
}

@end

static void
myDPSErrorProc (DPSContext ctxt, DPSErrorCode errCode, unsigned arg1, unsigned args)
{
  DPSDefaultErrorProc (ctxt, errCode, arg1, arg1);
}

//
// Methods for XWindows implementation
//
@implementation NSDPSContext (GNUstepXDPS)

+ (Display*) currentXDisplay
{
  return [(NSDPSContext*)[self currentContext] xDisplay];
}

- (Display*)xDisplay
{
  return XDPY;
}

- (void *) xrContext
{
  return context;
}

- (DPSContext)xDPSContext
{
  return dps_context;
}

- (void)createDPSContext
{
  int x, y, supported;
  unsigned long valuemask;
  XGCValues values;

  /* Create a GC for the initial window */
  values.foreground = ((RContext *)context)->black;
  values.background = ((RContext *)context)->white;
  values.function = GXcopy;
  values.plane_mask = AllPlanes;
  values.clip_mask = None;
  valuemask = (GCFunction | GCPlaneMask | GCClipMask 
         | GCForeground|GCBackground);
  ((RContext *)context)->copy_gc = 
    XCreateGC(XDPY, XDRW, valuemask, &values);

  /* Create the context if need be */
  if (!dps_context)
    {
      /* Pass None as the drawable argument; the program will execute correctly
         but will not render any text or graphics. */
      dps_context = XDPSCreateSimpleContext(XDPY, None, 
					    ((RContext *)context)->copy_gc, 
					    0, 0,
					    DPSDefaultTextBackstop,
					    (DPSErrorProc)myDPSErrorProc, NULL);
      if (dps_context == NULL)
	{
	  NSLog(@"Could not connect to DPS\n");
	  NSLog(@"Trying again...\n");
       	  dps_context = XDPSCreateSimpleContext(XDPY, None, 
					   ((RContext *)context)->copy_gc,
					   0, 0,
					   DPSDefaultTextBackstop,
					   (DPSErrorProc)myDPSErrorProc, NULL);

	  if (dps_context == NULL)
	    {
	      NSLog(@"DPS is not available\n");
	      exit(1);
	    }
	}

      // Make it the active context
      DPSSetContext(dps_context);
      XDPSRegisterContext(dps_context, NO);

      // Use pass-through event handling
      XDPSSetEventDelivery(XDPY, dps_event_pass_through);
    }

  PSWinitcontext (XGContextFromGC (((RContext *)context)->copy_gc), 
		  XDRW, 0, 0);
  PSWGetTransform (dps_context, ctm, invctm, &x, &y);
  PSWinitcontext (XGContextFromGC (((RContext *)context)->copy_gc), 
  		  None, 0, 0);
  PSWRevision(&dps_revision);

  /* Check for operator extensions */
  DPSWKnownExtensions(dps_context, &ext_flags);
  /* Check if composite-related extensions work */
  /* FIXME: This crashes DPS on some implementations */
  //DPSWWorkingExtensions(dps_context, &supported);
  supported = 0;
  if (supported == 0)
    ext_flags = (ext_flags 
		 & ~(COMPOSITE_EXT | ALPHAIMAGE_EXT | COMPOSITERECT_EXT
		     | DISSOLVE_EXT | READIMAGE_EXT | SETALPHA_EXT));
  /* FIXME: alphaimage and composite work badly in DGS 5.50. Perhaps when 
     we find a version that works we can put an additional test here. For now,
     just turn them off.
  */
  ext_flags = (ext_flags & ~(COMPOSITE_EXT | ALPHAIMAGE_EXT | DISSOLVE_EXT ));

  NSDebugLLog(@"NSDPSContext", @"Using DPS Revision: %d\n", dps_revision);
  NSDebugLLog(@"NSDPSContext", @"DPS Default Matrix: [%f %f %f %f %f %f]\n", 
	ctm[0], ctm[1], ctm[2], ctm[3], ctm[4], ctm[5]);
  NSDebugLLog(@"NSDPSContext", @"DPS Extensions flags: %d\n", ext_flags); 
  if ([[NSUserDefaults standardUserDefaults] boolForKey: @"DPSDefaultMatrix"]
      == NO)
    {
      NSDebugLog(@"Reseting default matrix\n");
      ctm[0] /= fabs(ctm[0]);
      ctm[3] /= fabs(ctm[3]);
      PSWSetMatrix(ctm);
    }
}

- (void) flush
{
  DPSFlushContext(dps_context);
}

- (Window) xDisplayRootWindow
{
  return RootWindow(XDPY, XSCR);
}

- (Window) xAppRootWindow
{
  return XDRW;
}

- (XColor)xColorFromColor: (XColor)color
{
  Colormap colormap = XDefaultColormap(XDPY, XSCR);
  XAllocColor(XDPY, colormap, &color);
  return color;
}

- (void) _localTransform: (float *)local_ctm inverse: (float *)local_inv
           offset: (NSPoint *)offset
{
  int x, y;
  if (local_ctm == NULL || local_inv == NULL)
    return;
  PSWGetTransform (dps_context, local_ctm, local_inv, &x, &y);
  if (offset)
    {
      offset->x = x;
      offset->y = y;
    }
}

- (NSPoint)userPointFromXPoint:(NSPoint)xPoint
{
  float lctm[6], linv[6];
  NSPoint offset, userPoint;

  [self _localTransform: lctm inverse: linv offset: &offset];
  //xPoint.x -= offset.x;
  //xPoint.y -= offset.y;
  userPoint.x = linv[A_COEFF] * xPoint.x + linv[C_COEFF] * xPoint.y
  		+ linv[TX_CONS];
  userPoint.y = linv[B_COEFF] * xPoint.x + linv[D_COEFF] * xPoint.y
                + linv[TY_CONS];
  return userPoint;
}

- (NSPoint)XPointFromUserPoint:(NSPoint)userPoint
{
  float lctm[6], linv[6];
  NSPoint offset, xPoint;

  [self _localTransform: lctm inverse: linv offset: &offset];
  xPoint.x = lctm[A_COEFF] * userPoint.x + lctm[C_COEFF] * userPoint.y
  		+  lctm[TX_CONS] + offset.x;
  xPoint.y = lctm[B_COEFF] * userPoint.x + lctm[D_COEFF] * userPoint.y
                +  lctm[TY_CONS] + offset.y;
  xPoint.x = floor (xPoint.x);
  xPoint.y = floor (xPoint.y);
  NSDebugLLog(@"CTM", @"Matrix [%f,%f,%f,%f,%f,%f] (%f,%f)\n",
  	lctm[A_COEFF], lctm[B_COEFF], lctm[C_COEFF], lctm[D_COEFF], 
	lctm[TX_CONS], lctm[TY_CONS], offset.x, offset.y);
  return xPoint;
}

- (NSRect)userRectFromXRect:(NSRect)xrect
{
  float lctm[6], linv[6];
  float x, y, w, h;

  [self _localTransform: lctm inverse: linv offset: NULL];
  x = linv[A_COEFF] * NSMinX(xrect) + linv[C_COEFF] * NSMinY(xrect)
  		+ linv[TX_CONS];
  y = linv[B_COEFF] * NSMinX(xrect) + linv[D_COEFF] * NSMinY(xrect)
                + linv[TY_CONS];
  w = linv[A_COEFF] * NSWidth(xrect) + linv[C_COEFF] * NSHeight(xrect);
  h = linv[B_COEFF] * NSWidth(xrect) + linv[D_COEFF] * NSHeight(xrect);
  if (h < 0)
    y -= h;
  h = fabs(h);
  if (w < 0)
    x -= w;
  w = fabs(w);
  return NSMakeRect(x, y, w, h);
}

- (NSRect)XRectFromUserRect:(NSRect)urect
{
  NSPoint offset;
  float lctm[6], linv[6];
  float x, y, w, h;

  [self _localTransform: lctm inverse: linv offset: &offset];
  x = lctm[A_COEFF] * NSMinX(urect) + lctm[C_COEFF] * NSMinY(urect)
  		+  lctm[TX_CONS] + offset.x;
  y = lctm[B_COEFF] * NSMinX(urect) + lctm[D_COEFF] * NSMinY(urect)
                +  lctm[TY_CONS] + offset.y;
  w = lctm[A_COEFF] * NSWidth(urect) + lctm[C_COEFF] * NSHeight(urect);
  h = lctm[B_COEFF] * NSWidth(urect) + lctm[D_COEFF] * NSHeight(urect);
  NSDebugLLog(@"CTM", @"Matrix [%f,%f,%f,%f,%f,%f] (%f,%f)\n",
  	lctm[A_COEFF], lctm[B_COEFF], lctm[C_COEFF], lctm[D_COEFF], 
	lctm[TX_CONS], lctm[TY_CONS], offset.x, offset.y);
  if (h < 0)
    y += h;
  h = fabs(floor(h));
  if (w < 0)
    x += w;
  w = fabs(floor(w));
  x = floor(x);
  y = floor(y);
  return NSMakeRect(x, y, w, h);
}

- (op_extensions_t) operatorExtensions
{
  return ext_flags;
}

@end
