/******************************** -*- C -*- ****************************
 *
 *	Byte Code Interpreter Module.
 *	This interprets the compiled bytecodes of a method.
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 2001, 2002 Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later
 * version.
 *
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.	 If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 ***********************************************************************/


/* This is basically how the interpreter works:
   
   The interpreter expects to be called in an environment where there
   already exists a well-defined method context.  The instruction
   pointer, stored in the global variable "ip", and the stack pointer,
   stored in the global variable "sp", should be set up to point into
   the current method and gst_method_context.  Other global variables,
   such as "_gst_this_method", "_gst_self", "_gst_temporaries",
   etc. should also be setup.  See the routine
   _gst_prepare_execution_environment for details.
   
   The interpreter checks to see if any change in its state is
   required, such as switching to a new process, dealing with an
   asynchronous signal and printing out the byte codes that are being
   executed, if that was requested by the user.
   
   After that, the byte code that ip points to is fetched and decoded.
   Some byte codes perform jumps, which are performed by merely
   adjusting the value of ip.  Some are message sends, which are
   described in more detail below.  Some instructions require more
   than one byte code to perform their work; ip is advanced as needed
   and the extension byte codes are fetched.  Some byte codes, which
   push booleans, are usually followed by jump byte codes; in this
   case, and if JUMP_LOOKAHEAD is defined, the two byte codes are
   merged for speed.
   
   After dispatching the byte code, the interpreter loops around to
   execute another byte code.  A particular bytecode signals that the
   execution of the method is over, and that the interpreter should
   return to its caller.  This bytecode is never generated by the
   compiler, it is only present in a private #'__terminate' method
   that is generated when bootstrapping.

   Note that the interpreter is not called recursively to implement
   message sends.  Rather the state of the interpreter is saved away
   in the currently executing context, and a new context is created
   and the global variables such as ip, sp, and _gst_temporaries are
   initialized accordingly.

   When a message send occurs, the _gst_send_message_internal routine
   is invoked.  It determines the class of the receiver, and checks to
   see if it already has cached the method definition for the given
   selector and receiver class.  If so, that method is used, and if
   not, the receiver's method dictionary is searched for a method with
   the proper selector.  If it's not found in that method dictionary,
   the method dictionary of the classes parent is examined, and on up
   the hierarchy, until a matching selector is found.

   If no selector is found, the receiver is sent a #doesNotUnderstand:
   message to indicate that a matching method could not be found.  The
   stack is modified, pushing a gst_message object that embeds
   information about the original selector and arguments, and
   _gst_send_message_internal calls itself recursively to look up
   #doesNotUnderstand:.  Note that if the object does not understand
   in turn the #doesNotUnderstand: message, a crash is extremely
   likely; things like this are however to be expected, since you're
   really playing a bad game and going against some logical things
   that the VM assumes for speed.

   If a method is found, it is examined for some special cases.  The
   special cases are primitive return of _gst_self, return of an
   instance variable, return of a literal object, or execution of a
   primitive method definition. This latter operation is performed by
   the _gst_execute_primitive_operation routine.  If the execution of
   this primitive interpreter fails, the normal message send operation
   is performed.

   If the found method is not one of the special cases, or if it is a
   primitive that failed to execute, a "normal" message send is
   performed.  This basically entails saving away what state the
   interpreter has, such as the values of ip, and sp, being careful to
   save their relative locations and not their physical addresses,
   because one or more garbage collections could occur before the
   method context is returned to, and the absolute pointers would be
   invalid.

   The SEND_MESSAGE routine then creates a new gst_method_context
   object, makes its parent be the currently executing
   gst_method_context, and sets up the interpreters global variables
   to reference the new method and new gst_method_context.  Once those
   variables are set, SEND_MESSAGE returns to the interpreter, which
   cheerfully begins executing the new method, totally unaware that
   the method that it was executing has changed.

   When a method returns, the context that called it is examined to
   restore the interpreter's global variables to the state that they
   were in before the callee was invoked.  The values of ip and sp are
   restored to their absolute address values, and the other global
   state variables are restored accordingly.  After the state has been
   restored, the interpreter continues execution, again totally
   oblivious to the fact that it's not running the same method it was
   on its previous byte code.

   Blocks are similarly implemented by send_block_value, which is
   simpler than _gst_send_message_internal however, because it
   contains no check for special cases, and no method lookup logic.
   Unlike the Blue Book, GNU Smalltalk stores bytecodes for blocks
   into separate CompiledBlock objects, not into the same
   CompiledMethods that holds the containing bytecodes.
   send_block_value expects to find a BlockClosure on the stack, and
   this BlockClosure object points to the CompiledBlock object to be
   activated.  */


#if defined(USE_GCC_DISPATCH) && !defined(PIPELINING)
static void * const *global_monitored_bytecodes;
static void * const *global_normal_bytecodes;
static void * const *dispatch_vec;
#endif

#define GET_CONTEXT_IP(ctx) 	TO_INT((ctx)->ipOffset)

#define SET_THIS_METHOD(method, ipOffset) do {				\
  gst_compiled_method _method = (gst_compiled_method) OOP_TO_OBJ (_gst_this_method = (method));	\
  method_base = _method->bytecodes;					\
  _gst_literals = OOP_TO_OBJ (_method->literals)->data;				\
  ip = method_base + (ipOffset);						\
} while(0)

#define SEND_TO_SUPER(sendSelector, sendArgs, dummy) \
    do { \
      OOP _methodClass; \
      _methodClass = GET_METHOD_CLASS (_gst_this_method); \
      _methodClass = SUPERCLASS (_methodClass); \
      _gst_send_message_internal(sendSelector, sendArgs, _gst_self, _methodClass); \
    } while(0)

#ifdef OPTIMIZE
#define INSTANCE_VARIABLE(receiver, index) \
  (OOP_TO_OBJ (receiver)->data[index])

#define STORE_INSTANCE_VARIABLE(receiver, index, oop) \
  OOP_TO_OBJ (receiver)->data[index] = (oop)

#define IN_BOUNDS(oop, index) true
#else
#define INSTANCE_VARIABLE(receiver, index) \
  (IN_BOUNDS (receiver, index) ? OOP_TO_OBJ (receiver)->data[index] \
    : (_gst_errorf("Index out of bounds %d", index), abort (), _gst_nil_oop))

#define STORE_INSTANCE_VARIABLE(receiver, index, oop) \
do {  \
  OOP __storeRecVarOOP = (oop); \
  if (!IN_BOUNDS (receiver, index)) { \
    _gst_errorf("Index out of bounds %d", index); \
    abort (); \
  } \
  OOP_TO_OBJ (receiver)->data[index] = __storeRecVarOOP; \
} while(0)

#define IN_BOUNDS(oop, index) ( \
	((unsigned int)(index)) < (unsigned int)NUM_OOPS (OOP_TO_OBJ (oop)) )
#endif /* OPTIMIZE */

#if REG_AVAILABILITY >= 2 && defined(LOCAL_REGS)
#define RECEIVER_VARIABLE(index)		  INSTANCE_VARIABLE (self_cache, index)
#define METHOD_TEMPORARY(index) 		  temp_cache[index]
#define METHOD_LITERAL(index) 		  lit_cache[index]
#define STORE_RECEIVER_VARIABLE(index, oop) STORE_INSTANCE_VARIABLE (self_cache, index, oop)
#define STORE_METHOD_TEMPORARY(index, oop)  temp_cache[index] = (oop)
#define STORE_METHOD_LITERAL(index, oop)    lit_cache[index] = (oop)
#else
#define RECEIVER_VARIABLE(index)		  INSTANCE_VARIABLE (_gst_self, index)
#define METHOD_TEMPORARY(index) 		  _gst_temporaries[index]
#define METHOD_LITERAL(index) 		  _gst_literals[index]
#define STORE_RECEIVER_VARIABLE(index, oop) STORE_INSTANCE_VARIABLE (_gst_self, index, oop)
#define STORE_METHOD_TEMPORARY(index, oop)  _gst_temporaries[index] = (oop)
#define STORE_METHOD_LITERAL(index, oop)    _gst_literals[index] = (oop)
#endif

#define METHOD_VARIABLE(index) \
  ASSOCIATION_VALUE (METHOD_LITERAL (index))

#define STORE_METHOD_VARIABLE(index, oop) \
  SET_ASSOCIATION_VALUE (METHOD_LITERAL (index), oop)

void
_gst_send_message_internal (OOP sendSelector, 
			    int sendArgs, 
			    OOP receiver,
			    OOP method_class)
{
  long hashIndex;
  OOP methodOOP;
  method_cache_entry * methodData;
  gst_method_context newContext;
  method_header header;

  /* hash the selector and the class of the receiver together using
     XOR.  Since both are addresses in the object table, and since
     object table entries are 2 longs in size, shift over by 3 bits
     (4 on 64-bit architectures) to remove the useless low order
     zeros. */

  _gst_sample_counter++;
  hashIndex = METHOD_CACHE_HASH (sendSelector, method_class);
  methodData = &method_cache[hashIndex];

  if UNCOMMON (methodData->selectorOOP != sendSelector
      || methodData->startingClassOOP != method_class)
    {
      /* :-( cache miss )-: */
      if (!lookup_method (sendSelector, methodData, sendArgs, method_class))
	{
	  SEND_MESSAGE (_gst_does_not_understand_colon_symbol, 1, false);
	  return;
	}
    }

  /* Note that execute_primitive_operation might invoke a call-in, and
     which might in turn modify the method cache in general and
     corrupt methodData in particular.  So, load everything before
     this can happen.  */

  header = methodData->methodHeader;
  methodOOP = methodData->methodOOP;

#ifndef OPTIMIZE
#ifdef DEBUG_CODE_FLOW
  {
#else /* !DEBUG_CODE_FLOW */
  if (header.numArgs != (unsigned) sendArgs)
    {
#endif /* !DEBUG_CODE_FLOW */
      OOP receiverClass;
      if (IS_INT (receiver))
	receiverClass = _gst_small_integer_class;
      else
	receiverClass = OOP_CLASS (receiver);

      if (methodData->methodClassOOP == receiverClass)
        printf ("%O>>%O\n", receiverClass, sendSelector);
      else
        printf ("%O(%O)>>%O\n", receiverClass, methodData->methodClassOOP, sendSelector);

      if (header.numArgs != (unsigned) sendArgs)
	{
	  _gst_errorf ("invalid number of arguments %d, expecting %d",
		       sendArgs, header.numArgs);

	  abort ();
	}
    }
#endif /* !OPTIMIZE */

  if UNCOMMON (header.headerFlag)
    {
      switch (header.headerFlag)
	{
	case 1:
	  /* 1, return the receiver - _gst_self is already on the stack...so we leave it */
	  _gst_self_returns++;
	  return;

	case 2:
	  {
	    long primIndex = header.primitiveIndex;
	    /* 2, return instance variable */
	    /* replace receiver with the returned instance variable */
	    SET_STACKTOP (INSTANCE_VARIABLE (receiver, primIndex));
	    _gst_inst_var_returns++;
	    return;
	  }

	case 3:
	  {
	    /* 3, return literal constant */
	    /* replace receiver with the returned literal constant */
	    SET_STACKTOP (GET_METHOD_LITERALS (methodOOP)[0]);
	    _gst_literal_returns++;
	    return;
	  }

	case 4:
	  if COMMON (!_gst_execute_primitive_operation(header.primitiveIndex,
						       sendArgs))
	    /* primitive succeeded.  Continue with the parent context */
	    return;

	  break;

	case 0:		/* only here so that the compiler skips a range check */
	case 5:
	case 6:
	case 7:
	default:
	  break;
	}
    }

  /* Prepare new state.  */

  newContext = activate_new_context (header.stack_depth, sendArgs);
  newContext->flags = MCF_IS_METHOD_CONTEXT;
  SET_THIS_METHOD (methodOOP, 0);
  _gst_self = receiver;

  /* push args and temps, set sp and _gst_temporaries */
  prepare_context ((gst_context_part) newContext, sendArgs, header.numTemps);
}

void
_gst_send_method (OOP methodOOP)
{
  int sendArgs;
  OOP receiver;
  method_header header;
  REGISTER (1, gst_compiled_method method);
  REGISTER (2, gst_method_context newContext);

  _gst_sample_counter++;

  method = (gst_compiled_method) OOP_TO_OBJ (methodOOP);
  header = method->header;
  sendArgs = header.numArgs;
  receiver = STACK_AT (sendArgs);

  if UNCOMMON (header.headerFlag)
    {
      switch (header.headerFlag)
	{
	case 1:
	  /* 1, return the receiver - _gst_self is already on the stack...so we leave it */
	  _gst_self_returns++;
	  return;

	case 2:
	  {
	    long primIndex = header.primitiveIndex;
	    /* 2, return instance variable */
	    /* replace receiver with the returned instance variable */
	    SET_STACKTOP (INSTANCE_VARIABLE (receiver, primIndex));
	    _gst_inst_var_returns++;
	    return;
	  }

	case 3:
	  {
	    /* 3, return literal constant */
	    /* replace receiver with the returned literal constant */
	    SET_STACKTOP (GET_METHOD_LITERALS (methodOOP)[0]);
	    _gst_literal_returns++;
	    return;
	  }

	case 4:
	  if COMMON (!_gst_execute_primitive_operation(header.primitiveIndex, 
						       sendArgs))
	    /* primitive succeeded.  Continue with the parent context */
	    return;

	  /* primitive failed.  Invoke the normal method.  */
	  break;

	case 0:		/* only here so that the compiler skips a range check */
	case 5:
	case 6:
	case 7:
	default:
	  break;
	}
    }

  /* prepare new state */
  newContext = activate_new_context (header.stack_depth, sendArgs);
  newContext->flags = MCF_IS_METHOD_CONTEXT;
  SET_THIS_METHOD (methodOOP, 0);
  _gst_self = receiver;

  /* push args and temps, set sp and _gst_temporaries */
  prepare_context ((gst_context_part) newContext, sendArgs, header.numTemps);
}


static mst_Boolean
send_block_value (int numArgs)
{
  OOP closureOOP;
  block_header header;
  REGISTER (1, gst_block_context blockContext);
  REGISTER (2, gst_block_closure closure);

  closureOOP = STACK_AT (numArgs);
  closure = (gst_block_closure) OOP_TO_OBJ (closureOOP);
  header = ((gst_compiled_block) OOP_TO_OBJ (closure->block))->header;
  if UNCOMMON (numArgs != header.numArgs)
    {
      /* check numArgs asap */
      return (true);
    }

  /* prepare the new state, loading data from the closure */
  /* gc might happen - so reload everything. */
  blockContext =
    (gst_block_context) activate_new_context (header.depth, numArgs);
  closure = (gst_block_closure) OOP_TO_OBJ (closureOOP);
  blockContext->outerContext = closure->outerContext;
  _gst_self = closure->receiver;
  SET_THIS_METHOD (closure->block, 0);

  /* push args and temps */
  prepare_context ((gst_context_part) blockContext, numArgs, header.numTemps);
  return (false);
}

void
_gst_validate_method_cache_entries (void)
{
}



OOP
_gst_interpret (OOP processOOP)
{
/******************** REGISTER DECLARATIONS *************************/

#ifdef LOCAL_REGS
# undef  sp
# undef  ip

#if REG_AVAILABILITY == 0
# define LOCAL_COUNTER    _gst_bytecode_counter
# define EXPORT_REGS()	  (_gst_sp = sp, _gst_ip = ip)
#else
  int LOCAL_COUNTER = 0;
# define EXPORT_REGS()	  (_gst_sp = sp, _gst_ip = ip, \
			   _gst_bytecode_counter += LOCAL_COUNTER, \
			   LOCAL_COUNTER = 0)
#endif

/* If we have a good quantity of registers, activate more caching mechanisms. */
#if  REG_AVAILABILITY >= 2
  OOP self_cache, *temp_cache, *lit_cache;
  OOP my_nil_oop = _gst_nil_oop, my_true_oop =
    _gst_true_oop, my_false_oop = _gst_false_oop;

# define _gst_nil_oop my_nil_oop
# define _gst_true_oop my_true_oop
# define _gst_false_oop my_false_oop
# define IMPORT_REGS()	(sp = _gst_sp, ip = _gst_ip, \
  			 self_cache = _gst_self, temp_cache = _gst_temporaries, \
  			 lit_cache = _gst_literals)

#else
#define IMPORT_REGS()	(sp = _gst_sp, ip = _gst_ip)
#endif

  REGISTER (1, ip_type ip);
  REGISTER (2, OOP * sp);
#else
#  define EXPORT_REGS()
#  define IMPORT_REGS()
#endif /* LOCAL_REGS */


#ifndef USE_GCC_DISPATCH

  /* Most of the difference in speed between switch-based dispatch and
     label-based dispatch comes from the fact that all bytecodes pass
     through an if in the former case, while when using labels we can
     use dispatch_vec instead of an if statement.
   
     Note that the `goto lab' in BC is completely gratuitous and it's
     only there to shut up the compiler's warnings. */

#define INTERPRETER		for(;;)
#define MONITOR_CODE		  if UNCOMMON (_gst_except_flag)
#define BYTECODES		  switch(*ip)
#define BC(num, lab)		    goto lab; case num: lab: START
#define NEXT_BC			      break

#else /* defined(USE_GCC_DISPATCH) */
  /* Indirect threaded bytecode interpretation when using GCC.
     NORMAL_BYTE_CODES is indexed by bytecode and holds onto the
     address of the label to jump to to execute that byte code.
     MONITORED_BYTE_CODES has all byte codes jump to a common place to
     check for exceptional conditions, and then jump thru
     NORMAL_BYTE_CODES.  DISPATCH_VEC points normally at
     NORMAL_BYTE_CODES, but when there is an exceptional condition, it
     points at MONITORED_BYTE_CODES.  TRUE_BYTE_CODES and
     FALSE_BYTE_CODES are used (if JUMP_LOOKAHEAD is defined) to
     dispatch conditional jumps immediately following comparisons (see
     comment in interp.c) without pushing and popping the result of
     the conditional.  */

  static void *const normal_byte_codes[256] = {
    /* 1-byte Push bytecodes */
    &&pushRecVar0, &&pushRecVar1, &&pushRecVar2, &&pushRecVar3,	/* 0 */
    &&pushRecVar4, &&pushRecVar5, &&pushRecVar6, &&pushRecVar7,	/* 4 */
    &&pushRecVar8, &&pushRecVar9, &&pushRecVar10, &&pushRecVar11,	/* 8 */
    &&pushRecVar12, &&pushRecVar13, &&pushRecVar14, &&pushRecVar15,	/* 12 */
    &&pushTemp0, &&pushTemp1, &&pushTemp2, &&pushTemp3,	/* 16 */
    &&pushTemp4, &&pushTemp5, &&pushTemp6, &&pushTemp7,	/* 20 */
    &&pushTemp8, &&pushTemp9, &&pushTemp10, &&pushTemp11,	/* 24 */
    &&pushTemp12, &&pushTemp13, &&pushTemp14, &&pushTemp15,	/* 28 */
    &&pushLit0, &&pushLit1, &&pushLit2, &&pushLit3,	/* 32 */
    &&pushLit4, &&pushLit5, &&pushLit6, &&pushLit7,	/* 36 */
    &&pushLit8, &&pushLit9, &&pushLit10, &&pushLit11,	/* 40 */
    &&pushLit12, &&pushLit13, &&pushLit14, &&pushLit15,	/* 44 */
    &&pushLit16, &&pushLit17, &&pushLit18, &&pushLit19,	/* 48 */
    &&pushLit20, &&pushLit21, &&pushLit22, &&pushLit23,	/* 52 */
    &&pushLit24, &&pushLit25, &&pushLit26, &&pushLit27,	/* 56 */
    &&pushLit28, &&pushLit29, &&pushLit30, &&pushLit31,	/* 60 */
    &&pushVar0, &&pushVar1, &&pushVar2, &&pushVar3,	/* 64 */
    &&pushVar4, &&pushVar5, &&pushVar6, &&pushVar7,	/* 68 */
    &&pushVar8, &&pushVar9, &&pushVar10, &&pushVar11,	/* 72 */
    &&pushVar12, &&pushVar13, &&pushVar14, &&pushVar15,	/* 76 */
    &&pushVar16, &&pushVar17, &&pushVar18, &&pushVar19,	/* 80 */
    &&pushVar20, &&pushVar21, &&pushVar22, &&pushVar23,	/* 84 */
    &&pushVar24, &&pushVar25, &&pushVar26, &&pushVar27,	/* 88 */
    &&pushVar28, &&pushVar29, &&pushVar30, &&pushVar31,	/* 92 */

    /* 1-byte Pop/Store bytecodes */
    &&stRecVar0, &&stRecVar1, &&stRecVar2, &&stRecVar3,	/* 96 */
    &&stRecVar4, &&stRecVar5, &&stRecVar6, &&stRecVar7,	/* 100 */
    &&stTemp0, &&stTemp1, &&stTemp2, &&stTemp3,	/* 104 */
    &&stTemp4, &&stTemp5, &&stTemp6, &&stTemp7,	/* 108 */

    /* Special push bytecodes */
    &&pushSelf, &&pushTrue, &&pushFalse, &&pushNil,	/* 112 */
    &&pushSig8, &&pushUnsig8, &&pushZero, &&pushOne,	/* 116 */

    /* Return bytecodes */
    &&retSelf, &&retTrue, &&retFalse, &&retNil,	/* 120 */
    &&explicitRet, &&retStackTop,	/* 124 */

    /* All sorts of bytecodes */
    &&bigLiteralOp,	/* 126 */

    /* Debugging */
    &&lineNo,	/* 127 */

    /* Long bytecodes */
    &&pushIndexedVal, &&storeStackTop, &&popAndStoreTop,	/* 128 */
    &&sendShort, &&sendLong, &&supSendShort, &&bigInstanceOp,	/* 131 */

    /* More stack bytecodes */
    &&popStack, &&dupStack, &&pushThisContext,	/* 135 */
    &&outerTempOp, &&nop, &&topSelf,	/* 138 */
    &&topOne, &&topIndexedVal,	/* 141 */

    /* Exit interpreter bytecode */
    &&endExecution,		/* 143 */

    /* jump bytecodes */
    &&shJmp1, &&shJmp2, &&shJmp3, &&shJmp4,	/* 144 */
    &&shJmp5, &&shJmp6, &&shJmp7, &&shJmp8,	/* 148 */
    &&shJmp1False, &&shJmp2False, &&shJmp3False, &&shJmp4False,	/* 152 */
    &&shJmp5False, &&shJmp6False, &&shJmp7False, &&shJmp8False,	/* 156 */
    &&longJmpNeg4, &&longJmpNeg3, &&longJmpNeg2, &&longJmpNeg1,	/* 160 */
    &&longJmp0, &&longJmp1, &&longJmp2, &&longJmp3,	/* 164 */
    &&popJmpTrue0, &&popJmpTrue1, &&popJmpTrue2, &&popJmpTrue3,	/* 168 */
    &&popJmpFalse0, &&popJmpFalse1, &&popJmpFalse2, &&popJmpFalse3,	/* 172 */

    /* Special 1-byte send bytecodes */
    &&sendPlus, &&sendMinus, &&sendLess, &&sendGreater,	/* 176 */
    &&sendLessEq, &&sendGreaterEq, &&sendEqual, &&sendNotEqual,	/* 180 */
    &&sendTimes, &&sendDivide, &&sendRemainder, &&sendAtSign,	/* 184 */
    &&sendBitShift, &&sendIntDivide, &&sendBitAnd, &&sendBitOr,	/* 188 */
    &&sendAt, &&sendAtPut, &&sendSize, &&sendNext,	/* 192 */
    &&sendNextPut, &&sendAtEnd, &&sendSameObject, &&sendClass,	/* 196 */
    &&sendBlockCopy, &&sendValue, &&sendValueColon, &&sendDoColon,	/* 200 */
    &&sendNew, &&sendNewColon, &&sendIsNil, &&sendNotNil,	/* 204 */

    /* General 1-byte send bytecodes */
    &&sendNoArg0, &&sendNoArg1, &&sendNoArg2, &&sendNoArg3,	/* 208 */
    &&sendNoArg4, &&sendNoArg5, &&sendNoArg6, &&sendNoArg7,	/* 212 */
    &&sendNoArg8, &&sendNoArg9, &&sendNoArg10, &&sendNoArg11,	/* 216 */
    &&sendNoArg12, &&sendNoArg13, &&sendNoArg14, &&sendNoArg15,	/* 220 */
    &&sendOneArg0, &&sendOneArg1, &&sendOneArg2, &&sendOneArg3,	/* 224 */
    &&sendOneArg4, &&sendOneArg5, &&sendOneArg6, &&sendOneArg7,	/* 228 */
    &&sendOneArg8, &&sendOneArg9, &&sendOneArg10, &&sendOneArg11,	/* 232 */
    &&sendOneArg12, &&sendOneArg13, &&sendOneArg14, &&sendOneArg15,	/* 236 */
    &&sendTwoArg0, &&sendTwoArg1, &&sendTwoArg2, &&sendTwoArg3,	/* 240 */
    &&sendTwoArg4, &&sendTwoArg5, &&sendTwoArg6, &&sendTwoArg7,	/* 244 */
    &&sendTwoArg8, &&sendTwoArg9, &&sendTwoArg10, &&sendTwoArg11,	/* 248 */
    &&sendTwoArg12, &&sendTwoArg13, &&sendTwoArg14, &&sendTwoArg15	/* 252 */
  };

#define MONITOR_CODE			monitor_byte_codes:		\
					  if COMMON (_gst_except_flag)

#define BC(num, lab)			lab: START

#ifdef PIPELINING
  gst_uchar b1, b2, b3, b4, b5;	/* pre-fetch queue */
  void *t1, *t2;		/* pre-decode queue */
  BRANCH_REGISTER (t);

#define BYTE1 b1
#define BYTE2 b2


#define LOAD1 (t = t1, t1 = t2, t2 = normal_byte_codes[b3],     \
               b1 = b2, b2 = b3, b3 = b4, b4 = b5,            \
               b5 = ip[6], ip++)

#define LOAD2 (t = t2, t1 = normal_byte_codes[b3],              \
               t2 = normal_byte_codes[b4], b1 = b3, b2 = b4,    \
               b3 = b5, b4 = ip[6], b5 = ip[7], ip += 2)

#define LOAD3 (t = normal_byte_codes[b3], t1 = normal_byte_codes[b4],	\
               t2 = normal_byte_codes[b5], b1 = b4, b2 = b5, b3 = ip[6],  \
               b4 = ip[7], b5 = ip[8], ip += 3)

#define FLUSH         goto monitor_byte_codes; }

#define CHECK_NEXT    IMPORT_REGS();                       \
                      t = normal_byte_codes[*ip],           \
                      b1 = ip[1], b2 = ip[2], b3 = ip[3], \
                      b4 = ip[4], b5 = ip[5],             \
                      t1 = t2 = &&monitor_byte_codes;       \
                      goto *t;

#define BYTECODES     t = normal_byte_codes[*ip],                           \
                      b1 = ip[1], b2 = ip[2], b3 = ip[3],                 \
                      b4 = ip[4], b5 = ip[5],                             \
                      t1 = normal_byte_codes[b1], t2 = normal_byte_codes[b2]; \
                      goto *t;

#define NEXT_BC       goto *t;

#undef JUMP_LOOKAHEAD

#else /* !PIPELINING */

  static void * const monitored_byte_codes[256] = {
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
    &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, &&monitor_byte_codes, 
  };


#define INTERPRETER					\
  global_normal_bytecodes = normal_byte_codes;		\
  global_monitored_bytecodes = monitored_byte_codes;	\
  dispatch_vec = normal_byte_codes;

#define BYTECODES			  goto *normal_byte_codes[*ip];

#define NEXT_BC				  goto *dispatch_vec[*ip]

#define LOAD				  BRANCH_REGISTER(prefetch); \
					  prefetch = dispatch_vec[*++ip];
#define PREFETCH			  goto *prefetch; }

/* The default definition does nothing, so it is not valid for us */
#undef DO_SET_EXCEPT_FLAG

#define DO_SET_EXCEPT_FLAG(x)						\
  dispatch_vec = x ? global_monitored_bytecodes : global_normal_bytecodes

#ifdef JUMP_LOOKAHEAD
  static void * const true_byte_codes[256] = {
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue,

    &&retTrue,                  /* return top    */

    &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,

    &&nop, &&nop, &&nop, &&nop, &&nop, &&nop, &&nop, &&nop,

    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,

    &&longJmp0, &&longJmp1, &&longJmp2, &&longJmp3,

    &&lookaheadLongJumpNotTaken, &&lookaheadLongJumpNotTaken,
    &&lookaheadLongJumpNotTaken, &&lookaheadLongJumpNotTaken,

    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
    &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue, &&lookaheadFailedTrue,
  };

  static void * const false_byte_codes[256] = {
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse,

    &&retFalse,                  /* return top    */

    &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,

    &&shJmp1, &&shJmp2, &&shJmp3, &&shJmp4,
    &&shJmp5, &&shJmp6, &&shJmp7, &&shJmp8,

    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,

    &&lookaheadLongJumpNotTaken, &&lookaheadLongJumpNotTaken,
    &&lookaheadLongJumpNotTaken, &&lookaheadLongJumpNotTaken,

    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&longJmp0, &&longJmp1, &&longJmp2, &&longJmp3,

    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
    &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse, &&lookaheadFailedFalse,
  };

#define DO_JUMP_LOOKAHEAD(x) {		\
  if (x) {				\
    POP_N_OOPS(1);			\
    goto *true_byte_codes[*ip];		\
  } else {				\
    POP_N_OOPS(1);			\
    goto *false_byte_codes[*ip];	\
  }					\
}
#endif /* JUMP_LOOKAHEAD */

#endif /* !PIPELINING */
#endif /* USE_GCC_DISPATCH */

/******************** COMMON DEFINITIONS *********************************/

#ifdef OPTIMIZE
# define INVALID(errMsg)
#else
# define INVALID(errMsg)		_gst_errorf(errMsg)
#endif

#define BRANCH_LABEL(lab)	lab:
#define START			LOCAL_COUNTER++; {
#define FALL(newLab)		goto newLab; }
#define END			NEXT_BC; }

/******************** DEFAULT DEFINITIONS ********************************/

#ifndef BYTE1
#define BYTE1		ip[1]
#define BYTE2		ip[2]
#define LOAD1		ip++
#define LOAD2		ip += 2
#define LOAD3		ip += 3
#define FLUSH		END
#endif

#define INCR1		ip++
#define INCR2		ip += 2
#define INCR3		ip += 3

#ifndef LOAD
#define LOAD		LOAD1
#define PREFETCH	END
#endif

#ifndef INTERPRETER
#define INTERPRETER
#endif

#ifndef CHECK_NEXT
#define CHECK_NEXT	SET_EXCEPT_FLAG(true)
#endif

#ifndef DO_JUMP_LOOKAHEAD
#define DO_JUMP_LOOKAHEAD(x) {		\
  SET_STACKTOP_BOOLEAN (x);		\
  NEXT_BC;				\
}
#endif

/******************** BODY OF THE INTERPRETER ***********************/

  IMPORT_REGS ();

  /* The first time through, evaluate the monitoring code in order to
     process the execution tracing flag.  */
  _gst_except_flag = true;

  _gst_register_oop (processOOP);
  in_interpreter = true;

  INTERPRETER
  {
    OOP returnedValue;

    MONITOR_CODE
    {
      EXPORT_REGS ();

      if (!disable_preemption)
	{
          _gst_disable_interrupts ();	/* block out everything! */
          if UNCOMMON (async_queue_index)
	    {
	      /* deal with any async signals        */
	      int i;
	      for (i = 0; i < async_queue_index; i++)
	        {
	          sync_signal (queued_async_signals[i].sem);
	          if (queued_async_signals[i].unregister)
		    _gst_unregister_oop (queued_async_signals[i].sem);
	        }

	      async_queue_index = 0;
	    }
          _gst_enable_interrupts ();
	}

      SET_EXCEPT_FLAG (false);
      if UNCOMMON (!IS_NIL (switch_to_process))
	{
	  change_process_context (switch_to_process);
	  IMPORT_REGS ();

	  if UNCOMMON (single_step_semaphore)
	    {
	      _gst_async_signal (single_step_semaphore);
	      single_step_semaphore = NULL;
	    }
	}

      else if UNCOMMON (time_to_preempt)
	ACTIVE_PROCESS_YIELD ();

      if (is_process_terminating (processOOP))
	{
	  gst_callin_process process = (gst_callin_process) OOP_TO_OBJ (processOOP);
          if (!IS_NIL (switch_to_process))
	    change_process_context (switch_to_process);

	  _gst_unregister_oop (processOOP);
	  in_interpreter = false;
	  return (process->returnedValue);
	}

      if UNCOMMON (_gst_abort_execution)
	{
	  OOP selectorOOP;
	  selectorOOP = _gst_intern_string ((char *) _gst_abort_execution);
	  _gst_abort_execution = NULL;
	  SEND_MESSAGE (selectorOOP, 0, false);
	}

      if UNCOMMON (_gst_execution_tracing)
	{
	  if (verbose_exec_tracing)
	    printf ("\t  --> %O\n", STACKTOP ());

	  printf (" %5ld:\t", (long) (ip - method_base));
	  _gst_print_bytecode_name (ip, ip - method_base, _gst_literals);
	  printf ("\n");
	  CHECK_NEXT;
	}

      if UNCOMMON (time_to_preempt)
	set_preemption_timer ();

      IMPORT_REGS ();
    }

    BYTECODES
    {
      BC (0, pushRecVar0)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (0));
        PREFETCH;

      BC (1, pushRecVar1)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (1));
	PREFETCH;

      BC (2, pushRecVar2)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (2));
	PREFETCH;

      BC (3, pushRecVar3)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (3));
	PREFETCH;

      BC (4, pushRecVar4)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (4));
	PREFETCH;

      BC (5, pushRecVar5)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (5));
        PREFETCH;

      BC (6, pushRecVar6)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (6));
        PREFETCH;

      BC (7, pushRecVar7)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (7));
        PREFETCH;

      BC (8, pushRecVar8)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (8));
        PREFETCH;

      BC (9, pushRecVar9)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (9));
        PREFETCH;

      BC (10, pushRecVar10)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (10));
        PREFETCH;

      BC (11, pushRecVar11)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (11));
        PREFETCH;

      BC (12, pushRecVar12)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (12));
        PREFETCH;

      BC (13, pushRecVar13)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (13));
        PREFETCH;

      BC (14, pushRecVar14)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (14));
        PREFETCH;

      BC (15, pushRecVar15)
	LOAD;
        PUSH_OOP (RECEIVER_VARIABLE (15));
        PREFETCH;


      BC (16, pushTemp0)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (0));
        PREFETCH;

      BC (17, pushTemp1)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (1));
        PREFETCH;

      BC (18, pushTemp2)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (2));
        PREFETCH;

      BC (19, pushTemp3)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (3));
        PREFETCH;

      BC (20, pushTemp4)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (4));
        PREFETCH;

      BC (21, pushTemp5)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (5));
        PREFETCH;

      BC (22, pushTemp6)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (6));
        PREFETCH;

      BC (23, pushTemp7)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (7));
        PREFETCH;

      BC (24, pushTemp8)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (8));
        PREFETCH;

      BC (25, pushTemp9)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (9));
        PREFETCH;

      BC (26, pushTemp10)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (10));
        PREFETCH;

      BC (27, pushTemp11)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (11));
        PREFETCH;

      BC (28, pushTemp12)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (12));
        PREFETCH;

      BC (29, pushTemp13)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (13));
        PREFETCH;

      BC (30, pushTemp14)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (14));
        PREFETCH;

      BC (31, pushTemp15)
	LOAD;
        PUSH_OOP (METHOD_TEMPORARY (15));
        PREFETCH;


      BC (32, pushLit0)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (0));
        PREFETCH;

      BC (33, pushLit1)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (1));
        PREFETCH;

      BC (34, pushLit2)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (2));
        PREFETCH;

      BC (35, pushLit3)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (3));
        PREFETCH;

      BC (36, pushLit4)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (4));
        PREFETCH;

      BC (37, pushLit5)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (5));
        PREFETCH;

      BC (38, pushLit6)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (6));
        PREFETCH;

      BC (39, pushLit7)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (7));
        PREFETCH;

      BC (40, pushLit8)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (8));
        PREFETCH;

      BC (41, pushLit9)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (9));
        PREFETCH;

      BC (42, pushLit10)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (10));
        PREFETCH;

      BC (43, pushLit11)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (11));
        PREFETCH;

      BC (44, pushLit12)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (12));
        PREFETCH;

      BC (45, pushLit13)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (13));
        PREFETCH;

      BC (46, pushLit14)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (14));
        PREFETCH;

      BC (47, pushLit15)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (15));
        PREFETCH;

      BC (48, pushLit16)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (16));
        PREFETCH;

      BC (49, pushLit17)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (17));
        PREFETCH;

      BC (50, pushLit18)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (18));
        PREFETCH;

      BC (51, pushLit19)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (19));
        PREFETCH;

      BC (52, pushLit20)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (20));
        PREFETCH;

      BC (53, pushLit21)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (21));
        PREFETCH;

      BC (54, pushLit22)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (22));
        PREFETCH;

      BC (55, pushLit23)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (23));
        PREFETCH;

      BC (56, pushLit24)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (24));
        PREFETCH;

      BC (57, pushLit25)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (25));
        PREFETCH;

      BC (58, pushLit26)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (26));
        PREFETCH;

      BC (59, pushLit27)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (27));
        PREFETCH;

      BC (60, pushLit28)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (28));
        PREFETCH;

      BC (61, pushLit29)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (29));
        PREFETCH;

      BC (62, pushLit30)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (30));
        PREFETCH;

      BC (63, pushLit31)
	LOAD;
        PUSH_OOP (METHOD_LITERAL (31));
        PREFETCH;


      BC (64, pushVar0)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (0));
        PREFETCH;

      BC (65, pushVar1)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (1));
        PREFETCH;

      BC (66, pushVar2)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (2));
        PREFETCH;

      BC (67, pushVar3)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (3));
        PREFETCH;

      BC (68, pushVar4)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (4));
        PREFETCH;

      BC (69, pushVar5)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (5));
        PREFETCH;

      BC (70, pushVar6)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (6));
        PREFETCH;

      BC (71, pushVar7)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (7));
        PREFETCH;

      BC (72, pushVar8)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (8));
        PREFETCH;

      BC (73, pushVar9)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (9));
        PREFETCH;

      BC (74, pushVar10)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (10));
        PREFETCH;

      BC (75, pushVar11)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (11));
        PREFETCH;

      BC (76, pushVar12)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (12));
        PREFETCH;

      BC (77, pushVar13)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (13));
        PREFETCH;

      BC (78, pushVar14)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (14));
        PREFETCH;

      BC (79, pushVar15)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (15));
        PREFETCH;

      BC (80, pushVar16)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (16));
        PREFETCH;

      BC (81, pushVar17)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (17));
        PREFETCH;

      BC (82, pushVar18)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (18));
        PREFETCH;

      BC (83, pushVar19)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (19));
        PREFETCH;

      BC (84, pushVar20)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (20));
        PREFETCH;

      BC (85, pushVar21)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (21));
        PREFETCH;

      BC (86, pushVar22)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (22));
        PREFETCH;

      BC (87, pushVar23)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (23));
        PREFETCH;

      BC (88, pushVar24)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (24));
        PREFETCH;

      BC (89, pushVar25)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (25));
        PREFETCH;

      BC (90, pushVar26)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (26));
        PREFETCH;

      BC (91, pushVar27)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (27));
        PREFETCH;

      BC (92, pushVar28)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (28));
        PREFETCH;

      BC (93, pushVar29)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (29));
        PREFETCH;

      BC (94, pushVar30)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (30));
        PREFETCH;

      BC (95, pushVar31)
	LOAD;
        PUSH_OOP (METHOD_VARIABLE (31));
        PREFETCH;


      BC (96, stRecVar0)
	LOAD;
        STORE_RECEIVER_VARIABLE (0, POP_OOP ());
        PREFETCH;

      BC (97, stRecVar1)
	LOAD;
        STORE_RECEIVER_VARIABLE (1, POP_OOP ());
        PREFETCH;

      BC (98, stRecVar2)
	LOAD;
        STORE_RECEIVER_VARIABLE (2, POP_OOP ());
        PREFETCH;

      BC (99, stRecVar3)
	LOAD;
        STORE_RECEIVER_VARIABLE (3, POP_OOP ());
        PREFETCH;

      BC (100, stRecVar4)
	LOAD;
        STORE_RECEIVER_VARIABLE (4, POP_OOP ());
        PREFETCH;

      BC (101, stRecVar5)
	LOAD;
        STORE_RECEIVER_VARIABLE (5, POP_OOP ());
        PREFETCH;

      BC (102, stRecVar6)
	LOAD;
        STORE_RECEIVER_VARIABLE (6, POP_OOP ());
        PREFETCH;

      BC (103, stRecVar7)
	LOAD;
        STORE_RECEIVER_VARIABLE (7, POP_OOP ());
        PREFETCH;


      BC (104, stTemp0)
	LOAD;
        STORE_METHOD_TEMPORARY (0, POP_OOP ());
        PREFETCH;

      BC (105, stTemp1)
	LOAD;
        STORE_METHOD_TEMPORARY (1, POP_OOP ());
        PREFETCH;

      BC (106, stTemp2)
	LOAD;
        STORE_METHOD_TEMPORARY (2, POP_OOP ());
        PREFETCH;

      BC (107, stTemp3)
	LOAD;
        STORE_METHOD_TEMPORARY (3, POP_OOP ());
        PREFETCH;

      BC (108, stTemp4)
	LOAD;
        STORE_METHOD_TEMPORARY (4, POP_OOP ());
        PREFETCH;

      BC (109, stTemp5)
	LOAD;
        STORE_METHOD_TEMPORARY (5, POP_OOP ());
        PREFETCH;

      BC (110, stTemp6)
	LOAD;
        STORE_METHOD_TEMPORARY (6, POP_OOP ());
        PREFETCH;

      BC (111, stTemp7)
	LOAD;
        STORE_METHOD_TEMPORARY (7, POP_OOP ());
        PREFETCH;


      BC (112, pushSelf)
	LOAD;
        UNCHECKED_PUSH_OOP (_gst_self);
        PREFETCH;

      BC (113, pushTrue)
	LOAD;
        UNCHECKED_PUSH_OOP (_gst_true_oop);
        PREFETCH;

      BC (114, pushFalse)
	LOAD;
        UNCHECKED_PUSH_OOP (_gst_false_oop);
        PREFETCH;

      BC (115, pushNil)
	LOAD;
        UNCHECKED_PUSH_OOP (_gst_nil_oop);
        PREFETCH;

      BC (116, pushSig8)
	REGISTER (3, int num);
        num = BYTE1;
	LOAD2;
        PUSH_INT ((signed char) num);
        END;

      BC (117, pushUnsig8)
	REGISTER (3, int num);
        num = BYTE1;
	LOAD2;
        PUSH_INT ((unsigned char) num);
        END;

      BC (118, pushZero)
	LOAD;
        PUSH_INT (0);
        PREFETCH;

      BC (119, pushOne)
	LOAD;
        PUSH_INT (1);
        PREFETCH;


      BC (120, retSelf)
	returnedValue = _gst_self;
        FALL (doRetStackTop);

      BC (121, retTrue)
	returnedValue = _gst_true_oop;
        FALL (doRetStackTop);
 
      BC (122, retFalse)
	returnedValue = _gst_false_oop;
        FALL (doRetStackTop);

      BC (123, retNil)
	returnedValue = _gst_nil_oop;
        FALL (doRetStackTop);


      BC (124, explicitRet)	/* return stack top from method */
	returnedValue = STACKTOP ();
        EXPORT_REGS ();
	if UNCOMMON (!unwind_method ())
	  {
	    SEND_MESSAGE (_gst_bad_return_error_symbol, 0, false);
	    IMPORT_REGS ();
	  }
	else
	  {
	    IMPORT_REGS ();
	    SET_STACKTOP (returnedValue);
	  }
	FLUSH;

      BC (125, retStackTop)
	returnedValue = STACKTOP ();
      BRANCH_LABEL (doRetStackTop)
	EXPORT_REGS ();
        unwind_context ();
	IMPORT_REGS ();
	SET_STACKTOP (returnedValue);
	FLUSH;

      /* 126, not in the blue book, is used for big literals access as follows:
         01111110 yyxxxxxx xxxxxxxx
         on method literal xxx do
         yy = 00 -> push constant
         yy = 01 -> push variable
         yy = 10 -> store variable
         yy = 11 -> pop/store variable
       */

      BC (126, bigLiteralOp)
	REGISTER (3, unsigned int num);
        num = BYTE1;
	num <<= 8;
	num |= BYTE2;

	switch (num >> 14)
	  {
	  case 0:
	    LOAD3;
	    PUSH_OOP (METHOD_LITERAL (num & 16383));
	    break;
	  case 1:
	    LOAD3;
	    PUSH_OOP (METHOD_VARIABLE (num & 16383));
	    break;
	  case 2:
	    LOAD3;
	    STORE_METHOD_VARIABLE (num & 16383, POP_OOP ());
	    break;
	  case 3:
	  default:
	    LOAD3;
	    STORE_METHOD_VARIABLE (num & 16383, STACKTOP ());
	    break;
	  }
	END;

      BC (127, lineNo)
	LOAD3;
	END;

      BC (128, pushIndexedVal)
	REGISTER (3, gst_uchar ival2);
        ival2 = BYTE1;
	switch (ival2 >> 6)
	  {
	  case 0:
	    LOAD2;
	    PUSH_OOP (RECEIVER_VARIABLE (ival2 & 63));
	    break;
	  case 1:
	    LOAD2;
	    PUSH_OOP (METHOD_TEMPORARY (ival2 & 63));
	    break;
	  case 2:
	    LOAD2;
	    PUSH_OOP (METHOD_LITERAL (ival2 & 63));
	    break;
	  case 3:
	  default:
	    LOAD2;
	    PUSH_OOP (METHOD_VARIABLE (ival2 & 63));
	    break;
	  }
	END;

      BC (129, storeStackTop)
	REGISTER (3, gst_uchar ival2);
        ival2 = BYTE1;
	switch (ival2 >> 6)
	  {
	  case 0:
	    LOAD2;
	    STORE_RECEIVER_VARIABLE (ival2 & 63, STACKTOP ());
	    break;
	  case 1:
	    LOAD2;
	    STORE_METHOD_TEMPORARY (ival2 & 63, STACKTOP ());
	    break;
	  case 2:
	    LOAD2;
	    INVALID ("Attempt to store into a method constant");
	    break;
	  case 3:
	  default:
	    LOAD2;
	    STORE_METHOD_VARIABLE (ival2 & 63, STACKTOP ());
	  }
	END;

      BC (130, popAndStoreTop)
	REGISTER (3, gst_uchar ival2);
        ival2 = BYTE1;
	switch (ival2 >> 6)
	  {
	  case 0:
	    LOAD2;
	    STORE_RECEIVER_VARIABLE (ival2 & 63, POP_OOP ());
	    break;
	  case 1:
	    LOAD2;
	    STORE_METHOD_TEMPORARY (ival2 & 63, POP_OOP ());
	    break;
	  case 2:
	    LOAD2;
	    INVALID ("Attempt to store into a method constant");
	    break;
	  case 3:
	  default:
	    LOAD2;
	    STORE_METHOD_VARIABLE (ival2 & 63, POP_OOP ());
	  }
	END;

      BC (131, sendShort)	/* send selector y (xxxyyyyy), x args */
	REGISTER (3, gst_uchar ival2);
        ival2 = BYTE1;
	INCR2;
	EXPORT_REGS ();
	SEND_MESSAGE (METHOD_LITERAL (ival2 & 31), ival2 >> 5, false);
	IMPORT_REGS ();
	FLUSH;

      BC (132, sendLong)	/* send selector y (yysxxxxx,yyyyyyyy) x args */
	REGISTER (3, gst_uchar ival2);  	/*     \___to super if s=1    */
        int ival3;
	ival2 = BYTE1;		/* the number of args */
	ival3 = BYTE2;		/* the selector */
	ival3 += (ival2 & 192) << 2;
	INCR3;
	EXPORT_REGS ();
	if UNCOMMON (ival2 & 32)
	  SEND_TO_SUPER (METHOD_LITERAL (ival3), ival2 & 31, true);
	else
	  SEND_MESSAGE (METHOD_LITERAL (ival3), ival2 & 31, false);

	IMPORT_REGS ();
	FLUSH;

      BC (133, supSendShort)	/* send super selector y (xxxyyyyy), x args */
	REGISTER (3, gst_uchar ival2);
        ival2 = BYTE1;
	INCR2;
	EXPORT_REGS ();
	SEND_TO_SUPER (METHOD_LITERAL (ival2 & 31), ival2 >> 5, true);
	IMPORT_REGS ();
	FLUSH;

      BC (134, bigInstanceOp)
	REGISTER (3, unsigned int num);
        num = BYTE1;
	num <<= 8;
	num |= BYTE2;
	
	switch (num >> 14)
	  {
	  case 0:
	    {
	      OOP value;
	      LOAD3;
	      value = POP_OOP ();
	      STORE_INSTANCE_VARIABLE (STACKTOP (), num & 16383, value);
	      break;
	    }
	  case 1:
	    LOAD3;
	    PUSH_OOP (RECEIVER_VARIABLE (num & 16383));
	    break;
	  case 2:
	    LOAD3;
	    STORE_RECEIVER_VARIABLE (num & 16383, POP_OOP ());
	    break;
	  case 3:
	  default:
	    LOAD3;
	    STORE_RECEIVER_VARIABLE (num & 16383, STACKTOP ());
	    break;
	  }
	END;
	
      BC (135, popStack)
	LOAD;
        POP_N_OOPS (1);
        PREFETCH;

      BC (136, dupStack)
	OOP tempOOP;
        LOAD;
	tempOOP = STACKTOP ();
	PUSH_OOP (tempOOP);
        PREFETCH;

      BC (137, pushThisContext)	/* push active context */
	LOAD;
        empty_context_stack ();
	PUSH_OOP (_gst_this_context_oop);
        PREFETCH;

      BC (138, outerTempOp)
	unsigned int numScopes;
        unsigned int varIndex;
	OOP contextOOP;
	gst_block_context context;

	varIndex = BYTE1;
	numScopes = BYTE2;
	
	context = (gst_block_context) OOP_TO_OBJ (_gst_this_context_oop);
	do
	  {
	    contextOOP = context->outerContext;
	    context = (gst_block_context) OOP_TO_OBJ (contextOOP);
	  }
	while (--numScopes);
	
#ifndef OPTIMIZE
	if (context != NULL)
	  {			/* we're on the right one */
#endif
	    switch (varIndex >> 6)
	      {
	      case 0:
		LOAD3;
		INVALID ("Invalid bytecode 138/0");
		break;
	      case 1:
		LOAD3;
		PUSH_OOP (context->contextStack[varIndex & 63]);
		break;
	      case 2:
		LOAD3;
		context->contextStack[varIndex & 63] = POP_OOP ();
		break;
	      case 3:
	      default:
		LOAD3;
		context->contextStack[varIndex & 63] = STACKTOP ();
		break;
#ifndef OPTIMIZE
	      }
#endif
	}
      END;

      /* :-( Sigh ... have to sacrifice 139 for nop... )-: */
      BC (139, nop)
	LOAD;
        PREFETCH;

      BC (140, topSelf)
	LOAD;
        UNCHECKED_SET_TOP (_gst_self);
        PREFETCH;

      BC (141, topOne)
	LOAD;
        UNCHECKED_SET_TOP (FROM_INT (1));
        PREFETCH;

      BC (142, topIndexedVal)
	REGISTER (3, gst_uchar ival2);
        ival2 = BYTE1;
	switch (ival2 >> 6)
	  {
	  case 0:
	    LOAD2;
	    SET_STACKTOP (RECEIVER_VARIABLE (ival2 & 63));
	    break;
	  case 1:
	    LOAD2;
	    SET_STACKTOP (METHOD_TEMPORARY (ival2 & 63));
	    break;
	  case 2:
	    LOAD2;
	    SET_STACKTOP (METHOD_LITERAL (ival2 & 63));
	    break;
	  case 3:
	  default:
	    LOAD2;
	    SET_STACKTOP (METHOD_VARIABLE (ival2 & 63));
	    break;
	  }
	END;

      BC (143, endExecution)
	/* This fixes a very rare condition, but it can happen: GC can
	   happen because send_message_internal allocates a context
	   while the interpreter is checking last_primitive to fill
	   the special cache for #at:, #at:put: and #size.  If there
	   are finalizable objects, the finalization callins might
	   change last_primitive to a bogus value.  To fix the
	   problem, we invalidate last_primitive every time the
	   interpreter exits. */
	last_primitive = 0;

	{
	  OOP activeProcessOOP = get_scheduled_process ();
	  gst_callin_process process = (gst_callin_process) OOP_TO_OBJ (activeProcessOOP);
	  if (IS_NIL (activeProcessOOP))
	    abort ();

	  process->returnedValue = POP_OOP ();
	  _gst_terminate_process (activeProcessOOP);
	  if (processOOP == activeProcessOOP)
	    SET_EXCEPT_FLAG (true);
	}
        FLUSH;


#define JMP_FAIL do { \
    UNPOP(1); \
    EXPORT_REGS(); \
    SEND_MESSAGE(_gst_intern_string("mustBeBoolean"), 0, false); \
    IMPORT_REGS(); \
  } while(0)

#define JMP_LONG_SIGNED(highDelta) do {		\
  REGISTER(3, gst_uchar		ival2);		\
  ival2 = BYTE1;				\
  ip += (long)((highDelta) << 8) + 2 + ival2;	\
} while(0)

      /* Forward jump by 1+1 to 8+1 (+1 is the bytecode itself) */
      BC (144, shJmp1)
	LOAD2;
        END;

      BC (145, shJmp2)
	LOAD3;
        END;

      BC (146, shJmp3)
	ip += 4;
        FLUSH;

      BC (147, shJmp4)
	ip += 5;
        FLUSH;

      BC (148, shJmp5)
	ip += 6;
        FLUSH;

      BC (149, shJmp6)
	ip += 7;
        FLUSH;

      BC (150, shJmp7)
	ip += 8;
        FLUSH;

      BC (151, shJmp8)
	ip += 9;
        FLUSH;

      BC (152, shJmp1False)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_false_oop)
	  LOAD2;		/* jump forward if false */
	else if UNCOMMON (tempOOP != _gst_true_oop)
	  {
	    LOAD2;
	    JMP_FAIL;
	  }
	else
	  LOAD1;

	END;

      BC (153, shJmp2False)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_false_oop)
	  LOAD3;		/* jump forward if false */
	else if UNCOMMON (tempOOP != _gst_true_oop)
	  {
	    LOAD3;
	    JMP_FAIL;
	  }
	else
	  LOAD1;
	END;

      BC (154, shJmp3False)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_false_oop)
	  ip += 4;		/* jump forward if false */
	else if UNCOMMON (tempOOP != _gst_true_oop)
	  {
	    ip += 4;
	    JMP_FAIL;
	  }
	else
	  {
	    LOAD1;
	    NEXT_BC;
	  }
      FLUSH;

      BC (155, shJmp4False)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_false_oop)
	  ip += 5;		/* jump forward if false */
	else if UNCOMMON (tempOOP != _gst_true_oop)
	  {
	    ip += 5;
	    JMP_FAIL;
	  }
	else
	  {
	    LOAD1;
	    NEXT_BC;
	  }
	FLUSH;

      BC (156, shJmp5False)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_false_oop)
	  ip += 6;		/* jump forward if false */
	else if UNCOMMON (tempOOP != _gst_true_oop)
	  {
	    ip += 6;
	    JMP_FAIL;
	  }
	else
	  {
	    LOAD1;
	    NEXT_BC;
	  }
	FLUSH;

      BC (157, shJmp6False)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_false_oop)
	  ip += 7;		/* jump forward if false */
	else if UNCOMMON (tempOOP != _gst_true_oop)
	  {
	    ip += 7;
	    JMP_FAIL;
	  }
	else
	  {
	    LOAD1;
	    NEXT_BC;
	  }
	FLUSH;

      BC (158, shJmp7False)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_false_oop)
	  ip += 8;		/* jump forward if false */
	else if UNCOMMON (tempOOP != _gst_true_oop)
	  {
	    ip += 8;
	    JMP_FAIL;
	  }
	else
	  {
	    LOAD1;
	    NEXT_BC;
	  }
      FLUSH;

      BC (159, shJmp8False)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_false_oop)
	  ip += 9;		/* jump forward if false */
	else if UNCOMMON (tempOOP != _gst_true_oop)
	  {
	    ip += 9;
	    JMP_FAIL;
	  }
	else
	  {
	    LOAD1;
	    NEXT_BC;
	  }
	FLUSH;


      BC (160, longJmpNeg4)
	JMP_LONG_SIGNED (-4);
        FLUSH;

      BC (161, longJmpNeg3)
	JMP_LONG_SIGNED (-3);
        FLUSH;

      BC (162, longJmpNeg2)
	JMP_LONG_SIGNED (-2);
        FLUSH;

      BC (163, longJmpNeg1)
	JMP_LONG_SIGNED (-1);
        FLUSH;

      BC (164, longJmp0)
	JMP_LONG_SIGNED (0);
        FLUSH;

      BC (165, longJmp1)
	JMP_LONG_SIGNED (1);
        FLUSH;

      BC (166, longJmp2)
	JMP_LONG_SIGNED (2);
        FLUSH;

      BC (167, longJmp3)
	JMP_LONG_SIGNED (3);
        FLUSH;


      BC (168, popJmpTrue0)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_true_oop)
	  JMP_LONG_SIGNED (0);
	else if UNCOMMON (tempOOP != _gst_false_oop)
	  {
	    JMP_LONG_SIGNED (0);
	    JMP_FAIL;
	  }
	else
	  {
	    LOAD2;
	    NEXT_BC;
	  }
	FLUSH;

      BC (169, popJmpTrue1)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_true_oop)
	  JMP_LONG_SIGNED (1);
	else if UNCOMMON (tempOOP != _gst_false_oop)
	  {
	    JMP_LONG_SIGNED (1);
	    JMP_FAIL;
	  }
	else
	  {
	    LOAD2;
	    NEXT_BC;
	  }
	FLUSH;

      BC (170, popJmpTrue2)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_true_oop)
	  JMP_LONG_SIGNED (2);
	else if UNCOMMON (tempOOP != _gst_false_oop)
	  {
	    JMP_LONG_SIGNED (2);
	    JMP_FAIL;
	  }
	else
	  {
	    LOAD2;
	    NEXT_BC;
	  }
	FLUSH;

      BC (171, popJmpTrue3)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_true_oop)
	  JMP_LONG_SIGNED (3);
	else if UNCOMMON (tempOOP != _gst_false_oop)
	  {
	    JMP_LONG_SIGNED (3);
	    JMP_FAIL;
	  }
	else
	  NEXT_BC;
	FLUSH;


      BC (172, popJmpFalse0)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_false_oop)
	  JMP_LONG_SIGNED (0);
	else if UNCOMMON (tempOOP != _gst_true_oop)
	  {
	    JMP_LONG_SIGNED (0);
	    JMP_FAIL;
	  }
	else
	  {
	    LOAD2;
	    NEXT_BC;
	  }
	FLUSH;

      BC (173, popJmpFalse1)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_false_oop)
	  JMP_LONG_SIGNED (1);
	else if UNCOMMON (tempOOP != _gst_true_oop)
	  {
	    JMP_LONG_SIGNED (1);
	    JMP_FAIL;
	  }
	else
	  {
	    LOAD2;
	    NEXT_BC;
	  }
	FLUSH;

      BC (174, popJmpFalse2)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_false_oop)
	  JMP_LONG_SIGNED (2);
	else if UNCOMMON (tempOOP != _gst_true_oop)
	  {
	    JMP_LONG_SIGNED (2);
	    JMP_FAIL;
	  }
	else
	  {
	    LOAD2;
	    NEXT_BC;
	  }
	FLUSH;

      BC (175, popJmpFalse3)
	REGISTER (3, OOP tempOOP);
        if UNCOMMON ((tempOOP = POP_OOP ()) == _gst_false_oop)
	  JMP_LONG_SIGNED (3);
	else if UNCOMMON (tempOOP != _gst_true_oop)
	  {
	    JMP_LONG_SIGNED (3);
	    JMP_FAIL;
	  }
	else
	  {
	    LOAD2;
	    NEXT_BC;
	  }
	FLUSH;

#undef JMP_FAIL
#undef JMP_LONG_SIGNED

#ifndef OPEN_CODE_MATH

#define RAW_INT_OP(op, noOverflow)
#define RAW_FLOATD_OP(op)
#define INTERP_BASIC_OP(op, fop)
#define RAW_BOOL_OP(operator)
#define RAW_BOOL_FLOATD_OP(operator)
#define INTERP_BASIC_BOOL(operator)

#else

#define RAW_INT_OP(op, noOverflow)			\
{							\
  long iarg1, iarg2;					\
  iarg1 = TO_INT(tempOOP);				\
  iarg2 = TO_INT(POP_OOP());				\
  op;							\
  if(noOverflow || COMMON (!INT_OVERFLOW(iarg1))) {	\
    SET_STACKTOP_INT(iarg1);				\
    NEXT_BC;						\
  } else {						\
    UNPOP(1);						\
  }							\
}

#define RAW_FLOATD_OP(op)				\
{							\
  OOP	oop2,oopResult;					\
  double farg1, farg2;					\
  oop2 = STACKTOP();					\
  if COMMON (IS_CLASS(oop2, _gst_floatd_class)) {	\
    farg1 = FLOATD_OOP_VALUE(tempOOP);			\
    farg2 = FLOATD_OOP_VALUE(oop2);			\
    POP_N_OOPS(1);					\
    EXPORT_REGS();					\
    oopResult = floatd_new(op);				\
    IMPORT_REGS();					\
    SET_STACKTOP(oopResult);				\
    NEXT_BC;						\
  }							\
}


#define INTERP_BASIC_OP(op, fop) {				\
  REGISTER(3, OOP	tempOOP);				\
  tempOOP = STACK_AT(1);					\
  if COMMON (ARE_INTS(tempOOP, STACKTOP())) {			\
    RAW_INT_OP(op, false);					\
  } else if COMMON (IS_CLASS(tempOOP, _gst_floatd_class)) {	\
    RAW_FLOATD_OP(fop);						\
  }								\
}

#define RAW_BOOL_OP(operator)				\
  DO_JUMP_LOOKAHEAD(	((long)tempOOP) operator ((long)POP_OOP()) );

#define RAW_BOOL_FLOATD_OP(operator)			\
{							\
  OOP	oop2;						\
  oop2 = STACKTOP();					\
  if COMMON (IS_CLASS(oop2, _gst_floatd_class)) {	\
    double farg1, farg2;				\
    farg1 = FLOATD_OOP_VALUE(tempOOP);			\
    farg2 = FLOATD_OOP_VALUE(oop2);			\
    POP_N_OOPS(1);					\
    DO_JUMP_LOOKAHEAD(farg1 operator farg2); 		\
  }							\
}

#define INTERP_BASIC_BOOL(operator) {				\
  REGISTER(3, OOP	tempOOP);				\
  tempOOP = STACK_AT(1);					\
  if COMMON (ARE_INTS(tempOOP, STACKTOP())) {			\
    RAW_BOOL_OP(operator);					\
  } else if COMMON (IS_CLASS(tempOOP, _gst_floatd_class)) {	\
    RAW_BOOL_FLOATD_OP(operator);				\
  }								\
}
#endif

#define MAYBE_PRIMITIVE(cond, symbol, func, id, numArgs)	\
  if COMMON ((cond)) {						\
    EXPORT_REGS();						\
    if COMMON (!func ((id), (numArgs))) {			\
      IMPORT_REGS();						\
      NEXT_BC;							\
    }								\
  } else {							\
    EXPORT_REGS();						\
  }								\
  SEND_MESSAGE((symbol), (numArgs), false);			\
  IMPORT_REGS();


      BC (176, sendPlus)
	LOAD1;
	INTERP_BASIC_OP (iarg1 += iarg2, farg1 + farg2);
	EXPORT_REGS ();
	SEND_MESSAGE (_gst_plus_symbol, 1, false);
	IMPORT_REGS ();
	FLUSH;

      BC (177, sendMinus)
	LOAD1;
	INTERP_BASIC_OP (iarg1 -= iarg2, farg1 - farg2);
	EXPORT_REGS ();
	SEND_MESSAGE (_gst_minus_symbol, 1, false);
	IMPORT_REGS ();
	FLUSH;

      BC (178, sendLess)
	LOAD1;
	INTERP_BASIC_BOOL (<);
	EXPORT_REGS ();
	SEND_MESSAGE (_gst_less_than_symbol, 1, false);
	IMPORT_REGS ();
	FLUSH;

      BC (179, sendGreater)
	LOAD1;
	INTERP_BASIC_BOOL (>);
	EXPORT_REGS ();
	SEND_MESSAGE (_gst_greater_than_symbol, 1, false);
	IMPORT_REGS ();
	FLUSH;

      BC (180, sendLessEq)
	LOAD1;
	INTERP_BASIC_BOOL (<=);
	EXPORT_REGS ();
	SEND_MESSAGE (_gst_less_equal_symbol, 1, false);
	IMPORT_REGS ();
	FLUSH;

      BC (181, sendGreaterEq)
	LOAD1;
	INTERP_BASIC_BOOL (>=);
	EXPORT_REGS ();
	SEND_MESSAGE (_gst_greater_equal_symbol, 1, false);
	IMPORT_REGS ();
	FLUSH;

      BC (182, sendEqual)
	LOAD1;
	INTERP_BASIC_BOOL ( ==);
	EXPORT_REGS ();
	SEND_MESSAGE (_gst_equal_symbol, 1, false);
	IMPORT_REGS ();
	FLUSH;

      BC (183, sendNotEqual)
	LOAD1;
	INTERP_BASIC_BOOL (!=);
	EXPORT_REGS ();
	SEND_MESSAGE (_gst_not_equal_symbol, 1, false);
	IMPORT_REGS ();
	FLUSH;

      BC (184, sendTimes)
	LOAD1;
	INTERP_BASIC_OP (iarg1 = mul_with_check (iarg1, iarg2), farg1 * farg2);
	EXPORT_REGS ();
	SEND_MESSAGE (_gst_times_symbol, 1, false);
	IMPORT_REGS ();
	FLUSH;

      BC (185, sendDivide)
	LOAD1;
	MAYBE_PRIMITIVE (IS_INT (STACK_AT (1)) && IS_INT (STACKTOP ()),
			 _gst_divide_symbol, VMpr_SmallInteger_divide, 10, 1);
	FLUSH;

      BC (186, sendRemainder)
	LOAD1;
	MAYBE_PRIMITIVE (IS_INT (STACK_AT (1)) && IS_INT (STACKTOP ()),
			 _gst_remainder_symbol, VMpr_SmallInteger_modulo, 11, 1);
	FLUSH;

      BC (187, sendAtSign)
	INCR1;
	EXPORT_REGS ();
	SEND_MESSAGE (_gst_at_sign_symbol, 1, false);
	IMPORT_REGS ();
	FLUSH;

      BC (188, sendBitShift)
	REGISTER (3, OOP tempOOP);
	LOAD1;
	tempOOP = STACK_AT (1);
	if COMMON (IS_INT (tempOOP))
          {
	    long iarg1, iarg2;
	    if COMMON (IS_INT (STACKTOP ()))
	      {
		iarg1 = TO_INT (tempOOP);
		iarg2 = TO_INT (POP_OOP ());
		if (iarg2 < 0)
		  {
		    SET_STACKTOP_INT (iarg1 >> -iarg2);
		    NEXT_BC;
		  }
		if COMMON (iarg2 < ST_INT_SIZE)
		  {
		    long result = iarg1 << iarg2;
		    if COMMON ((result >> iarg2) == iarg1
			&& !INT_OVERFLOW(result))
		      {
			SET_STACKTOP_INT (result);
			NEXT_BC;
		      }
		  }
	      }
	    UNPOP (1);
	  }
        EXPORT_REGS ();
        SEND_MESSAGE (_gst_bit_shift_colon_symbol, 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (189, sendIntDivide)
        LOAD1;
	MAYBE_PRIMITIVE (IS_INT (STACK_AT (1)) && IS_INT (STACKTOP ()),
			 _gst_integer_divide_symbol,
			 VMpr_SmallInteger_intDiv, 12, 1);
	FLUSH;

      BC (190, sendBitAnd)
	REGISTER (3, OOP tempOOP);
        LOAD1;
	tempOOP = STACK_AT (1);
	if COMMON (ARE_INTS (tempOOP, STACKTOP ()))
	  RAW_INT_OP (iarg1 &= iarg2, true);

        EXPORT_REGS ();
	SEND_MESSAGE (_gst_bit_and_colon_symbol, 1, false);
	IMPORT_REGS ();
	FLUSH;

      BC (191, sendBitOr)
	REGISTER (3, OOP tempOOP);
        LOAD1;
        tempOOP = STACK_AT (1);
	if COMMON (ARE_INTS (tempOOP, STACKTOP ()))
	  RAW_INT_OP (iarg1 |= iarg2, true);

	EXPORT_REGS ();
	SEND_MESSAGE (_gst_bit_or_colon_symbol, 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (192, sendAt)
	REGISTER (3, OOP tempOOP);
	LOAD1;
	tempOOP = STACK_AT (1);
	if (COMMON (IS_OOP (tempOOP))
	    && at_cache_class == (tempOOP = OOP_CLASS (tempOOP)))
          {
	    EXPORT_REGS ();
	    if (!_gst_execute_primitive_operation (at_cache_prim, 1))
	      {
		IMPORT_REGS ();
		NEXT_BC;
	      }
	  }
        else
	  EXPORT_REGS ();

        /* Not the same class that is in the cache, or the primitive failed --
	   send the message, and modify the cache if the send is resolved to
	   a primitive. */
        last_primitive = 0;
        SEND_MESSAGE (_gst_at_colon_symbol, 1, false);
        IMPORT_REGS ();
        if COMMON (last_primitive)
	  {
	    at_cache_prim = last_primitive;
	    at_cache_class = tempOOP;
	    NEXT_BC;
	  }
        FLUSH;

      BC (193, sendAtPut)
	REGISTER (3, OOP tempOOP);
        LOAD1;
        tempOOP = STACK_AT (2);
	if (COMMON (IS_OOP (tempOOP))
	    && at_put_cache_class == (tempOOP = OOP_CLASS (tempOOP)))
	  {
	    EXPORT_REGS ();
	    if (!_gst_execute_primitive_operation (at_put_cache_prim, 2))
	      {
	        IMPORT_REGS ();
	        NEXT_BC;
	      }
	  }
        else
	  EXPORT_REGS ();

        /* Not the same class that is in the cache, or the primitive failed --
           send the message, and modify the cache if the send is resolved to
           a primitive. */
        last_primitive = 0;
        SEND_MESSAGE (_gst_at_colon_put_colon_symbol, 2, false);
        IMPORT_REGS ();
        if COMMON (last_primitive)
	  {
	    at_put_cache_prim = last_primitive;
	    at_put_cache_class = tempOOP;
	    NEXT_BC;
	  }
        FLUSH;
      
      BC (194, sendSize)
	REGISTER (3, OOP tempOOP);
        LOAD1;
        tempOOP = STACKTOP ();
	if (COMMON (IS_OOP (tempOOP))
	    && size_cache_class == (tempOOP = OOP_CLASS (tempOOP)))
	  {
	    EXPORT_REGS ();
	    if (!_gst_execute_primitive_operation (size_cache_prim, 0))
	      {
	        IMPORT_REGS ();
	        NEXT_BC;
	      }
	  }
        else
	  EXPORT_REGS ();

        /* Not the same class that is in the cache, or the primitive failed --
           send the message, and modify the cache if the send is resolved to
           a primitive. */
        last_primitive = 0;
        SEND_MESSAGE (_gst_size_symbol, 0, false);
        IMPORT_REGS ();
        if COMMON (last_primitive)
	  {
	    size_cache_prim = last_primitive;
	    size_cache_class = tempOOP;
	    NEXT_BC;
	  }
        FLUSH;

      BC (195, sendNext)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (_gst_next_symbol, 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (196, sendNextPut)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (_gst_next_put_colon_symbol, 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (197, sendAtEnd)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (_gst_at_end_symbol, 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (198, sendSameObject)
	REGISTER (3, OOP tempOOP);
        OOP oop1;
        LOAD1;
        tempOOP = POP_OOP ();
        oop1 = STACKTOP ();
        DO_JUMP_LOOKAHEAD (oop1 == tempOOP);
        END;

      BC (199, sendClass)
	REGISTER (3, OOP tempOOP);
        LOAD1;
        tempOOP = STACKTOP ();
        SET_STACKTOP (IS_INT (tempOOP) ? _gst_small_integer_class :
		      OOP_CLASS (tempOOP));
        END;

      BC (200, sendBlockCopy)
	REGISTER (3, OOP tempOOP);
        LOAD1;
        MAYBE_PRIMITIVE (IS_OOP (tempOOP = STACK_AT (1))
		         && OOP_CLASS (tempOOP) == _gst_block_closure_class,
		         _gst_block_copy_colon_symbol, 
			 VMpr_BlockClosure_blockCopy, 80, 1);
        FLUSH;

      BC (201, sendValue)
	REGISTER (3, OOP tempOOP);
        INCR1;
        EXPORT_REGS ();
        if (UNCOMMON (IS_INT (tempOOP = STACKTOP ()))
	    || UNCOMMON (OOP_CLASS (tempOOP) != _gst_block_closure_class)
	    || UNCOMMON (send_block_value (0)))
	  SEND_MESSAGE (_gst_value_symbol, 0, false);

        IMPORT_REGS ();
        FLUSH;

      BC (202, sendValueColon)
	REGISTER (3, OOP tempOOP);
        INCR1;
        EXPORT_REGS ();
        if (UNCOMMON (IS_INT (tempOOP = STACK_AT (1)))
	    || UNCOMMON (OOP_CLASS (tempOOP) != _gst_block_closure_class)
	    || UNCOMMON (send_block_value (1)))
	  SEND_MESSAGE (_gst_value_colon_symbol, 1, false);

        IMPORT_REGS ();
        FLUSH;

      BC (203, sendDoColon)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (_gst_do_colon_symbol, 1, false);
        IMPORT_REGS ();
        FLUSH;
      
      BC (204, sendNew)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (_gst_new_symbol, 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (205, sendNewColon)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (_gst_new_colon_symbol, 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (206, sendIsNil)
	LOAD1;
        DO_JUMP_LOOKAHEAD (STACKTOP () == _gst_nil_oop);
        END;

      BC (207, sendNotNil)
	LOAD1;
        DO_JUMP_LOOKAHEAD (STACKTOP () != _gst_nil_oop);
        END;


      BC (208, sendNoArg0)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (0), 0, false);
        IMPORT_REGS ();
	FLUSH;

      BC (209, sendNoArg1)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (1), 0, false);
        IMPORT_REGS ();
        FLUSH;
  
      BC (210, sendNoArg2)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (2), 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (211, sendNoArg3)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (3), 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (212, sendNoArg4)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (4), 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (213, sendNoArg5)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (5), 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (214, sendNoArg6)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (6), 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (215, sendNoArg7)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (7), 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (216, sendNoArg8)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (8), 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (217, sendNoArg9)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (9), 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (218, sendNoArg10)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (10), 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (219, sendNoArg11)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (11), 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (220, sendNoArg12)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (12), 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (221, sendNoArg13)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (13), 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (222, sendNoArg14)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (14), 0, false);
        IMPORT_REGS ();
        FLUSH;

      BC (223, sendNoArg15)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (15), 0, false);
        IMPORT_REGS ();
        FLUSH;


      BC (224, sendOneArg0)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (0), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (225, sendOneArg1)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (1), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (226, sendOneArg2)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (2), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (227, sendOneArg3)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (3), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (228, sendOneArg4)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (4), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (229, sendOneArg5)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (5), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (230, sendOneArg6)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (6), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (231, sendOneArg7)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (7), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (232, sendOneArg8)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (8), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (233, sendOneArg9)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (9), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (234, sendOneArg10)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (10), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (235, sendOneArg11)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (11), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (236, sendOneArg12)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (12), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (237, sendOneArg13)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (13), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (238, sendOneArg14)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (14), 1, false);
        IMPORT_REGS ();
        FLUSH;

      BC (239, sendOneArg15)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (15), 1, false);
        IMPORT_REGS ();
        FLUSH;


      BC (240, sendTwoArg0)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (0), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (241, sendTwoArg1)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (1), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (242, sendTwoArg2)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (2), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (243, sendTwoArg3)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (3), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (244, sendTwoArg4)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (4), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (245, sendTwoArg5)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (5), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (246, sendTwoArg6)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (6), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (247, sendTwoArg7)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (7), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (248, sendTwoArg8)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (8), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (249, sendTwoArg9)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (9), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (250, sendTwoArg10)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (10), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (251, sendTwoArg11)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (11), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (252, sendTwoArg12)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (12), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (253, sendTwoArg13)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (13), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (254, sendTwoArg14)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (14), 2, false);
        IMPORT_REGS ();
        FLUSH;

      BC (255, sendTwoArg15)
	INCR1;
        EXPORT_REGS ();
        SEND_MESSAGE (METHOD_LITERAL (15), 2, false);
        IMPORT_REGS ();
        FLUSH;

#ifdef JUMP_LOOKAHEAD
      BRANCH_LABEL (lookaheadFailedTrue)
        UNCHECKED_PUSH_OOP (_gst_true_oop);
        NEXT_BC;

      BRANCH_LABEL (lookaheadFailedFalse)
        UNCHECKED_PUSH_OOP (_gst_false_oop);
        NEXT_BC;

      BRANCH_LABEL (lookaheadLongJumpNotTaken) 
        LOAD2;
	LOCAL_COUNTER++;
        NEXT_BC;
#endif

    }				/* BYTECODES */
  }				/* INTERPRETER */
}				/* _gst_interpret() */


/* Always use outer ip/sp outside _gst_interpret */
#ifdef LOCAL_REGS
#  define ip		_gst_ip
#  define sp		_gst_sp
#  if REG_AVAILABILITY >= 2
#    undef _gst_nil_oop
#    undef _gst_true_oop
#    undef _gst_false_oop
#  endif
#  if REG_AVAILABILITY == 0
#    undef LOCAL_COUNTER
#  endif
#endif
