/* Source Installer, Copyright (c) 2005 Claudio Fontana

 srcmisc.c - miscellaneous utility functions

 This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

 This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
     along with this program (look for the file called COPYING);
     if not, write to the Free Software Foundation, Inc.,
         51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

     You can contact the author (Claudio Fontana) by sending a mail
     to claudio@gnu.org
*/

#include "src_stdinc.h"

char* srcinst_chomp(char* str) {
  size_t len;

  if ((len = strlen(str)) == 0)
    return str;

  if (str[len - 1] == '\n')
    str[len - 1] = 0;
  
  return str;
}

char* srcinst_rstrip(char* s, char c) {
  size_t l;

  if ((l = strlen(s)))
    if (s[l - 1] == c)
      s[l - 1] = 0;

  return s;
}

char* srcinst_readline(char* str, int size, FILE* f) {
  char* rv;

  if (!(rv = fgets(str, size, f)))
    return 0;

  return srcinst_chomp(rv);
}

int srcinst_chown_user(char* pathname) {
  uid_t r_uid; gid_t r_gid;
  uid_t e_uid; gid_t e_gid;
  int rv = 0;

  r_uid = getuid(); r_gid = getgid();
  e_uid = geteuid(); e_gid = getegid();

  if (e_uid != r_uid || e_gid != r_gid)
    rv = chown(pathname, r_uid, r_gid);
  
  return !rv;
}

char* srcinst_file_readlink(char* filename) {
  char* buffer; size_t size; int count;

  if (srcinst_file_ltype(filename) != SRCINST_TYPE_LINK)
    return 0;

  buffer = 0; size = SRCINST_BUFSIZE; 

  while (1) {
    buffer = srcinst_realloc(buffer, size + 1);
    if ((count = readlink(filename, buffer, size)) < 0) {
      free(buffer); return 0;

    } else if (count < size) {
      buffer = srcinst_realloc(buffer, count + 1);
      buffer[count] = 0; return buffer;
      
    } else {
      /* maybe we need a bigger buffer? */
      size += SRCINST_BUFSIZE;
    }
  }
  /* unreachable code */
  return 0;
}

int srcinst_file_ltype(char* filename) {
  struct stat buf;

  return
    lstat(filename, &buf) != 0 ? SRCINST_TYPE_NONE :
    S_ISREG(buf.st_mode) ? SRCINST_TYPE_FILE :
    S_ISDIR(buf.st_mode) ? SRCINST_TYPE_DIR :
    S_ISLNK(buf.st_mode) ? SRCINST_TYPE_LINK :
    SRCINST_TYPE_OTHER;
}

int srcinst_file_type(char* filename) {
  struct stat buf;

  return
    stat(filename, &buf) != 0 ? SRCINST_TYPE_NONE :
    S_ISREG(buf.st_mode) ? SRCINST_TYPE_FILE :
    S_ISDIR(buf.st_mode) ? SRCINST_TYPE_DIR :
    SRCINST_TYPE_OTHER;
}

int srcinst_file_perm(char* filename, unsigned int* perm) {
  struct stat buf;

  if (stat(filename, &buf) != 0)
    return 0;
  
  *perm = (buf.st_mode & 07777);
  return 1;
}

int srcinst_file_owner(char* filename, unsigned int* usr, unsigned int* grp) {
  struct stat buf;
  
  if (stat(filename, &buf) != 0)
    return 0;

  *usr = buf.st_uid; *grp = buf.st_gid;
  return 1;
}

off_t srcinst_file_size(char* filename) {
  struct stat buf;
  return stat(filename, &buf) != 0 ? -1 : buf.st_size;
}

/* format string may contain rwx */
int srcinst_file_access(char* filename, char* format) {
  int mode = F_OK;

  if (strchr(format, 'r')) { mode |= R_OK; }
  if (strchr(format, 'w')) { mode |= W_OK; }
  if (strchr(format, 'x')) { mode |= X_OK; }

  return access(filename, mode) == 0;
}

/* return the extension of filename, or the empty string if not present.
   Does not alloc on the heap, it returns a pointer in the original string. */

char* srcinst_file_ext(char* filename) {
  char* dot; size_t len = strlen(filename);

  for (dot = filename + (len - 1); dot >= filename; dot--) {
    if (*dot == '.')
      return dot;
  }

  return filename + len;
}

char* srcinst_save_cwd(void) {
  char buffer[SRCINST_BUFSIZE];
  
  if (!getcwd(buffer, SRCINST_BUFSIZE)) {
    /* buffer might be too small, or cwd could be invalid */
    srcinst_warning(SRCINST_ERR_CORE, "could not get current directory", "srcinst_save_cwd");
    return 0;
  }
  
  return srcinst_strdup(buffer);
}

int srcinst_load_cwd(char* saved_cwd) {
  int rv;

  if (!saved_cwd) 
    return 1;
  
  if (!(rv = (chdir(saved_cwd) == 0)))
    srcinst_warning(SRCINST_ERR_CORE, "current directory lost", "srcinst_load_cwd");

  free(saved_cwd);
  return rv;
}

char* srcinst_file_normalize(char* filename) {
  char* basename; char* dirname;
  char* oldcwd, *newcwd;
  char* rv;
  SRCINST_TYPE ft;

  oldcwd = srcinst_save_cwd();

  dirname = dir_name(filename); basename = base_name(filename);
  rv = 0;

  if (chdir(dirname) != 0) {
    /* invalid filename, cause it is not reachable */
    goto end_proc;
  }

  if ((ft = srcinst_file_ltype(basename)) == SRCINST_TYPE_DIR) {
    if (chdir(basename) != 0) {
      /* invalid, not reachable */
      goto end_proc;
    }
  }

  newcwd = srcinst_save_cwd();
  rv = (ft == SRCINST_TYPE_DIR ?
	srcinst_strdup(newcwd) : srcinst_fnjoin(newcwd, basename));
  free(newcwd);
  
  srcinst_load_cwd(oldcwd);

 end_proc:
  free(dirname);
  return rv;
}

/* Get the compression format from a file extension. */
SRCINST_COMP srcinst_ext_to_compression_format(char* ext) {

  if (strcmp(ext, ".gz") == 0)
    return SRCINST_COMP_GZ;
  
  if (strcmp(ext, ".bz2") == 0)
    return SRCINST_COMP_BZ2;

  if (strcmp(ext, ".tgz") == 0)
    return SRCINST_COMP_TGZ;

  if (strcmp(ext, ".Z") == 0)
    return SRCINST_COMP_Z;

  if (strcmp(ext, ".tbz2") == 0)
    return SRCINST_COMP_TBZ2;

  return SRCINST_COMP_UNKNOWN;
}

/* very similar, but cut dot and tgz mixed stuff. */
SRCINST_COMP srcinst_str_to_compression_format(char* str) {
  if (strcmp(str, "gz") == 0)
    return SRCINST_COMP_GZ;
  
  if (strcmp(str, "bz2") == 0)
    return SRCINST_COMP_BZ2;

  if (strcmp(str, "Z") == 0)
    return SRCINST_COMP_Z;

  return SRCINST_COMP_UNKNOWN;
}

SRCINST_EXPORT srcinst_str_to_export_format(char* str) {
  if (strcmp(str, "xml") == 0)
    return SRCINST_EXPORT_XML;
  
  if (strcmp(str, "txt") == 0)
    return SRCINST_EXPORT_TXT;
  
  if (strcmp(str, "lst") == 0)
    return SRCINST_EXPORT_LST;
  
  return SRCINST_EXPORT_UNKNOWN;
}

/* Get the archive format from a file extension. */
SRCINST_ARC srcinst_ext_to_archive_format(char* ext) {

  if (strcmp(ext, ".tar") == 0)
    return SRCINST_ARC_TAR;
  
  if (strcmp(ext, ".zip") == 0)
    return SRCINST_ARC_ZIP;

  if (strcmp(ext, ".shar") == 0)
    return SRCINST_ARC_SHAR;

  return SRCINST_ARC_UNKNOWN;
}

char* srcinst_compression_format_to_ext(SRCINST_COMP fmt) {
  switch (fmt) {
  case SRCINST_COMP_GZ:
    return ".gz";
  case SRCINST_COMP_BZ2:
    return ".bz2";
  case SRCINST_COMP_TGZ:
    return ".tgz";
  case SRCINST_COMP_TBZ2:
    return ".tbz2";
  case SRCINST_COMP_Z:
    return ".Z";
  default:
    return 0;
  }
}

char* srcinst_compression_format_to_str(SRCINST_COMP fmt) {
  switch (fmt) {
  case SRCINST_COMP_GZ:
    return "gz";
  case SRCINST_COMP_BZ2:
    return "bz2";
  case SRCINST_COMP_Z:
    return "Z";
  default:
    return 0;
  }
}

char* srcinst_archive_format_to_ext(SRCINST_ARC fmt) {
  switch (fmt) {
  case SRCINST_ARC_TAR:
    return ".tar";
  case SRCINST_ARC_ZIP:
    return ".zip";
  case SRCINST_ARC_SHAR:
    return ".shar";
  default:
    return 0;
  }
}

int srcinst_bsearch(void* array, int n, char* k, char* (*accessor)(void*,int)) {
  int i, low, high;
  
  low = 0; high = n;
  
  while (low < high) {
    int check;
    
    i = (low + high) / 2;
    check = strcmp(k, accessor(array, i));
    
    if (check < 0) {
      high = i;
      
    } else if (check > 0) {
      low = i + 1;
      
    } else {
      /* found */
      break;
    }
  }
  
  if (low >= high) {
    return -1;
  }
  return i;
}

int srcinst_lsearch(char** array, int n, char* k) {
  int i;
  for (i = 0; i < n; i++) {
    if (strcmp(k, array[i]) == 0)
      return i;
  }
  return -1;
}

void srcinst_qsort(char** array, int n, int (*compar)(char**, char**)) {
  qsort(array, n, sizeof(char*), compar);
}

char* srcinst_replace(char* haystack, char* needle, char* rep) {
  char* buffer; size_t n;
  char* ptr, *match, *end;
  size_t len, len_needle, len_rep;

  buffer = srcinst_malloc(SRCINST_BUFSIZE); n = 1;
  ptr = buffer; end = buffer + (SRCINST_BUFSIZE - 1);
  len_needle = strlen(needle); len_rep = strlen(rep);
  
  while ((match = strstr(haystack, needle))) {
    len = match - haystack;
    
    if (ptr + len + len_rep >= end) {
      n += 1 + (len + len_rep) / SRCINST_BUFSIZE;
      buffer = srcinst_realloc(buffer, SRCINST_BUFSIZE * n);
      end = buffer + ((SRCINST_BUFSIZE * n) - 1);
    }
    memcpy(ptr, haystack, len); ptr += len;
    memcpy(ptr, rep, len_rep); ptr += len_rep;
    haystack = match + len_needle;
  }

  len = strlen(haystack);

  if (ptr + len >= end) {
    n += 1 + len / SRCINST_BUFSIZE;
    buffer = srcinst_realloc(buffer, SRCINST_BUFSIZE * n);
  }
  
  memcpy(ptr, haystack, len); ptr[len] = 0;
  return buffer;
}

char* srcinst_lower(char* str) {
  char* ptr;
  ptr = str = srcinst_strdup(str);
  while ((*ptr = tolower(*ptr))) ptr++;
  return str;
}

char* srcinst_upper(char* str) {
  char* ptr;
  ptr = str = srcinst_strdup(str);
  while ((*ptr = toupper(*ptr))) ptr++;
  return str;
}

char* srcinst_offtostr(off_t n) {
  char offstr[INT_BUFSIZE_BOUND(off_t)];
  return srcinst_strdup(offtostr(n, offstr));
}

char* srcinst_basename(char* filename) {
  return base_name(filename);
}

char* srcinst_dirname(char* filename) {
  return dir_name(filename);
}
