/* Source Installer, Copyright (c) 2005 Claudio Fontana

   srccfgopt.c - get configure script options

   This program is free software; you can redistribute it and/or modify
       it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
       (at your option) any later version.

   This program is distributed in the hope that it will be useful,
       but WITHOUT ANY WARRANTY; without even the implied warranty of
       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
       GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
       along with this program (look for the file called COPYING);
       if not, write to the Free Software Foundation, Inc.,
           51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

       You can contact the author (Claudio Fontana) by sending a mail
       to claudio@gnu.org
*/

#include "src_stdinc.h"
#include "actions.h"
#include "package_info.h"

struct _cfg_keywords {
  char* name;			/* verbose name of the section */
  char* keyword;		/* regex of the keyword of the section */
  char* pattern;		/* regex of the option to match within sect */
  struct srcinst_configure_options* o;
};

#define SRC_MAIN_PATTERN "^  (--[[:alnum:]_-]+)(\\[?=[[:alnum:]_-]+\\]?)?[[:space:]]*([^[:space:]].*)?"
#define SRC_ENV_PATTERN "^  ([[:alnum:]_]+)(\\[?=[[:alnum:]_-]+\\]?)?[[:space:]]*([^[:space:]].*)?"

static struct _cfg_keywords _cfg_keywords[] = {
  { "Optional Features", "^Optional Features", SRC_MAIN_PATTERN, 0 },
  { "Optional Packages and X", "^(Optional Packages|X features)", SRC_MAIN_PATTERN, 0 },
  { "Environment Variables", "^Some influential environment variables", SRC_ENV_PATTERN, 0 },
  { "Optional Features and Packages", "^--enable and --with options recognized", SRC_MAIN_PATTERN, 0 },
  { 0, 0, 0, 0 }
};

static void _init_configure_options(struct srcinst_configure_options* o) {
  o->options = 0; o->count = 0;
}

static void _free_configure_options(struct srcinst_configure_options* o) {
  int i;
  if (o->options) {
    for (i = 0; i < o->count; i++) {
      if (o->options[i].key)
	free(o->options[i].key);
      if (o->options[i].comment)
	free(o->options[i].comment);
      if (o->options[i].value)
	free(o->options[i].value);
    }
    free(o->options);
  }
  _init_configure_options(o);
}

static void _init_configure_option(struct srcinst_configure_option* o, char* key, char* param, char* comment, char* value) {
  o->key = srcinst_strdup(key);
  o->param = param ? srcinst_strdup(param) : 0;
  o->comment = comment ? srcinst_strdup(comment) : 0;
  o->value = value ? srcinst_strdup(value) : 0;
}

static void _add_configure_option(struct srcinst_configure_options* o, char* key, char* param, char* comment, char* value) {
  int idx; idx = o->count++;
  o->options = srcinst_realloc(o->options, sizeof(struct srcinst_configure_option) * o->count);
  _init_configure_option(o->options + idx, key, param, comment, value);
}

static void _add_base_configure_options(struct srcinst_configure_options* o) {
  int idx; idx = o->count;
  o->count += 13;
  o->options = srcinst_realloc(o->options, sizeof(struct srcinst_configure_option) * o->count);

  /*
    _init_configure_option(o->options + idx++, "--prefix", "=PREFIX", "install files in PREFIX [default=/usr/local]", 0);
  */

  _init_configure_option(o->options + idx++, "--exec-prefix", "=EPREFIX", "install architecture-dependent files in EPREFIXs [default=PREFIX]", 0);
  _init_configure_option(o->options + idx++, "--bindir", "=DIR", "user executables [EPREFIX/bin]", 0);
  _init_configure_option(o->options + idx++, "--sbindir", "=DIR", "system admin executables [EPREFIX/sbin]", 0);
  _init_configure_option(o->options + idx++, "--libexecdir", "=DIR", "program executables [EPREFIX/libexec]", 0);
  _init_configure_option(o->options + idx++, "--datadir", "=DIR", "read-only architecture-independent data [PREFIX/share]", 0);
  _init_configure_option(o->options + idx++, "--sysconfdir", "=DIR", "read-only single-machine data [PREFIX/etc]", 0);
  _init_configure_option(o->options + idx++, "--sharedstatedir", "=DIR", "modifiable architecture-independent data [PREFIX/com]", 0);
  _init_configure_option(o->options + idx++, "--localstatedir", "=DIR", "modifiable single-machine data [PREFIX/var]", 0);
  _init_configure_option(o->options + idx++, "--libdir", "=DIR", "object code libraries [EPREFIX/lib]", 0);
  _init_configure_option(o->options + idx++, "--includedir", "=DIR", "C header files [PREFIX/include]", 0);
  _init_configure_option(o->options + idx++, "--oldincludedir", "=DIR", "C header files for non-gcc [/usr/include]", 0);
  _init_configure_option(o->options + idx++, "--infodir", "=DIR", "info documentation [PREFIX/info]", 0);
  _init_configure_option(o->options + idx++, "--mandir", "=DIR", "man documentation [PREFIX/man]", 0);
  
}

void _action_free_configure_opt(struct srcinst_configure_sets* sets) {
  _free_configure_options(&sets->base);
  _free_configure_options(&sets->feats);
  _free_configure_options(&sets->packs);
  _free_configure_options(&sets->envs);
  _free_configure_options(&sets->olds);
  _free_configure_options(&sets->res1);
  _free_configure_options(&sets->res2);
}

SRCINST_ERR _action_detect_configure_opt(struct _package_info* i, struct srcinst_configure_sets* sets) {
  struct srcinst_string_list output;
  
  if (!_check_build_ready(i))
    return SRCINST_ERR_INVALID;
  
  if (!_detect_configure())
    return SRCINST_ERR_MISSING;

  _init_configure_options(&sets->base);
  _init_configure_options(&sets->feats);
  _init_configure_options(&sets->packs);
  _init_configure_options(&sets->envs);
  _init_configure_options(&sets->olds);
  _init_configure_options(&sets->res1);
  _init_configure_options(&sets->res2);
  _init_configure_options(&sets->res3);

  if (srcinst_file_access("./configure", "x")) {
    if (!srcinst_spawn_wait_expect_lines(&output, "./configure", "--help", 0))
      return SRCINST_ERR_ACTION;

  } else {
    if (!srcinst_spawn_wait_expect_lines(&output, PRG_sh, "./configure", "--help", 0))
      return SRCINST_ERR_ACTION;
  }

  _add_base_configure_options(&sets->base);

  _cfg_keywords[0].o = &sets->feats; _cfg_keywords[1].o = &sets->packs;
  _cfg_keywords[2].o = &sets->envs; _cfg_keywords[3].o = &sets->olds;

  do {
    SRCINST_REGEX skip; SRCINST_REGEX cont; SRCINST_MATCH cont_m;
    int lineno; struct _cfg_keywords* section;
    skip = srcinst_init_regex("^  --[[:alnum:]]+-(PACKAGE|FEATURE)", -1);
    /* printf("PASS 1\n"); fflush(stdout); */
    cont = srcinst_init_regex("^[[:space:]]+([^[:space:]].*)", -1);
    /* printf("PASS 2\n"); fflush(stdout); */
    cont_m = srcinst_init_match(2);

    for (lineno = 0; lineno < output.count; lineno++) {
      for (section = _cfg_keywords; section->keyword; section++) {
	char* line; SRCINST_REGEX r; line = output.strings[lineno];
	r = srcinst_init_regex(section->keyword, REG_EXTENDED | REG_ICASE);
	/* printf("PASS 3 lineno=%i\n", lineno); fflush(stdout); */
	if (srcinst_exec_regex(r, line, 0)) {
	  /* matched section start */
	  SRCINST_REGEX reg; SRCINST_MATCH m;
	  reg = srcinst_init_regex(section->pattern, -1);
	  /* printf("PASS 4 lineno=%i\n", lineno); fflush(stdout); */
	  m = srcinst_init_match(4);
	  
	  while (1) {
	    char* key, *param, *comment; int fwidx;
	    if (++lineno >= output.count) break;
	    line = output.strings[lineno];
	    if (!srcinst_exec_regex(reg, line, m))   /* sub, par, com */
	      break;
	    if (srcinst_exec_regex(skip, line, 0))
	      continue;
	    key = srcinst_get_match(m, 1);
	    param = srcinst_get_match(m, 2);   /* can be (char*)0 */
	    comment = srcinst_get_match(m, 3); /* can be (char*)0 */
	    
	    fwidx = 0;		/* scan forward to fetch continued comments */
	    while (1) {
	      char* fwline;
	      ++fwidx;
	      if (lineno + fwidx >= output.count) break;
	      fwline = output.strings[lineno + fwidx];
	      if (*fwline == '\0') break;
	      if (srcinst_exec_regex(reg, fwline, 0)) break;
	      if (!srcinst_exec_regex(cont, fwline, cont_m)) break;
	      if (comment) comment = srcinst_append(comment, " ");
	      comment = srcinst_append(comment, fwline + srcinst_get_match_offset(cont_m, 1));
	    }
	    _add_configure_option(section->o, key, param, comment, 0);
	    if (key) free(key);
	    if (param) free(param);
	    if (comment) free(comment);
	    lineno += fwidx - 1;
	  }
	  srcinst_free_match(m);
	  srcinst_free_regex(reg);
	}
	srcinst_free_regex(r);
	if (lineno >= output.count)
	  break;
      }
    }
    srcinst_free_regex(skip);
    srcinst_free_regex(cont);
    srcinst_free_match(cont_m);
  } while (0);
  
  return SRCINST_ERR_OK;
}
