/* Source Installer, Copyright (c) 2005,2006,2007 Claudio Fontana

 srcmisc.h - miscellaneous utility functions

 This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
     (at your option) any later version.

 This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
     along with this program (look for the file called COPYING);
     if not, write to the Free Software Foundation, Inc.,
         51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

     You can contact the author (Claudio Fontana) by sending a mail
     to claudio@gnu.org
*/

#ifndef SRCINST_MISC_H
#define SRCINST_MISC_H


/* modify the string pointed by STR, removing the last character if it is
   a newline. Otherwise, the string is not altered. */

char* srcinst_chomp(char* str);

/* modify the string pointed by S, cutting the last char if it is equal to C.
   Returns S. */
   
char* srcinst_rstrip(char* s, char c);

/* following function just calls fgets(3), and strcinst_chomp. */

char* srcinst_readline(char* str, int size, FILE* f);

/* following function chowns a pathname to the real uid and gid */
int srcinst_chown_user(char* pathname);

/* the following are return values of srcinst functions that return the
   "type" of a file. A value of NONE means that the file does not exist, or
   its existence is impossible to determine. */

typedef enum {
  SRCINST_TYPE_NONE, SRCINST_TYPE_FILE, SRCINST_TYPE_DIR, SRCINST_TYPE_LINK,
  SRCINST_TYPE_OTHER

} SRCINST_TYPE;


/* These are format codes. */

typedef enum {
  SRCINST_COMP_UNKNOWN,
  SRCINST_COMP_Z, SRCINST_COMP_BZ2, SRCINST_COMP_GZ,
  SRCINST_COMP_TGZ, SRCINST_COMP_TBZ2

} SRCINST_COMP;

typedef enum {
  SRCINST_ARC_UNKNOWN,
  SRCINST_ARC_SHAR, SRCINST_ARC_TAR, SRCINST_ARC_ZIP

} SRCINST_ARC;

typedef enum {
  SRCINST_EXPORT_UNKNOWN,
  SRCINST_EXPORT_LST, SRCINST_EXPORT_TXT, SRCINST_EXPORT_XML

} SRCINST_EXPORT;

/* alloc a new string containing the 0 terminated text in the symbolic link. */
char* srcinst_file_readlink(char* filename);

/* this one returns the file type. A link is properly reported as link. */
int srcinst_file_ltype(char* filename);

/* very similar to the last, this one resolves links before reporting type */
int srcinst_file_type(char* filename);

/* get permissions of FILENAME, and write them as an int pointed to by PERM.
   Returns a boolean value, 0=err, 1=ok */
int srcinst_file_perm(char* filename, mode_t* perm);

/* get FILENAME owner and group */
int srcinst_file_owner(char* filename, uid_t* usr, gid_t* grp);

/* want the file size in bytes? Here's how to get it. LARGE_FILE warning. */
off_t srcinst_file_size(char* filename);

/* see man 2 access. FORMAT is scanned for 'r', 'w' and 'x' chars. 
   Return value is accessible(1) or not accessible(0).
   Existence-only check is not possible with this foo,
   use file_[l]type instead */

int srcinst_file_access(char* filename, char* format);

/* ok, watch out for the next one.
   It validates the existence of the directory part of the PATHNAME,
   then returns the absolute path of PATHNAME. Accepts both abs and rel paths.
   On failure, (char*)0 is returned */

char* srcinst_file_normalize(char* pathname);

/* get the extension part of a filename. Does not alloc a new string,
   it just returns a pointer into the original string.
   The dot is included.
   If the file has no extension, the return value points at the end of
   the string ("")
*/

/* get the current working directory, return a new string for it.
   Returns (char*)0 on failure. */
char* srcinst_save_cwd(void);

/* restore current working directory and free(saved_cwd);
   Returns 0=lost cwd, or 1=success */
int srcinst_load_cwd(char* saved_cwd);

/* get file extension part of FILENAME, dot included. The returned pointer
   points _into_ FILENAME */
char* srcinst_file_ext(char* filename);

/* from string (gz, bz2, Z) to compression format */
SRCINST_COMP srcinst_str_to_compression_format(char* str);

/* from string (txt, lst, xml) to export format */
SRCINST_EXPORT srcinst_str_to_export_format(char* str);

/* from extension (.gz, .bz2, .Z, .tgz, .tbz2) to compression format */
SRCINST_COMP srcinst_ext_to_compression_format(char* ext);

/* from extension (.tar, .shar, .zip) to archive format */
SRCINST_ARC srcinst_ext_to_archive_format(char* ext);

/* from compression format to extension. Returns pointer to static data */
char* srcinst_compression_format_to_ext(SRCINST_COMP fmt);

/* from archive format to extension. Returns pointer to static data */
char* srcinst_archive_format_to_ext(SRCINST_ARC fmt);

/* from compression format to string (gz, bz2, Z) */
char* srcinst_compression_format_to_str(SRCINST_COMP fmt);

/* a function that takes a generic HAYSTACK pointer that represents an array of
   N elements and a K keyword to search. The i-th keyword element of HAYSTACK
   is accessed through an ACCESSOR function also passed.
   The ACCESSOR simply takes the HAYSTACK pointer as first argument and an
   integer index as the second argument.
   For an example of an accessor, see srcinst_get_parsed_data_key() in parse.c
 */

int srcinst_bsearch(void* haystack, int n, char* k, char* (*accessor)(void*, int));

/* this function searches the ARRAY of N strings for a string matching K and
   returns its position, using linear search. */

int srcinst_lsearch(char** array, int n, char* k);

/* this function sorts the ARRAY of N strings, using CMP to enforce order */

void srcinst_qsort(char** array, int n, int (*cmp)(char**, char**));

/* this function returns a new string alloc-ed on the heap. The string
   contains the haystack strings, where _all_ occurrences of NEEDLE have been
   replaced by REP */

char* srcinst_replace(char* haystack, char* needle, char* rep);

/* alloc new string as a lower-case version of STR */
char* srcinst_lower(char* str);

/* alloc new string as an upper-case version of STR */
char* srcinst_upper(char* str);

/* alloc new converted string */
char* srcinst_offtostr(off_t n);

/* return a pointer into filename (do _not_ free) */
char* srcinst_basename(char* filename);

/* return a newly allocated string: _do_ free */
char* srcinst_dirname(char* filename);

#endif /* SRCINST_MISC_H */

