;;;; tib.el - Emacs interface to the Tib bibliography system.
;;;; Requires Olin Shiver's comint package.

;; Copyright (C) 1990 by Sebastian Kremer <sk@thp.uni-koeln.de>

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;; Lispdir entry for the Elisp Archive:
;;    tib|Sebastian Kremer|sk@thp.uni-koeln.de
;;    |Enter and check bibliographic references for Tib.
;;    |$Date: 90/10/30 11:10:29 $|$Revision: 1.2 $|

;; For your .emacs:

;;    (global-set-key "\C-c[" 'tib-insert-ref)
;;    (global-set-key "\C-c." 'tib-check-reference)
;;    (autoload 'tib-insert-ref "tib" "Insert a tib reference." t)
;;    (autoload 'tib-check-reference "tib" "Check a tib reference." t)
;;    (autoload 'tiblook "tib" "Run tiblook in a buffer." t)

;; Instead of global-set-key, you might want to bind it in
;; (la)tex-mode-map only, in your (la)tex-mode-hook.

(defconst tib-version
  "$Id: tib.el,v 1.2 90/10/30 11:10:29 sk Exp $")

(require 'comint)



























;;; Support for editing tex files containing tib references:

(defun tib-insert-ref ()
  "Insert a tib reference \"[..]\" and put cursor into it."
  (interactive)
  (insert "[.")
  (save-excursion (insert ".]")))

(defconst tib-ref-beginning "\\(\\[\\.\\)\\|\\(,[ \t\n]*\\)"
  "Regexp matching the beginning of a Tib reference.")

(defconst tib-ref-end "\\(\\.\\]\\)\\|\\(,[ \t\n]*\\)"
  "Regexp matching the end of a Tib reference.")

(defun tib-reference-at-point ()
  (save-excursion
    (let ((low (save-excursion (forward-line -1)
			       (point)))
	  (high (save-excursion (forward-line 1)
				(end-of-line)
				(point)))
	  start)
      (or (setq start
		(or (if (looking-at "\\[\\.") (match-end 0))
		    (save-excursion
		      (if (re-search-backward tib-ref-beginning low t)
			  (match-end 0)))
		    (save-excursion
		      (if (re-search-forward tib-ref-beginning high t)
			  (match-end 0)))))
	  (error "Could not find beginning of tib reference."))
      (goto-char start)
      (or (re-search-forward tib-ref-end high t)
	  (error "Could not find end of tib reference."))
      (buffer-substring start (match-beginning 0)))))

(defun tib-check-reference (reference)
  "Check the tib reference around point .
If not on \"\[.\", has a preference for looking backwards.
Pops up a *tiblook* buffer and feeds it tib-reference-at-point.
See also variables tib-ref-beginning and tib-ref-end."
  (interactive (list (tib-reference-at-point)))
  (message "Checking '%s'" reference)
  (let ((buf (current-buffer)))
    (tiblook)				; pop to tiblook buffer
    (goto-char (point-max))
    (insert "\t")			; to separate it visually
    (insert reference)			; from manual input
    (comint-send-input)
    ;; wait for output so that point moves and output is visible:
    (sit-for 1)
    (recenter -1)
    ;; go back to editing buffer:
    (pop-to-buffer buf)))






















;;; The tiblook interface is a customized comint mode.

(defvar tiblook-mode-map (copy-keymap comint-mode-map))

(defvar tiblook-mode-hook nil
  "*Hook for customising tiblook mode.")

(defvar tiblook-prompt-pattern "^\*"
  "Regexp matching tiblook's prompt.")

(defun tiblook-mode ()
  "Major mode for interacting with tiblook.
Return after the end of the process' output sends the text from the
    end of process to the end of the current line.
Return before end of process output copies rest of line to end (skipping
    the prompt) and sends it.

If you accidentally suspend your process, use \\[comint-continue-subjob]
to continue it.

\\{tiblook-mode-map}
Customisation: Entry to this mode runs the hooks on comint-mode-hook and
tiblook-mode-hook (in that order).
"
  (interactive)
  (comint-mode)
  (setq comint-prompt-regexp tiblook-prompt-pattern)
  (setq major-mode 'tiblook-mode)
  (setq mode-name "Tiblook")
  (use-local-map tiblook-mode-map)
  (run-hooks 'tiblook-mode-hook))

(defun tiblook ()
  "Run an inferior tiblook, with I/O through buffer *tiblook*.
If buffer exists but tiblook process is not running, make new process.
If buffer exists and tiblook process is running, just switch to *tiblook*.
The buffer is put in tiblook-mode.
See also variable tiblook-prompt-pattern.

\(Type \\[describe-mode] in the tiblook buffer for a list of commands.)"
  (interactive)
  (cond ((not (comint-check-proc "*tiblook*"))
	 (set-buffer (make-comint "tiblook" "tiblook"))
	 (tiblook-mode)))
  (pop-to-buffer "*tiblook*"))

(provide 'tib)


