/*
** Datei: DVIAMIGA.C
** Autor: Markus Zahn
*/

#include <ctype.h>
#include <math.h>
#include <setjmp.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <proto/asl.h>
#include <proto/dos.h>
#include <proto/exec.h>
#include <proto/gadtools.h>
#include <proto/graphics.h>
#include <proto/icon.h>
#include <proto/intuition.h>
#include <proto/layers.h>
#include <proto/wb.h>

#include <devices/printer.h>
#include <exec/memory.h>
#include <graphics/gfxbase.h>
#include <intuition/gadgetclass.h>
#include <intuition/intuitionbase.h>

#include "dvimisc.h"
#include "dvi.h"
#include "dvidvi.h"
#include "dviframe.h"
#include "dvihdcp.h"
#include "dvisplin.h"
#include "dvidraw.h"
#include "dviver.h"
#include "dviamiga.h"
#include "dvi_req.h"
#include "dvishrink.h"

#ifndef MIN
#define MIN(a,b) ((a)<(b) ? (a):(b))
#endif
#ifndef MAX
#define MAX(a,b) ((b)<(a) ? (a):(b))
#endif
#ifndef ABS
#define ABS(a) ((a)>0 ? (a):-(a))
#endif
#define ON_OFF(a) ((a) ? "ON":"OFF")
#define COMMENT(exp) ((exp) ? ";":"" )

#define BUFSIZE 1024 /* Groesse des Druckerpuffers */

/*
 * Prototypen
 */
void about_dvi( void );
void close_all( void );
void close_sbm_window( void );
void ClearBusyPointer( struct Window *window );
void dir_setup( void );
void dvi_clean_amiga( void );
int dvi_filereq( void );
int dvi_screenmode( void );
int format_pages_amiga( int first, int last, int step );
void get_dimen( char *string, double *dimen );
void get_opts( int argc );
int goto_page( int nr );
void halt( char *format, ... );
void main_loop( void );
void margins_setup( void );
int open_all( void );
int open_menu( void );
int open_screen( void );
int open_sbm_window( void );
int open_window( void );
void out_newline( void );
void out_string( char *s );
void parse_opts( struct RDArgs *parse );
void prbyte( int c );
void print_pages( int start, int end, int step );
void print_range( void );
void printer_setup( void );
int process_idcmp_messages( struct MsgPort *msgport, int *newpage, int *update );
int process_wb_messages( struct MsgPort *app_menu_port );
void quote_opts( char *opt, char *quoted );
int request_integer( int *value, char *txt );
int request_range( int *first, int *last, int *step );
void SetBusyPointer( struct Window *window );
static void set_sizes( void );
int stop_key( void );
void update_window( void );
void write_opts( void );

/*
 * Globale Variablen fr Screen, Window, Libraries etc.
 * Die System LibraryBases werden vom SAS/C > 6.50 automatisch initialisiert!
 */
struct Library *PSbase = NULL; /* post.library */
struct Process *process = NULL;
APTR console_window, console_task;
struct Screen *scr = NULL;
struct Window *win = NULL, *sbm_win = NULL;
struct BitMap *bmap = NULL;
struct Menu *mnu = NULL;
struct IOStdReq *prt_req;
APTR vis = NULL;
BPTR inout_handle = MKBADDR( NULL ), old_in_handle = MKBADDR( NULL );
struct MsgPort *app_menu_port = NULL;
struct AppMenuItem *app_menu_item = NULL;
static struct NewMenu new_menu[] =
{
  NM_TITLE, (STRPTR)"Project", NULL, 0, 0, NULL,
   NM_ITEM, (STRPTR)"Load DVI...", (STRPTR)"L", 0, 0, NULL,
   NM_ITEM, (STRPTR)"Update DVI", (STRPTR)"U", NM_ITEMDISABLED, 0, NULL,
   NM_ITEM, (STRPTR)"Save Options", (STRPTR)"S", 0, 0, NULL,
   NM_ITEM, (STRPTR)NM_BARLABEL, NULL, 0, 0, NULL,
   NM_ITEM, (STRPTR)"About...", NULL, 0, 0, NULL,
   NM_ITEM, (STRPTR)NM_BARLABEL, NULL, 0, 0, NULL,
   NM_ITEM, (STRPTR)"Quit", (STRPTR)"Q", 0, 0, NULL,
  NM_TITLE, (STRPTR)"Screen", NULL, NM_MENUDISABLED, 0, NULL,
   NM_ITEM, (STRPTR)"Next Page", NULL, 0, 0, NULL,
   NM_ITEM, (STRPTR)"Previous Page", NULL, 0, 0, NULL,
   NM_ITEM, (STRPTR)"Goto Page...", (STRPTR)"G", 0, 0, NULL,
   NM_ITEM, (STRPTR)NM_BARLABEL, NULL, 0, 0, NULL,
   NM_ITEM, (STRPTR)"Magnification...", (STRPTR)"M", 0, 0, NULL,
  NM_TITLE, (STRPTR)"Printer", NULL, NM_MENUDISABLED, 0, NULL,
   NM_ITEM, (STRPTR)"Print Range...", (STRPTR)"P", 0, 0, NULL,
   NM_ITEM, (STRPTR)"Print Page", NULL, 0, 0, NULL,
   NM_ITEM, (STRPTR)"Print Document", NULL, 0, 0, NULL,
  NM_TITLE, (STRPTR)"Setup", NULL, 0, 0, NULL,
   NM_ITEM, (STRPTR)"Directories...", NULL, 0, 0, NULL,
   NM_ITEM, (STRPTR)"Margins...", NULL, 0, 0, NULL,
   NM_ITEM, (STRPTR)"Options", NULL, 0, 0, NULL,
#define FIRST_OPTIONS 22
    NM_SUB, (STRPTR)"Eject", NULL, CHECKIT | MENUTOGGLE, 0, NULL,
    NM_SUB, (STRPTR)"Landscape", NULL, CHECKIT | MENUTOGGLE, 0, NULL,
    NM_SUB, (STRPTR)"Memory", NULL, CHECKIT | MENUTOGGLE, 0, NULL,
    NM_SUB, (STRPTR)"Pictures", NULL, CHECKIT | MENUTOGGLE, 0, NULL,
    NM_SUB, (STRPTR)"Separate", NULL, CHECKIT | MENUTOGGLE, 128, NULL,
    NM_SUB, (STRPTR)"Showfonts", NULL, CHECKIT | MENUTOGGLE, 0, NULL,
    NM_SUB, (STRPTR)"Singlesheet", NULL, CHECKIT | MENUTOGGLE, 0, NULL,
    NM_SUB, (STRPTR)"Thinout", NULL, CHECKIT | MENUTOGGLE, 16, NULL,
    NM_SUB, (STRPTR)"Tracechars", NULL, CHECKIT | MENUTOGGLE, 0, NULL,
    NM_SUB, (STRPTR)"Tracemem", NULL, CHECKIT | MENUTOGGLE, 0, NULL,
   NM_ITEM, (STRPTR)"Printer...", NULL, 0, 0, NULL,
   NM_ITEM, (STRPTR)NM_BARLABEL, NULL, 0, 0, NULL,
   NM_ITEM, (STRPTR)"Screenmode...", NULL, 0, 0, NULL,
  NM_END, NULL, NULL, 0, 0, NULL
};
/* data for a busy pointer.
** this data must be in chip memory!!!
*/
UWORD __chip waitPointer[] =
    {
    0x0000, 0x0000,     /* reserved, must be NULL */

    0x0400, 0x07C0,
    0x0000, 0x07C0,
    0x0100, 0x0380,
    0x0000, 0x07E0,
    0x07C0, 0x1FF8,
    0x1FF0, 0x3FEC,
    0x3FF8, 0x7FDE,
    0x3FF8, 0x7FBE,
    0x7FFC, 0xFF7F,
    0x7EFC, 0xFFFF,
    0x7FFC, 0xFFFF,
    0x3FF8, 0x7FFE,
    0x3FF8, 0x7FFE,
    0x1FF0, 0x3FFC,
    0x07C0, 0x1FF8,
    0x0000, 0x07E0,

    0x0000, 0x0000,     /* reserved, must be NULL */
    };

/*
 * Globale Variablen
 */
static char buffer[BUFSIZE]; /* Druckerpuffer */
static int count = 0;
char scr_title[80], win_title[80];
char opt_name[MAX_PATH_LEN];
jmp_buf halt_jump;
int bmp_width, bmp_height, act_page = 1;
ami_options aop =
{
  0,          /* Magnification fr Ausdruck */
  0,          /* kein Redirect */
  "",         /* Name der Redirection Datei */
  0,          /* Bitmapgroesse beim Ausdruck */
  "",         /* Modename des DVI Screens */
  INVALID_ID, /* DisplayModeID des DVI Screens */
  DEPTH       /* Tiefe des DVI Screens */
};
struct prt_def p_defs[] =
{
  180, 180, p6low, "p6low",
  360, 360, p6high, "p6high",
  360, 180, p6mid, "p6mid",
  240, 216, fx80, "fx80",
  300, 300, hplj, "hplj",
  100, 100, hpljlow, "hpljlow",
  300, 300, bj300, "bj300"
};
/* und f"ur dvishrink */
int dst_width, dst_height;
int eff_width, eff_height;
UTYPE u_shrinkfactor = SRC, o_shrinkfactor = 0;
int drawframe = TRUE;

/*
 * Version String fr die Abfrage durch 'version'
 */
char version[] = "\0$VER: DVI "DVI_VER" "__AMIGADATE__;

/*
void StripIntuiMessages( struct MsgPort *mp, struct Window *win )
{
  struct IntuiMessage *msg;
  struct Node *succ;

  msg = (struct IntuiMessage *)mp->mp_MsgList.lh_Head;
  while( succ = msg->ExecMessage.mn_Node.ln_Succ )
  {
    if( msg->IDCMPWindow == win )
    {
      Remove( succ );
      ReplyMsg( (struct Message *)msg );
    }
    msg = (struct IntuiMessage *)succ;
  }
}
*/

int open_sbm_window( void )
{
  int plane, ok = TRUE;

  bmp_width = frame_width * 8;
  bmp_height = frame_height + MAX_PINS;

  if( GfxBase->LibNode.lib_Version > 38 )
    bmap = AllocBitMap( bmp_width,
      bmp_height,
      scr->RastPort.BitMap->Depth,
      BMF_CLEAR,
      scr->RastPort.BitMap );
  else
  {
    bmap = AllocMem( sizeof( struct BitMap ), MEMF_PUBLIC | MEMF_CLEAR );
    if( bmap != NULL )
    {
      InitBitMap( bmap, scr->RastPort.BitMap->Depth, bmp_width, bmp_height );

      for( plane = 0; plane < scr->RastPort.BitMap->Depth; plane++ )
      {
        bmap->Planes[plane] = AllocRaster( bmp_width, bmp_height );
        ok = ok && ( bmap->Planes[plane] != NULL );
      }

      if( !ok )
      {
        for( plane = 0; plane < scr->RastPort.BitMap->Depth; plane++ )
          if( bmap->Planes[plane] != NULL )
            FreeRaster( bmap->Planes[plane], bmp_width, bmp_height );
        FreeMem( bmap, sizeof( struct BitMap ) );
        bmap = NULL;
        halt( "Error: Cannot allocate memory for output window." );
        return( FALSE );
      }
    }
  }

  if( bmap == NULL )
  {
    halt( "Error: Cannot allocate memory for output window." );
    return( FALSE );
  }

  /*
   * Speicher da, alles ok, Bitmap initialisiert.
   * Jetzt: SuperBitmap Window
   */
  if( sbm_win = OpenWindowTags( NULL, WA_Top, (ULONG)( scr->BarHeight + 1 ),
      WA_Left, (ULONG)0, WA_Width, (ULONG)scr->Width,
      WA_Height, (ULONG)( scr->Width, scr->Height - scr->BarHeight - 1 ),
      WA_MaxHeight, (ULONG)scr->Height,
      WA_MaxWidth, (ULONG)scr->Width,
      WA_Flags, WFLG_ACTIVATE | WFLG_SUPER_BITMAP | WFLG_GIMMEZEROZERO |
        WFLG_REPORTMOUSE | WFLG_NOCAREREFRESH,
      WA_CloseGadget, (ULONG)TRUE,
      WA_DepthGadget, (ULONG)TRUE,
      WA_NewLookMenus, (ULONG)TRUE,
      WA_IDCMP, IDCMP_VANILLAKEY | IDCMP_RAWKEY | IDCMP_MENUPICK |
        IDCMP_CLOSEWINDOW | IDCMP_MOUSEBUTTONS,
      WA_PubScreen, (ULONG)scr,
      WA_SuperBitMap, bmap,
      TAG_DONE ) )
  {
/*
    sbm_win->UserPort = win->UserPort;
    ModifyIDCMP( sbm_win, WFLG_ACTIVATE | WFLG_SUPER_BITMAP | WFLG_GIMMEZEROZERO |
      WFLG_REPORTMOUSE | WFLG_NOCAREREFRESH );
*/
/*
    if( ( sbm_win->UserData = calloc( 1, sizeof( struct RastPort ) ) ) != NULL )
    {
      memcpy( sbm_win->UserData, sbm_win->RPort, sizeof( struct RastPort ) );
      ((struct RastPort *)( sbm_win->UserData ))->Layer = NULL;

      if( ( ((struct RastPort *)( sbm_win->UserData ))->BitMap =
          AllocBitMap( sbm_win->Width + 15, 1,
            sbm_win->WScreen->RastPort.BitMap->Depth, 0, NULL ) ) != NULL )
      {
*/
        SetRast( sbm_win->RPort, 0 );
        SetMenuStrip( sbm_win, mnu );
        GT_RefreshWindow( sbm_win, NULL );

#ifdef SHRINK
        set_sizes(); /* f"ur dvishrink */
        initColors();
        init_filter();
#endif

        return( TRUE );
/*
      }
      else
      {
        close_sbm_window();
        halt( "Error: Cannot allocate memory for output window." );
      }
    }
    else
    {
      close_sbm_window();
      halt( "Error: Cannot allocate memory for output window." );
    }
*/
  }
  else
  {
    close_sbm_window();
    halt( "Error: Cannot allocate memory for output window." );
  }
  return( FALSE );
}

void close_sbm_window( void )
{
  int plane;

#ifdef SHRINK
  kill_filter(); /* f"ur dvishrink */
#endif

  if( sbm_win != NULL )
  {
/*
    Forbid();
    StripIntuiMessages( sbm_win->UserPort, sbm_win );
    sbm_win->UserPort = NULL;
    ModifyIDCMP( sbm_win, 0 );
    Permit();
*/
/*
    if( sbm_win->UserData != NULL )
    {
      if( ((struct RastPort *)( sbm_win->UserData ))->BitMap != NULL )
        FreeBitMap( ((struct RastPort *)( sbm_win->UserData ))->BitMap );
      free( sbm_win->UserData );
    }
*/
    CloseWindow( sbm_win );
    sbm_win = NULL;
  }

  if( bmap != NULL )
  {
    if( GfxBase->LibNode.lib_Version > 38 )
      FreeBitMap( bmap );
    else
    {
      for( plane = 0; plane < scr->RastPort.BitMap->Depth; plane++ )
        if( bmap->Planes[plane] != NULL )
          FreeRaster( bmap->Planes[plane], bmp_width, bmp_height );
      FreeMem( bmap, sizeof( struct BitMap ) );
    }
    bmap = NULL;
  }
}

/* Setzt globale Variable "dst_width,dst_height,eff_width und eff_height" */
static void set_sizes(void)
{
    int     width, height;
    width = frame_width * 8;             /* Breite des "frame_buffer" in Pixel */
    height = frame_height;               /* Hoehe des "frame_buffer" in Pixel */

    /* width und height so aufrunden, dass dst_.. % SRC == 0 */
    if (width % SRC != 0)
        dst_width = (width / SRC + 1) * SRC;
    else
        dst_width = width;
    if (height % SRC != 0)
        dst_height = (height / SRC + 1) * SRC;
    else
        dst_height = height;

    /* effektive Groessen von XImage und Pixmap */
    eff_width = dst_width * EXFACTOR;
    eff_height = dst_height * EXFACTOR;
}

void about_dvi( void )
{
  struct EasyStruct req =
  {
    sizeof( struct EasyStruct ),
    0,
    (UBYTE *)"About DVI",
    (UBYTE *)"   DVI %s  1994\n"
             "\n"
             "  Ingo Eichenseher\n"
             "  Gerhard Wilhelms\n"
             "    Markus Zahn\n"
             "\n"
             "Amiga Version  1994\n"
             "    Markus Zahn\n"
             "Zahn@Uni-Augsburg.DE",
    (UBYTE *)"OK"
  };

  EasyRequest( win, &req, NULL, version_string );
}

void close_all( void )
{
  if( PSbase )
    CloseLibrary( PSbase );

  if( app_menu_item != NULL )
    RemoveAppMenuItem( app_menu_item );
  if( app_menu_port != NULL )
    DeleteMsgPort( app_menu_port );

  if( mnu != NULL )
  {
    ClearMenuStrip( win );
    FreeMenus( mnu );
  }

  process->pr_WindowPtr = console_window;
  process->pr_ConsoleTask = console_task;

  if( BADDR( old_in_handle ) != NULL )
    SelectInput( old_in_handle );
  if( BADDR( inout_handle ) != NULL )
    if( Close( inout_handle ) )
      win = NULL; /* Das mit CON: verbundene Fenster wird automatisch geschlossen */

  if( win != NULL )
    CloseWindow( win );
  if( vis != NULL )
    FreeVisualInfo( vis );
  if( scr !=NULL )
    CloseScreen( scr );
}

void ClearBusyPointer( struct Window *window )
{
  if( window != NULL )
  {
    if( IntuitionBase->LibNode.lib_Version > 38 )
      SetWindowPointer( window, TAG_DONE );
    else
      ClearPointer( window );
  }
}

void dir_setup( void )
{
  int end = FALSE, ok = FALSE;
  struct IntuiMessage *imsg;
  struct Gadget *gad;
  struct StringInfo *str_info;
  options op_save;
  
  memcpy( &op_save, &op, sizeof( options ) ); /* Options zwischenspeichern */

  if( OpenDirectoriesWindow() == 0 )
  {
    GT_SetGadgetAttrs( DirectoriesGadgets[GD_pk_path], DirectoriesWnd, NULL,
      GTST_String, (ULONG)op.pk_path, TAG_DONE );
    GT_SetGadgetAttrs( DirectoriesGadgets[GD_tfm_path], DirectoriesWnd, NULL,
      GTST_String, (ULONG)op.tfm_path, TAG_DONE );
    GT_SetGadgetAttrs( DirectoriesGadgets[GD_img_path], DirectoriesWnd, NULL,
      GTST_String, (ULONG)op.img_path, TAG_DONE );
    GT_SetGadgetAttrs( DirectoriesGadgets[GD_gr_path], DirectoriesWnd, NULL,
      GTST_String, (ULONG)op.input_path, TAG_DONE );
    GT_SetGadgetAttrs( DirectoriesGadgets[GD_vf_path], DirectoriesWnd, NULL,
      GTST_String, (ULONG)op.vf_path, TAG_DONE );
    GT_SetGadgetAttrs( DirectoriesGadgets[GD_call_mf], DirectoriesWnd, NULL,
      GTST_String, (ULONG)op.pk_callmf, TAG_DONE );

    ActivateGadget( DirectoriesGadgets[GD_pk_path], DirectoriesWnd, NULL );

    while( !end )
    {
      WaitPort( DirectoriesWnd->UserPort );
      while( imsg = GT_GetIMsg( DirectoriesWnd->UserPort ) )
      {
        gad = imsg->IAddress;

        switch( imsg->Class )
        {
          case CLOSEWINDOW:
            end = TRUE;
            ok = FALSE;
            break;
          case GADGETUP:
            switch( gad->GadgetID )
            {
              case GD_dir_ok:
                end = TRUE;
                ok = TRUE;
                break;
              case GD_dir_cancel:
                end = TRUE;
                ok = FALSE;
                break;
              default:
                break;
            }
            break;
          default:
            break;
        }
        GT_ReplyIMsg( imsg );
      }
    }
    str_info = DirectoriesGadgets[GD_pk_path]->SpecialInfo;
    strcpy( op.pk_path, str_info->Buffer );
    str_info = DirectoriesGadgets[GD_tfm_path]->SpecialInfo;
    strcpy( op.tfm_path, str_info->Buffer );
    str_info = DirectoriesGadgets[GD_img_path]->SpecialInfo;
    strcpy( op.img_path, str_info->Buffer );
    str_info = DirectoriesGadgets[GD_gr_path]->SpecialInfo;
    strcpy( op.input_path, str_info->Buffer );
    str_info = DirectoriesGadgets[GD_vf_path]->SpecialInfo;
    strcpy( op.vf_path, str_info->Buffer );
    str_info = DirectoriesGadgets[GD_call_mf]->SpecialInfo;
    strcpy( op.pk_callmf, str_info->Buffer );

    CloseDirectoriesWindow();
  }

  if( !ok )
    memcpy( &op, &op_save, sizeof( options ) );
}

void dvi_clean_amiga( void )
{
  if( sbm_win != NULL )
    close_sbm_window();
  dvi_clean();
}

int dvi_filereq( void )
{
  struct FileRequester *freq = NULL;
  BPTR file_lock = NULL, dir_lock = NULL;
  int file_exists = FALSE;
  static WORD left = 0, top = 0, width = 250, height = 200;

  if( freq = AllocFileRequest() )
  {
    if( AslRequestTags( freq, ASL_Hail, (ULONG)"Load DVI",
        ASLFR_PositiveText, "Load",
        ASLFR_Window, win,
        ASLFR_DoPatterns, TRUE,
        ASLFR_InitialPattern, "#?.dvi",
        ASLFR_InitialLeftEdge, left,
        ASLFR_InitialTopEdge, top,
        ASLFR_InitialWidth, width,
        ASLFR_InitialHeight, height,
        TAG_DONE ) )
    {
      if( dir_lock = Lock( freq->rf_Dir, ACCESS_READ ) )
      {
        dir_lock = CurrentDir( dir_lock );
        if( file_lock = Lock( freq->rf_File, ACCESS_READ ) )
        {
          UnLock( file_lock );

          strcpy( dvi_name, freq->rf_File );
          file_exists = TRUE;

          dvi_clean_amiga();
          act_page = 1;
        }
        else
          dir_lock = CurrentDir( dir_lock );
        UnLock( dir_lock );
      }

      left = freq->fr_LeftEdge;
      top = freq->fr_TopEdge;
      width = freq->fr_Width;
      height = freq->fr_Height;
    }
    FreeFileRequest( freq );
  }
  return( file_exists );
}

int dvi_screenmode( void )
{
  struct ScreenModeRequester *req;
  struct NameInfo ninfo;
  static WORD left = 0, top = 0, width = 250, height = 200;
  int newid = FALSE;

  if( req = AllocAslRequestTags( ASL_ScreenModeRequest, TAG_DONE ) )
  {
    if( AslRequestTags( req, ASLSM_Window, (ULONG)win,
        ASLSM_TitleText, "Select Screen Mode",
        ASLSM_DoDepth, TRUE,
        ASLSM_MaxDepth, DEPTH,
        ASLSM_InitialDisplayID, aop.displayid,
        ASLSM_InitialDisplayDepth, scr->RastPort.BitMap->Depth,
        ASLSM_InitialLeftEdge, left,
        ASLSM_InitialTopEdge, top,
        ASLSM_InitialWidth, width,
        ASLSM_InitialHeight, height,
        TAG_DONE ) )
    {
      newid = ( aop.displayid != req->sm_DisplayID ) ||
        ( scr->RastPort.BitMap->Depth != req->sm_DisplayDepth );

      aop.depth = req->sm_DisplayDepth;
      aop.displayid = req->sm_DisplayID;
      if( GetDisplayInfoData( NULL, (UBYTE *)&ninfo, sizeof( ninfo ), DTAG_NAME,
          req->sm_DisplayID ) )
        strcpy( aop.screenmode, ninfo.Name );
      else
        print( "No matching ScreenMode available, oops! :-(" );

      left = req->sm_LeftEdge;
      top = req->sm_TopEdge;
      width = req->sm_Width;
      height = req->sm_Height;
    }
    FreeAslRequest( req );
  }

  if( newid )
  {
    print( "Switching screenmode..." );
    dvi_clean_amiga();

    close_all();
    if( !open_all() )
      return( -1 );
    else
      return( 1 );
  }

  return( 0 );
}

int format_pages_amiga( int first, int last, int step )
{
  int page, sbm_sync = FALSE, x, y, bit;
  static int standard = FALSE;

  page = format_pages( first, last, step );
  if( sbm_win == NULL )
  {
    open_sbm_window();
    SetBusyPointer( sbm_win );
    standard = ( GetBitMapAttr( bmap, BMA_FLAGS ) & BMF_STANDARD );
  }

#ifdef SHRINK
  if( u_shrinkfactor != SRC )
  {
    shrink(); /* shrinkt den frame_buffer zurecht und setzt die SuperBitMap */

    LockLayerRom( sbm_win->RPort->Layer );
    SyncSBitMap( sbm_win->RPort->Layer );
    UnlockLayerRom( sbm_win->RPort->Layer );
  }
  else
#endif
  {
    if( standard )
      memcpy( bmap->Planes[0], frame_buffer, frame_size );
    else
    {
      int tmp_width = frame_width * 8;
      struct BitMap *tmp_bmp;
      char *tmp_ptr;

      if( ( tmp_bmp = AllocBitMap( tmp_width, 1, 1, 0, NULL ) ) != NULL )
      {
        for( y = 0, tmp_ptr = frame_buffer; y < frame_height; y++, tmp_ptr += frame_width )
        {
          WaitBlit();
          memcpy( tmp_bmp->Planes[0], tmp_ptr, frame_width );
          BltBitMap( tmp_bmp, 0, 0, bmap, 0, y, tmp_width, 1, 0xc0, 0xff, NULL );
        }
        FreeBitMap( tmp_bmp );
      }
      else
        halt( "Unable to allocate memory for temporary bitmap!" );
    }
  }

  o_shrinkfactor = u_shrinkfactor;
  return( page );
}

void get_dimen( char *string, double *dimen )
{
  char *s=string, c;
  double r;

  while( *s == ' ' )  /* Leerzeichen ueberlesen */
    s++;
  if( *s == '-' || *s == '+' )
    s++;
  while( isdigit( *s ) || *s == '.' )
    s++;
  if( s == string )
  {
    print( "%s - Illegal dimension", string );
    return;
  }
  strlwr( s ), c = *s, *s = 0, r = atof( string ), *s = c;
  while( *s == ' ' )  /* Leerzeichen ueberlesen */
    s++;
  if( r == 0.0 && *s == '\0' )
    *dimen = 0.0;
  else if( strcmp( s, "in" ) == 0 )
    *dimen = r;
  else if( strcmp( s, "cm" ) == 0 )
    *dimen= r / 2.54;
  else if( strcmp( s, "pt" ) == 0 )
    *dimen= r / 72.27;
  else if( strcmp( s, "pc" ) == 0 )
    *dimen= r * 12.0 / 72.27;
  else if( strcmp( s, "bp" ) == 0 )
    *dimen= r / 72.0;
  else if( strcmp( s, "mm" ) == 0 )
    *dimen= r / 25.4;
  else if( strcmp( s, "dd" ) == 0 )
    *dimen= r * 1238.0 / 72.27;
  else if( strcmp( s, "cc" ) == 0 )
    *dimen= r * 12 * 1238.0 / 72.27;
  else if( strcmp( s, "sp" ) == 0 )
    *dimen= r / 72.27 / 65536.0;
  else
  {
    print( "%s - Illegal dimension", string );
    return;
  }
}

void get_opts( int argc )
{
  int i;
  char arg[MAX_PATH_LEN], *opt_dir, save_dvi_name[MAX_PATH_LEN];
  FILE *opt_file;
  struct RDArgs *rda;
  BPTR lock;
  struct DiskObject *dobj;

  if( ( rda = AllocDosObject( DOS_RDARGS, NULL ) ) != NULL )
  {
    rda->RDA_Source.CS_Buffer = arg;

    opt_dir = getenv( "TEXCONFIG" );
    strcpy( opt_name, ( opt_dir != NULL ) ? opt_dir : "TeX:config" );

    /* Zuerst die Optionen aus der Optionsdatei einlesen */
    if( ( opt_file = fopene( "dvi.opt", "r", opt_name, opt_name ) ) != NULL )
    {
      lock = Lock( opt_name, ACCESS_READ );
      NameFromLock( lock, opt_name, MAX_PATH_LEN ); /* Absoluter Name */
      UnLock( lock );
      while( fgets( arg, MAX_PATH_LEN - 2, opt_file ) != NULL )
      {
        if( strlen( arg ) > 1 )
        {
          arg[strlen(arg)-1] = '\0';
          quote_opts( arg, arg );
          rda->RDA_Source.CS_Length = strlen( arg );
          rda->RDA_Source.CS_CurChr = 0;
          parse_opts( rda );
        }
      }
      fclose( opt_file );
    }

    if( argc == 0 ) /* Workbenchargumente */
    {
      for( i = 1; _WBArgv[i] != NULL; i++ )
      {
        quote_opts( _WBArgv[i], arg );
        rda->RDA_Source.CS_Length = strlen( arg );
        rda->RDA_Source.CS_CurChr = 0;
        parse_opts( rda );
      }
      /* Jetzt eventuell noch Tooltype Optionen des WB Arguments */
      if( dvi_name[0] != '\0' )
      {
        strcpy( save_dvi_name, dvi_name );
        if( ( dobj = GetDiskObject( dvi_name ) ) != NULL )
        {
          for( i = 0; dobj->do_ToolTypes[i] != NULL; i++ )
          {
            quote_opts( dobj->do_ToolTypes[i], arg );
            rda->RDA_Source.CS_Length = strlen( arg );
            rda->RDA_Source.CS_CurChr = 0;
            parse_opts( rda );
          }
          FreeDiskObject( dobj );
        }
        strcpy( dvi_name, save_dvi_name );
      }
    }
    else
      parse_opts( NULL ); /* Kommandozeile */

    FreeDosObject( DOS_RDARGS, rda );
  }
}

int goto_page( int nr )
{
  if( nr == act_page || !frame_valid )
    return( FALSE );
  else
    if( ( nr > 0 ) && ( nr <= dvi_info.pages ) )
      act_page = nr;
  return( act_page == nr );
}

void halt( char *format, ... )
{
  extern void vprint( char *, va_list );
  va_list l;

  va_start( l, format );
  vprint( format, l );
  va_end( l );
  dvi_clean_amiga();
  longjmp( halt_jump, 1 );
}

void main_loop( void )
{
  int quit = FALSE, update = FALSE, newpage = dvi_name[0];
  ULONG signals, sig_mask;

  while( !quit )
  {
    if( newpage && dvi_name[0] != '\0' )
    {
      SetBusyPointer( win );
      SetBusyPointer( sbm_win );
      newpage = FALSE;

      if( !setjmp( halt_jump ) )
      {
        act_page = format_pages_amiga( act_page, act_page, 1 );

        OnMenu( win, FULLMENUNUM( 0, 1, NOSUB ) ); /* Update DVI anschalten */
        OnMenu( win, FULLMENUNUM( 1, NOITEM, NOSUB ) ); /* Screen Menu an */
        OnMenu( win, FULLMENUNUM( 2, NOITEM, NOSUB ) ); /* Printer Menue an */
        update = TRUE;
      }
      else
      {
        dvi_clean_amiga(); /* Nur zur Sicherheit */
        close_sbm_window();

        OffMenu( win, FULLMENUNUM( 0, 1, NOSUB ) ); /* Update DVI abschalten */
        OffMenu( win, FULLMENUNUM( 1, NOITEM, NOSUB ) ); /* Screen Menu aus */
        OffMenu( win, FULLMENUNUM( 2, NOITEM, NOSUB ) ); /* Printer Menue aus */
        update = FALSE;
      }

      ClearBusyPointer( win );
      ClearBusyPointer( sbm_win );
    }

    if( update && frame_valid && ( sbm_win != NULL ) )
    {
      SetBusyPointer( win );
      SetBusyPointer( sbm_win );

#ifdef SHRINK
      if( u_shrinkfactor != o_shrinkfactor )
      {
        shrink();
        LockLayerRom( sbm_win->RPort->Layer );
        SyncSBitMap( sbm_win->RPort->Layer );
        UnlockLayerRom( sbm_win->RPort->Layer );
        o_shrinkfactor = u_shrinkfactor;
      }
#endif

      sprintf( win_title, "DVI File: %s - [%d]", FilePart( (UBYTE *)dvi_name ),
        act_page );
      SetWindowTitles( sbm_win, (UBYTE *)win_title, (UBYTE *)~0 );
      update_window();
      update = FALSE;

      ClearBusyPointer( win );
      ClearBusyPointer( sbm_win );
    }

    sig_mask = 1 << win->UserPort->mp_SigBit;
    if( sbm_win != NULL )
      sig_mask |= 1 << sbm_win->UserPort->mp_SigBit;
    if( app_menu_port != NULL )
      sig_mask |= 1 << app_menu_port->mp_SigBit;

    signals = Wait( sig_mask );

    if( app_menu_port )
      if( signals & ( 1 << app_menu_port->mp_SigBit ) )
        newpage = process_wb_messages( app_menu_port );
    if( signals & ( 1 << win->UserPort->mp_SigBit ) )
      quit = process_idcmp_messages( win->UserPort, &newpage, &update );
    if( sbm_win != NULL )
      if( signals & ( 1 << sbm_win->UserPort->mp_SigBit ) )
        quit = process_idcmp_messages( sbm_win->UserPort, &newpage, &update );
  }
}

void margins_setup( void )
{
  int end = FALSE, ok = FALSE;
  struct IntuiMessage *imsg;
  struct Gadget *gad;
  struct StringInfo *str_info;
  options op_save;
  char str[MAX_PATH_LEN];
  
  memcpy( &op_save, &op, sizeof( options ) ); /* Options zwischenspeichern */

  if( OpenMarginsWindow() == 0 )
  {
    sprintf( str, "%.6lf in", op.hoffset );
    GT_SetGadgetAttrs( MarginsGadgets[GD_h_offset], MarginsWnd, NULL,
      GTST_String, (ULONG)str, TAG_DONE );
    sprintf( str, "%.6lf in", op.voffset );
    GT_SetGadgetAttrs( MarginsGadgets[GD_v_offset], MarginsWnd, NULL,
      GTST_String, (ULONG)str, TAG_DONE );
    sprintf( str, "%.6lf in", op.hspread );
    GT_SetGadgetAttrs( MarginsGadgets[GD_h_spread], MarginsWnd, NULL,
      GTST_String, (ULONG)str, TAG_DONE );
    sprintf( str, "%.6lf in", op.vspread );
    GT_SetGadgetAttrs( MarginsGadgets[GD_v_spread], MarginsWnd, NULL,
      GTST_String, (ULONG)str, TAG_DONE );
    sprintf( str, "%.6lf in", op.hmargin );
    GT_SetGadgetAttrs( MarginsGadgets[GD_h_margin], MarginsWnd, NULL,
      GTST_String, (ULONG)str, TAG_DONE );
    sprintf( str, "%.6lf in", op.vmargin );
    GT_SetGadgetAttrs( MarginsGadgets[GD_v_margin], MarginsWnd, NULL,
      GTST_String, (ULONG)str, TAG_DONE );
    sprintf( str, "%.6lf in", op.width );
    GT_SetGadgetAttrs( MarginsGadgets[GD_width], MarginsWnd, NULL,
      GTST_String, (ULONG)str, TAG_DONE );
    sprintf( str, "%.6lf in", op.height );
    GT_SetGadgetAttrs( MarginsGadgets[GD_height], MarginsWnd, NULL,
      GTST_String, (ULONG)str, TAG_DONE );

    ActivateGadget( MarginsGadgets[GD_h_offset], MarginsWnd, NULL );

    while( !end )
    {
      WaitPort( MarginsWnd->UserPort );
      while( imsg = GT_GetIMsg( MarginsWnd->UserPort ) )
      {
        gad = imsg->IAddress;

        switch( imsg->Class )
        {
          case CLOSEWINDOW:
            end = TRUE;
            ok = FALSE;
            break;
          case GADGETUP:
            switch( gad->GadgetID )
            {
              case GD_marg_ok:
                end = TRUE;
                ok = TRUE;
                break;
              case GD_marg_cancel:
                end = TRUE;
                ok = FALSE;
                break;
              default:
                break;
            }
            break;
          default:
            break;
        }
        GT_ReplyIMsg( imsg );
      }
    }
    str_info = MarginsGadgets[GD_h_offset]->SpecialInfo;
    get_dimen( str_info->Buffer, &op.hoffset );
    str_info = MarginsGadgets[GD_v_offset]->SpecialInfo;
    get_dimen( str_info->Buffer, &op.voffset );
    str_info = MarginsGadgets[GD_h_spread]->SpecialInfo;
    get_dimen( str_info->Buffer, &op.hspread );
    str_info = MarginsGadgets[GD_v_spread]->SpecialInfo;
    get_dimen( str_info->Buffer, &op.vspread );
    str_info = MarginsGadgets[GD_h_margin]->SpecialInfo;
    get_dimen( str_info->Buffer, &op.hmargin );
    str_info = MarginsGadgets[GD_v_margin]->SpecialInfo;
    get_dimen( str_info->Buffer, &op.vmargin );
    str_info = MarginsGadgets[GD_width]->SpecialInfo;
    get_dimen( str_info->Buffer, &op.width );
    str_info = MarginsGadgets[GD_height]->SpecialInfo;
    get_dimen( str_info->Buffer, &op.height );

    CloseMarginsWindow();
  }

  if( !ok )
    memcpy( &op, &op_save, sizeof( options ) );
}

int open_all( void )
{
  char out_name[64];

  if( open_screen() )
  {
    if( open_window() )
    {
      sprintf( out_name, "CON://///WINDOW %x", win );
      inout_handle = Open( out_name, MODE_OLDFILE );

      if( BADDR( inout_handle ) != NULL )
      {
        old_in_handle = SelectInput( inout_handle );

        console_window = process->pr_WindowPtr;
        console_task = process->pr_ConsoleTask;

        process->pr_WindowPtr = win; /* aktuelles Window fuer Systemrequester */
        process->pr_ConsoleTask = ( (struct FileHandle *)BADDR( inout_handle ) )->fh_Type;

        if( app_menu_port = CreateMsgPort() )
        {
          app_menu_item = AddAppMenuItem( 1, NULL, "DVI", app_menu_port, TAG_DONE );

          if( open_menu() )
          {
            if( ( PSbase = (struct Library *)OpenLibrary( "post.library", 17 ) ) == NULL )
              print( "Warning: unable to open post.library 17+" );
            return( TRUE );
          }
          else
            print( "Unable to create menu!" );
        }
        else
          print( "Unable to create message port!" );
      }
      else
        print( "Unable to attach console to control window!" );
    }
    else
      print( "Unable to open window!" );
  }
  else
    print( "Unable to open screen!" );

  return( FALSE );
}

int open_menu( void )
{
  if( mnu = CreateMenus( new_menu, GTMN_NewLookMenus, (ULONG)TRUE, TAG_DONE ) )
  {
    LayoutMenus( mnu, vis, GTMN_NewLookMenus, (ULONG)TRUE, TAG_DONE );
    SetMenuStrip( win, mnu );
    GT_RefreshWindow( win, NULL );
    if( AslBase->lib_Version < 38 )
      OffMenu( win, FULLMENUNUM( 3, 5, NOSUB ) ); /* Screenmode... abschalten */
  }
  return( mnu != NULL );
}

int open_screen( void )
{
  struct Screen *default_screen;
  struct DrawInfo *draw_info;
  UWORD pen = (UWORD)~0;
  ULONG displayid;
  struct NameInfo ninfo;

  if( aop.screenmode[0] == '\0' )
  {
    if( default_screen = LockPubScreen( NULL ) )
    {
      draw_info = GetScreenDrawInfo( default_screen );
      aop.displayid = displayid = GetVPModeID( &(default_screen->ViewPort) );

      if( scr = OpenScreenTags( NULL,
          SA_Title, (ULONG)scr_title,
          SA_DisplayID, displayid,
          SA_Depth, draw_info ? MIN( (ULONG)DEPTH, (ULONG)( draw_info->dri_Depth ) ) : (ULONG)DEPTH,
          SA_Pens, draw_info ? (ULONG)( draw_info->dri_Pens ) : (ULONG)&pen,
          TAG_DONE ) )
        vis = GetVisualInfo( scr, TAG_DONE );

      if( draw_info )
        FreeScreenDrawInfo( default_screen, draw_info );
      UnlockPubScreen( NULL, default_screen );
    }
  }
  else
  {
    if ( aop.displayid == INVALID_ID )
    {
      for( displayid = NextDisplayInfo( INVALID_ID ); displayid != INVALID_ID;
          displayid = NextDisplayInfo( displayid ) )
      {
        if( GetDisplayInfoData( NULL, (UBYTE *)&ninfo, sizeof( ninfo ), DTAG_NAME,
            displayid ) )
          if( strcmp( ninfo.Name, aop.screenmode ) == 0 )
            break;
      }
    }
    else
      displayid = aop.displayid;

    if( displayid != INVALID_ID )
    {
      aop.displayid = displayid;
      if( scr = OpenScreenTags( NULL,
          SA_Title, (ULONG)scr_title,
          SA_DisplayID, displayid,
          SA_Depth, ( aop.depth > 0 ) ? MIN( (ULONG)DEPTH, (ULONG)aop.depth ) : (ULONG)DEPTH,
          SA_Pens, (ULONG)&pen,
          TAG_DONE ) )
        vis = GetVisualInfo( scr, TAG_DONE );
    }
    else
      print( "Invalid Screenmode: \"%s\"", aop.screenmode );
  }

  return( scr && vis );
}

int open_window( void )
{
  win = OpenWindowTags( NULL, WA_Top, (ULONG)( scr->BarHeight + 1 ),
    WA_Left, (ULONG)0, WA_Width, (ULONG)scr->Width,
    WA_Height, 130, /* analog zum Shellfenster */
    WA_MinHeight, (ULONG)50, /* analog zum Shellfenster */
    WA_MinWidth, (ULONG)490, /* analog zum Shellfenster */
    WA_MaxHeight, (ULONG)scr->Height,
    WA_MaxWidth, (ULONG)scr->Width,
    WA_Title, (ULONG)"Control",
    WA_Flags, WFLG_ACTIVATE | WFLG_NOCAREREFRESH | WFLG_SMART_REFRESH,
    WA_CloseGadget, (ULONG)TRUE,
    WA_DepthGadget, (ULONG)TRUE,
    WA_DragBar, (ULONG)TRUE,
    WA_SizeGadget, (ULONG)TRUE,
    WA_NewLookMenus, (ULONG)TRUE,
    WA_IDCMP, IDCMP_VANILLAKEY | IDCMP_RAWKEY | IDCMP_MENUPICK |
        IDCMP_CLOSEWINDOW,
    WA_PubScreen, (ULONG)scr,
    TAG_DONE );
  return( win != NULL );
}

void out_newline( void )
{
  if( BADDR( inout_handle ) != NULL )
    FPutC( inout_handle, '\n' );
  else
    fputc( '\n', stdout );
}

void out_string( char *s )
{
  if( BADDR( inout_handle ) != NULL )
  {
    FPuts( inout_handle, s );
    Flush( inout_handle );
  }
  else
  {
    fputs( s, stdout );
    fflush( stdout );
  }
}

void parse_opts( struct RDArgs *parse )
{
  char *template = "DVIFile,CallMF/K,Copies/N,Density/K,Eject/T,Height/K,HMargin/K,HOffset/K,HResolution/N,HSpread/K,IMGPath/K,Landscape/T,Logfile/K,Magnification/N,Memory/T,Output/K,Path/K,PathMem/N,Pictures/T,PixMem/N,PKPath/K,Redirect/K,ScreenMode/K,Separate/T,ShowFonts/T,SingleSheet/T,TFMPath/K,Thinout/T,TraceChars/T,TraceMem/T,VFPath/K,VMargin/K,VOffset/K,VResolution/N,VSpread/K,Width/K";
  char dummy[MAX_PATH_LEN], *str;
  struct RDArgs *rda;
  LONG arg[36];
  int i;
  long j;

  for( i = 0; i < 36; i++ )
    arg[i] = 0;

  if( ( rda = ReadArgs( template, arg, parse ) ) != NULL )
  {
    if( arg[0] )
    {
      strcpy( dvi_name, (char *)arg[0] );
    }
    if( arg[1] )
    {
      strcpy( op.pk_callmf, (char *)arg[1] );
    }
    if( arg[2] )
    {
      op.copies = MAX( 1, *(LONG *)arg[2] );
    }
    if( arg[3] )
    {
      op.density = atof( (char *)arg[3] );
      op.density = MIN( MAX( 0.0, op.density ), 1.0 );
    }
    if( arg[4] )
    {
      op.formfeed = TRUE;
      new_menu[FIRST_OPTIONS].nm_Flags |= CHECKED;
    }
    if( arg[5] )
    {
      get_dimen( (char *)arg[5], &op.height );
    }
    if( arg[6] )
    {
      get_dimen( (char *)arg[6], &op.hmargin );
    }
    if( arg[7] )
    {
      get_dimen( (char *)arg[7], &op.hoffset );
    }
    if( arg[8] )
    {
      if( ( j = *(LONG *)arg[8] ) > 0 )
        op.hres = j;
    }
    if( arg[9] )
    {
      get_dimen( (char *)arg[9], &op.hspread );
    }
    if( arg[10] )
    {
      strcpy( op.img_path, (char *)arg[10] );
    }
    if( arg[11] )
    {
      op.landscape = TRUE;
      new_menu[FIRST_OPTIONS+1].nm_Flags |= CHECKED;
    }
    if( arg[12] )
    {
      strcpy( op.log_name, (char *)arg[12] );
    }
    if( arg[13] )
    {
      if( ( j = *(LONG *)arg[13] ) > 0 )
        op.new_mag = j;
    }
    if( arg[14] )
    {
      op.dvimemory = TRUE;
      new_menu[FIRST_OPTIONS+2].nm_Flags |= CHECKED;
    }
    if( arg[15] )
    {
      strcpy( dummy, (char *)arg[15] );
      for( i = 0; dummy[i] != '\0'; i++ )
        dummy[i] = toupper( dummy[i] );

      if( strcmp( dummy, "P6LOW" ) == 0 )
        op.shipno = P6LOW;
      else if( strcmp( dummy, "P6HIGH" ) == 0 )
        op.shipno = P6HIGH;
      else if( strcmp( dummy, "P6MID" ) == 0 )
        op.shipno = P6MID;
      else if( strcmp( dummy, "FX80" ) == 0 )
        op.shipno = FX80;
      else if( strcmp( dummy, "HPLJ" ) == 0 )
        op.shipno = HPLJ;
      else if( strcmp( dummy, "HPLJLOW" ) == 0 )
        op.shipno = HPLJLOW;
      else if( strcmp( dummy, "BJ300" ) == 0 )
        op.shipno = BJ300;
      else
        op.shipno = NO_PRINTER;
    }
    if( arg[16] )
    {
      strcpy( op.dvi_path, (char *)arg[16] );
    }
    if( arg[17] )
    {
      if( ( j = *(LONG *)arg[17] ) > 0 )
        op.pathmem = j;
    }
    if( arg[18] )
    {
      op.show_img = TRUE;
      new_menu[FIRST_OPTIONS+3].nm_Flags |= CHECKED;
    }
    if( arg[19] )
    {
      if( ( j = *(LONG *)arg[19] ) > 0 )
        op.pixmem = j;
    }
    if( arg[20] )
    {
      strcpy( op.pk_path, (char *)arg[20] );
    }
    if( arg[21] )
    {
      strcpy( aop.redir_name, (char *)arg[21] );
      aop.redirect = ( aop.redir_name[0] != '\0' );
    }
    if( arg[22] )
    {
      strcpy( dummy, (char *)arg[22] );
      if( str = strchr( dummy, ',' ) )
      {
        *str = '\0';
        aop.depth = atoi( ++str );
        for( j = 0, i = 1; i < aop.depth; j++ )
          i *= 2;
        aop.depth = i;
      }
      strcpy( aop.screenmode, dummy );
    }
    if( arg[23] )
    {
      op.separate = TRUE;
      new_menu[FIRST_OPTIONS+4].nm_Flags |= CHECKED;
      op.thin_out = FALSE;
      new_menu[FIRST_OPTIONS+7].nm_Flags &= ~CHECKED;
    }
    if( arg[24] )
    {
      op.showfonts = TRUE;
      new_menu[FIRST_OPTIONS+5].nm_Flags |= CHECKED;
    }
    if( arg[25] )
    {
      op.singlesheet = TRUE;
      new_menu[FIRST_OPTIONS+6].nm_Flags |= CHECKED;
    }
    if( arg[26] )
    {
      strcpy( op.tfm_path, (char *)arg[26] );
    }
    if( arg[27] )
    {
      op.thin_out = TRUE;
      new_menu[FIRST_OPTIONS+7].nm_Flags |= CHECKED;
      op.separate = FALSE;
      new_menu[FIRST_OPTIONS+4].nm_Flags &= ~CHECKED;
    }
    if( arg[28] )
    {
      op.tracechars = TRUE;
      new_menu[FIRST_OPTIONS+8].nm_Flags |= CHECKED;
    }
    if( arg[29] )
    {
      op.tracemem = TRUE;
      new_menu[FIRST_OPTIONS+9].nm_Flags |= CHECKED;
    }
    if( arg[30] )
    {
      strcpy( op.vf_path, (char *)arg[30] );
    }
    if( arg[31] )
    {
      get_dimen( (char *)arg[31], &op.vmargin );
    }
    if( arg[32] )
    {
      get_dimen( (char *)arg[32], &op.voffset );
    }
    if( arg[33] )
    {
      if( ( j = *(LONG *)arg[33] ) > 0 )
        op.vres = j;
    }
    if( arg[34] )
    {
      get_dimen( (char *)arg[34], &op.vspread );
    }
    if( arg[35] )
    {
      get_dimen( (char *)arg[35], &op.width );
    }

    FreeArgs( rda );
  }
  else if( parse == NULL ) /* Kommandozeile fehlerhaft */
    PrintFault( IoErr(), "" );
}

void print_pages( int start, int end, int step )
{
  options op_save;
  ami_options aop_save;
  int old_valid = frame_valid;
  struct MsgPort *prt_port;
  
  memcpy( &op_save, &op, sizeof( options ) ); /* Options zwischenspeichern */
  memcpy( &aop_save, &aop, sizeof( ami_options ) );

  op.hres = p_defs[op.shipno].hres; /* Optionen an Drucker anpassen */
  op.vres = p_defs[op.shipno].vres;
  shipout = p_defs[op.shipno].shipout;
  op.pixmem = aop.pixmem; /* Wieviel Speicher f"ur die Bitmap? */

  if( shipout && frame_valid )
  {
    if( aop.redirect )
    {
      if( aop.redir_name[0] == '\0' )
        strcpy( op.redirect, "output.prt" );
      else
        strcpy( op.redirect, aop.redir_name );

      SetBusyPointer( win );
      ClearMenuStrip( win ); /* Menues aus */

      dvi_clean_amiga();

      if( !setjmp( halt_jump ) )
        format_pages( start, end, step );

      ResetMenuStrip( win, mnu ); /* Menues an */
    }
    else
    {
      if( prt_port = CreateMsgPort() )
      {
        if( prt_req = (struct IOStdReq*)CreateIORequest( prt_port, sizeof( struct IOStdReq ) ) )
        {
          if( OpenDevice( "printer.device", 0, (struct IORequest*)prt_req, 0 ) == 0 )
          {
            SetBusyPointer( win );
            ClearMenuStrip( win ); /* Menues aus */

            dvi_clean_amiga();

            if( !setjmp( halt_jump ) )
              format_pages( start, end, step );

            if( prt_req->io_Error == PDERR_NOERR )
            {
              prt_req->io_Command = PRD_RAWWRITE;
              prt_req->io_Data = buffer;
              prt_req->io_Length = count;

              DoIO( (struct IORequest*)prt_req ); /* Druckerpuffer leeren */
            }
            count = 0; /* Keine Daten mehr im Druckerpuffer */

            ResetMenuStrip( win, mnu ); /* Menues an */
            CloseDevice( (struct IORequest*)prt_req );
          }
          else
            print( "Unable to open printer device!" );
          DeleteIORequest( (struct IORequest*)prt_req );
        }
        else
          print( "Unable to create IORequest!" );
        DeleteMsgPort( prt_port );
      }
      else
        print( "Unable to open printer port!" );
    }
  }

  memcpy( &aop, &aop_save, sizeof( ami_options ) ); /* Options restaurieren */
  memcpy( &op, &op_save, sizeof( options ) );

  /* Dieser Teil entfllt bei SuperBitmap Window, shipout = NULL nach oben */
  if( shipout )
  {
    shipout = NULL;
    if( old_valid /* && !setjmp( halt_jump ) */ )
    {
      dvi_clean_amiga();
      format_pages_amiga( act_page, act_page, 1 );
    }
  }
  ClearBusyPointer( win );
}

void print_range( void )
{
  static int first = 1, last = 1, step = 1;

  if( request_range( &first, &last, &step ) )
    print_pages( first, last, step );
}

void prbyte( int c )
{
  if( red_fp == NULL || !aop.redirect )
  {
    if( count < BUFSIZE )
      buffer[count++] = c;
    else
    {
      prt_req->io_Command = PRD_RAWWRITE;
      prt_req->io_Data = buffer;
      prt_req->io_Length = BUFSIZE;

      DoIO( (struct IORequest*)prt_req );

      if( prt_req->io_Error == PDERR_CANCEL )
        halt( "printing aborted!" );

      count = 0;
      buffer[count++] = c;
    }
  }
  else
  {
    if( fputc( c, red_fp ) == EOF )
      halt( "writing to redirection file %s aborted!", op.redirect );
  }
}

void printer_setup( void )
{
  int end = FALSE, ok = FALSE;
  struct IntuiMessage *imsg;
  struct Gadget *gad;
  struct StringInfo *str_info;
  options op_save;
  ami_options aop_save;
  
  memcpy( &op_save, &op, sizeof( options ) ); /* Options zwischenspeichern */
  memcpy( &aop_save, &aop, sizeof( ami_options ) );

  if( OpenPrinter_SetupWindow() == 0 )
  {
    GT_SetGadgetAttrs( Printer_SetupGadgets[GD_printer], Printer_SetupWnd, NULL,
      GTMX_Active, (ULONG)( op.shipno + 1 ), TAG_DONE );
    GT_SetGadgetAttrs( Printer_SetupGadgets[GD_redirect], Printer_SetupWnd, NULL,
      GTMX_Active, (ULONG)( aop.redirect ), TAG_DONE );
    GT_SetGadgetAttrs( Printer_SetupGadgets[GD_redir_file], Printer_SetupWnd, NULL,
      GTST_String, (ULONG)( aop.redir_name ), GA_Disabled, !aop.redirect, TAG_DONE );
    GT_SetGadgetAttrs( Printer_SetupGadgets[GD_copies], Printer_SetupWnd, NULL,
      GTIN_Number, (ULONG)( op.copies ), TAG_DONE );
    GT_SetGadgetAttrs( Printer_SetupGadgets[GD_mag], Printer_SetupWnd, NULL,
      GTIN_Number, (ULONG)( aop.prt_mag ), TAG_DONE );

    ActivateGadget( Printer_SetupGadgets[GD_copies], Printer_SetupWnd, NULL );

    while( !end )
    {
      WaitPort( Printer_SetupWnd->UserPort );
      while( imsg = GT_GetIMsg( Printer_SetupWnd->UserPort ) )
      {
        gad = imsg->IAddress;

        switch( imsg->Class )
        {
          case CLOSEWINDOW:
            end = TRUE;
            ok = FALSE;
            break;
          case GADGETDOWN:
            switch( gad->GadgetID )
            {
              case GD_printer:
                op.shipno = imsg->Code - 1;
                break;
              case GD_redirect:
                aop.redirect = imsg->Code;
                GT_SetGadgetAttrs( Printer_SetupGadgets[GD_redir_file],
                  Printer_SetupWnd, NULL, GA_Disabled, !aop.redirect, TAG_DONE );
                break;
              default:
                break;
            }
            break;
          case GADGETUP:
            switch( gad->GadgetID )
            {
              case GD_psetup_ok:
                end = TRUE;
                ok = TRUE;
                break;
              case GD_psetup_cancel:
                end = TRUE;
                ok = FALSE;
                break;
              default:
                break;
            }
            break;
          default:
            break;
        }
        GT_ReplyIMsg( imsg );
      }
    }

    str_info = Printer_SetupGadgets[GD_redir_file]->SpecialInfo;
    strcpy( aop.redir_name, (char *)str_info->Buffer );

    str_info = Printer_SetupGadgets[GD_copies]->SpecialInfo;
    op.copies = str_info->LongInt;

    str_info = Printer_SetupGadgets[GD_mag]->SpecialInfo;
    op.new_mag = str_info->LongInt;

    ClosePrinter_SetupWindow();
  }

  if( !ok )
  {
    memcpy( &aop, &aop_save, sizeof( ami_options ) ); /* Options restaurieren */
    memcpy( &op, &op_save, sizeof( options ) );
  }
}

int process_idcmp_messages( struct MsgPort *msgport, int *newpage, int *update )
{
  static int dx = 0, dy = 0;
  int gopage = 1, newmag = 0, val;
  struct IntuiMessage *imsg, cpy_msg;
  static ULONG micros = 0, seconds = 0;

  if( msgport != NULL )
    while( imsg = GT_GetIMsg( msgport ) )
    {
      cpy_msg = *imsg;
      GT_ReplyIMsg( imsg );

      switch( cpy_msg.Class )
      {
        case IDCMP_CLOSEWINDOW:
          return( TRUE );
          break;
        case IDCMP_VANILLAKEY:
          switch( cpy_msg.Code )
          {
            case '\x0D':
              *newpage = goto_page( act_page + 1 );
              if( cpy_msg.Qualifier & SHIFT_QUALIFIER )
              {
                dx = -sbm_win->RPort->Layer->Scroll_X;
                dy = -sbm_win->RPort->Layer->Scroll_Y;
              }
              break;
            case '+':
#ifdef SHRINK
              if( frame_valid )
              {
                if( u_shrinkfactor < NUMTICKS )
                  u_shrinkfactor++;
                if( u_shrinkfactor != o_shrinkfactor )
                  *update = TRUE;
              }
#else
              *newpage = goto_page( act_page + 1 );
#endif
              break;
            case '\x08':
              *newpage = goto_page( act_page - 1 );
              if( cpy_msg.Qualifier & SHIFT_QUALIFIER )
              {
                dx = -sbm_win->RPort->Layer->Scroll_X;
                dy = -sbm_win->RPort->Layer->Scroll_Y;
              } 
              break;
            case '-':
#ifdef SHRINK
              if( frame_valid )
              {
                if( u_shrinkfactor > 1 )
                  u_shrinkfactor--;
                if( u_shrinkfactor != o_shrinkfactor )
                  *update = TRUE;
              }
#else
              *newpage = goto_page( act_page - 1 );
#endif
              break;
            default:
/*
              print( "VANILLAKEY: %d", cpy_msg.Code );
*/
              break;
          }
          if( dx || dy )
          {
            ScrollLayer( 0, sbm_win->RPort->Layer, dx, dy );
            dx = dy = 0;
          }
          break;
        case IDCMP_RAWKEY:
          switch( cpy_msg.Code )
          {
            case CRSR_UP:
              dx = 0;
              if( ( sbm_win->RPort->Layer->Scroll_Y - PIX_UPDOWN >=  0 ) &&
                  !( cpy_msg.Qualifier & ( SHIFT_QUALIFIER | ALT_QUALIFIER ) ) )
              {
                dy = -PIX_UPDOWN;
              }
              else
              {
                dy = -sbm_win->RPort->Layer->Scroll_Y;
                if( cpy_msg.Qualifier & ALT_QUALIFIER )
                  dy = MAX( dy, -sbm_win->GZZHeight * 3 / 4 );
              }
              break;
            case CRSR_DOWN:
              dx = 0;
              if( ( sbm_win->RPort->Layer->Scroll_Y + PIX_UPDOWN +
                  sbm_win->GZZHeight <= bmp_height ) &&
                  !( cpy_msg.Qualifier & ( SHIFT_QUALIFIER | ALT_QUALIFIER ) ) )
              {
                dy = PIX_UPDOWN;
              }
              else
              {
                dy = MAX( 0, bmp_height - sbm_win->GZZHeight - 
                  sbm_win->RPort->Layer->Scroll_Y );
                if( cpy_msg.Qualifier & ALT_QUALIFIER )
                  dy = MIN( dy, sbm_win->GZZHeight * 3 / 4 );
              }
              break;
            case CRSR_RIGHT:
              dy = 0;
              if( ( sbm_win->RPort->Layer->Scroll_X + PIX_LEFTRIGHT +
                  sbm_win->GZZWidth <= bmp_width ) &&
                  !( cpy_msg.Qualifier & ( SHIFT_QUALIFIER | ALT_QUALIFIER ) ) )
              {
                dx = PIX_LEFTRIGHT;
              }
              else
              {
                dx = MAX( 0, bmp_width - sbm_win->GZZWidth - 
                  sbm_win->RPort->Layer->Scroll_X );
                if( cpy_msg.Qualifier & ALT_QUALIFIER )
                  dx = MIN( dx, sbm_win->GZZWidth * 3 / 4 );
              }
              break;
            case CRSR_LEFT:
              dy = 0;
              if( ( sbm_win->RPort->Layer->Scroll_X - PIX_LEFTRIGHT >= 0 ) &&
                  !( cpy_msg.Qualifier & ( SHIFT_QUALIFIER | ALT_QUALIFIER ) ) )
              {
                dx = -PIX_LEFTRIGHT;
              }
              else
              {
                dx = -sbm_win->RPort->Layer->Scroll_X;
                if( cpy_msg.Qualifier & ALT_QUALIFIER )
                  dx = MAX( dx, -sbm_win->GZZWidth * 3 / 4 );
              }
              break;
            default:
              dx = dy = 0;
/*
              print( "RAWKEY: %d", cpy_msg.Code );
*/
              break;
          }
          if( dx || dy )
          {
            ScrollLayer( 0, sbm_win->RPort->Layer, dx, dy );
            dx = dy = 0;
          }
          break;
        case IDCMP_MOUSEBUTTONS:
          switch( cpy_msg.Code )
          {
            case SELECTDOWN:
              if( DoubleClick( seconds, micros, cpy_msg.Seconds, cpy_msg.Micros ) )
                *newpage = goto_page( act_page + 1 );
              seconds = cpy_msg.Seconds;
              micros = cpy_msg.Micros;
              ModifyIDCMP( sbm_win, sbm_win->IDCMPFlags | IDCMP_DELTAMOVE |
                IDCMP_MOUSEMOVE );
              break;
            case SELECTUP:
              ModifyIDCMP( sbm_win, sbm_win->IDCMPFlags &
                ~( IDCMP_DELTAMOVE ^ IDCMP_MOUSEMOVE ) );
              break;
            default:
              break;
          }
          break;
        case IDCMP_MOUSEMOVE:
          dx += cpy_msg.MouseX;
          dy += cpy_msg.MouseY;
          if( ABS( dx ) > PIX_LEFTRIGHT )
          {
            if( dx < 0 && sbm_win->RPort->Layer->Scroll_X - dx +
                sbm_win->GZZWidth > bmp_width )
            {
              dx = -MAX( 0,
                bmp_width - sbm_win->GZZWidth - sbm_win->RPort->Layer->Scroll_X );
            }
            else if( dx > 0 && sbm_win->RPort->Layer->Scroll_X - dx < 0 )
            {
              dx = sbm_win->RPort->Layer->Scroll_X;
            }
            if( dx )
            {
              ScrollLayer( 0, sbm_win->RPort->Layer, -dx, 0 );
              dx = 0;
            }
          }
          if( ABS( dy ) > PIX_UPDOWN )
          {
            if( dy < 0 && sbm_win->RPort->Layer->Scroll_Y - dy +
                sbm_win->GZZHeight > bmp_height )
            {
              dy = -MAX( 0,
                bmp_height - sbm_win->GZZHeight - sbm_win->RPort->Layer->Scroll_Y );
            }
            else if( dy > 0 && sbm_win->RPort->Layer->Scroll_Y - dy < 0 )
            {
              dy = sbm_win->RPort->Layer->Scroll_Y;
            }
            if( dy )
            {
              ScrollLayer( 0, sbm_win->RPort->Layer, 0, -dy );
              dy = 0;
            }
          }
          break;
        case IDCMP_MENUPICK:
          switch( MENUNUM( cpy_msg.Code ) )
          {
            case 0: /* Project Menu */
              switch( ITEMNUM( cpy_msg.Code ) )
              {
                case 0: /* Load DVI */
                  *newpage = dvi_filereq();
                  return( FALSE );
                  break;
                case 1: /* update DVI */
                  if( frame_valid )
                  {
                    *newpage = TRUE;
                    dvi_clean_amiga();
                    return( FALSE );
                  }
                  break;
                case 2: /* Save Options */
                  write_opts();
                  break;
                case 4: /* About */
                  about_dvi();
                  break;
                case 6: /* Quit */
                  return( TRUE );
                  break;
                default:
                  break;
              }
              break;
            case 1: /* Screen Menu */
              switch( ITEMNUM( cpy_msg.Code ) )
              {
                case 0: /* Next Page */
                  *newpage = goto_page( act_page + 1 );
                  break;
                case 1: /* Previous Page */
                  *newpage = goto_page( act_page - 1 );
                  break;
                case 2: /* Goto Page */
                  if( request_integer( &gopage, "Goto Page" ) )
                    *newpage = goto_page( gopage );
                  break;
                case 4: /* Magnification */
                  if( request_integer( &newmag, "Magnification" ) )
                  {
                    if( op.new_mag != newmag )
                    {
                      op.new_mag = newmag;
                      dvi_clean_amiga();
                      *newpage = TRUE;
                      return( FALSE );
                    }
                  }
                  break;
                default:
                  break;
              }
              break;
            case 2: /* Printer Menu */
              switch( ITEMNUM( cpy_msg.Code ) )
              {
                case 0: /* Print Range */
                  print_range();
                  break;
                case 1: /* Print Page */
                  print_pages( act_page, act_page, 1 );
                  break;
                case 2: /* Print Document */
                  print_pages( 1, dvi_info.pages, 1 );
                  break;
                default:
                  break;
              }
              *update = TRUE;
              return( FALSE );
            case 3: /* Setup Menu */
              switch( ITEMNUM( cpy_msg.Code ) )
              {
                case 0: /* Directories */
                  dir_setup();
                  break;
                case 1: /* Margins */
                  margins_setup();
                  break;
                case 2: /* Options */
                  switch( SUBNUM( cpy_msg.Code ) )
                  {
                    case 0: /* Eject */
                      op.formfeed = !op.formfeed;
                      break;
                    case 1: /* Landscape */
                      op.landscape = !op.landscape;
                      if( frame_valid )
                      {
                        *newpage = TRUE;
                        dvi_clean_amiga();
                        return( FALSE );
                      }
                      break;
                    case 2: /* Memory */
                      op.dvimemory = !op.dvimemory;
                      if( frame_valid )
                      {
                        *newpage = TRUE;
                        dvi_clean_amiga();
                        return( FALSE );
                      }
                      break;
                    case 3: /* Pictures */
                      op.show_img = !op.show_img;
                      break;
                    case 4: /* Separate */
                      op.separate = !op.separate;
                      if( op.separate )
                        op.thin_out = FALSE;
                      break;
                    case 5: /* Showfonts */
                      op.showfonts = !op.showfonts;
                      break;
                    case 6: /* Singlesheet */
                      op.singlesheet = !op.singlesheet;
                      break;
                    case 7: /* Thinout */
                      op.thin_out = !op.thin_out;
                      if( op.thin_out )
                        op.separate = FALSE;
                      break;
                    case 8: /* Tracechars */
                      op.tracechars = !op.tracechars;
                      if( frame_valid )
                      {
                        *newpage = TRUE;
                        dvi_clean_amiga();
                        return( FALSE );
                      }
                      break;
                    case 9: /* Tracemem */
                      op.tracemem = !op.tracemem;
                      if( frame_valid )
                      {
                        *newpage = TRUE;
                        dvi_clean_amiga();
                        return( FALSE );
                      }
                      break;
                    default:
                      break;
                  }
                  break;
                case 3: /* Printer */
                  printer_setup();
                  break;
                case 5: /* Screenmode */
                  if( ( val = dvi_screenmode() ) >= 0 )
                  {
                    *newpage = ( val > 0 );
                    return( FALSE );
                  }
                  else
                    return( TRUE );
                  break;
                default:
                  break;
              }
            default:
              break;
          }
          break;
        default:
          break;
      }
    }
  return( FALSE );
}

int process_wb_messages( struct MsgPort *app_menu_port )
{
  struct AppMessage *app_message;
  BPTR dir_lock, file_lock;
  int newpage = FALSE;

  while( app_message = (struct AppMessage *)GetMsg( app_menu_port ) )
  {
    if( ( app_message->am_Type == 9 ) && ( app_message->am_ID == 1 ) &&
        ( app_message->am_NumArgs > 0 ) )
    {
      if( dir_lock = DupLock( app_message->am_ArgList->wa_Lock ) )
      {
        dir_lock = CurrentDir( dir_lock );
        if( file_lock = Lock( app_message->am_ArgList->wa_Name, ACCESS_READ ) )
        {
          UnLock( file_lock );

          strcpy( dvi_name, app_message->am_ArgList->wa_Name );
          ScreenToFront( scr );
          newpage = TRUE;

          dvi_clean_amiga();
          act_page = 1;
        }
        else
          dir_lock = CurrentDir( dir_lock );
        UnLock( dir_lock );
      }
    }
    ReplyMsg( (struct Message *)app_message );
  }
  return( newpage );
}

void quote_opts( char *opt, char *quoted )
{
  char *str, temp[MAX_PATH_LEN];

  if( ( str = strchr( opt, '=' ) ) == NULL )
    sprintf( temp, "\"%s\"\n", opt );
  else
  {
    strncpy( temp, opt, str - opt ), temp[str-opt] = '\0';
    strcat( temp, "=\"" );
    strcat( temp, str + 1 );
    strcat( temp, "\"\n" );
  }
  strcpy( quoted, temp );
}

int request_integer( int *value, char *txt )
{
  struct IntuiMessage *imsg;
  struct Gadget *gad;
  struct StringInfo *str_info;
  int end = FALSE, ok = FALSE;

  if( OpenRequest_IntegerWindow() == 0 )
  {
    GT_SetGadgetAttrs( Request_IntegerGadgets[GD_reqint_txt], Request_IntegerWnd, NULL,
      GTTX_Text, (ULONG)txt, TAG_DONE );
    GT_SetGadgetAttrs( Request_IntegerGadgets[GD_reqint_int], Request_IntegerWnd, NULL,
      GTIN_Number, (ULONG)*value, TAG_DONE );

    SetWindowTitles( Request_IntegerWnd, (UBYTE *)txt, (UBYTE *)~0 );
    ActivateGadget( Request_IntegerGadgets[GD_reqint_int], Request_IntegerWnd, NULL );

    while( !end )
    {
      WaitPort( Request_IntegerWnd->UserPort );
      while( imsg = GT_GetIMsg( Request_IntegerWnd->UserPort ) )
      {
        gad = imsg->IAddress;

        switch( imsg->Class )
        {
          case CLOSEWINDOW:
            end = TRUE;
            ok = FALSE;
            break;
          case GADGETUP:
            switch( gad->GadgetID )
            {
              case GD_reqint_int:
              case GD_reqint_ok:
                end = TRUE;
                ok = TRUE;
                break;
              case GD_reqint_cancel:
                end = TRUE;
                ok = FALSE;
                break;
              default:
                break;
            }
            break;
          default:
            break;
        }
        GT_ReplyIMsg( imsg );
      }
    }
    str_info = Request_IntegerGadgets[GD_reqint_int]->SpecialInfo;
    *value = str_info->LongInt;

    CloseRequest_IntegerWindow();
  }

  return( ok );
}

int request_range( int *first, int *last, int *step )
{
  int end = FALSE, print = FALSE;
  struct IntuiMessage *imsg;
  struct Gadget *gad;
  struct StringInfo *str_info;

  if( OpenSelect_RangeWindow() == 0 )
  {
    GT_SetGadgetAttrs( Select_RangeGadgets[GD_from], Select_RangeWnd, NULL,
      GTIN_Number, (ULONG)*first, TAG_DONE );
    GT_SetGadgetAttrs( Select_RangeGadgets[GD_to], Select_RangeWnd, NULL,
      GTIN_Number, (ULONG)*last, TAG_DONE );
    GT_SetGadgetAttrs( Select_RangeGadgets[GD_step], Select_RangeWnd, NULL,
      GTIN_Number, (ULONG)*step, TAG_DONE );

    ActivateGadget( Select_RangeGadgets[GD_from], Select_RangeWnd, NULL );

    while( !end )
    {
      WaitPort( Select_RangeWnd->UserPort );
      while( imsg = GT_GetIMsg( Select_RangeWnd->UserPort ) )
      {
        gad = imsg->IAddress;

        switch( imsg->Class )
        {
          case CLOSEWINDOW:
            end = TRUE;
            print = FALSE;
            break;
          case GADGETUP:
            switch( gad->GadgetID )
            {
              case GD_range_ok:
                end = TRUE;
                print = TRUE;
                break;
              case GD_range_cancel:
                end = TRUE;
                print = FALSE;
                break;
              default:
                break;
            }
            break;
          default:
            break;
        }
        GT_ReplyIMsg( imsg );
      }
    }
    str_info = Select_RangeGadgets[GD_from]->SpecialInfo;
    *first = str_info->LongInt;

    str_info = Select_RangeGadgets[GD_to]->SpecialInfo;
    *last = str_info->LongInt;

    str_info = Select_RangeGadgets[GD_step]->SpecialInfo;
    *step = str_info->LongInt;

    CloseSelect_RangeWindow();
  }

  return( print );
}

void SetBusyPointer( struct Window *window )
{
  if( window != NULL )
  {
    if( IntuitionBase->LibNode.lib_Version > 38 )
      SetWindowPointer( window, WA_BusyPointer, (ULONG)TRUE, TAG_DONE );
    else
      SetPointer( window, waitPointer, 16, 16, -6, 0 );
  }
}

int stop_key( void )
{
        return( 0 );
}

int wait_for_sheet( void )
{
  struct EasyStruct req =
  {
    sizeof( struct EasyStruct ),
    0,
    (UBYTE *)"Paper Request",
    (UBYTE *)"Insert new sheet",
    (UBYTE *)"OK|Cancel"
  };

  prt_req->io_Command = PRD_RAWWRITE;
  prt_req->io_Data = buffer;
  prt_req->io_Length = count;

  DoIO( (struct IORequest*)prt_req ); /* Druckerpuffer leeren */
  count = 0;

  return( !EasyRequest( win, &req, NULL ) );
}

void update_window( void )
{
  if( frame_valid )
  {
      LockLayerRom( sbm_win->RPort->Layer );
      CopySBitMap( sbm_win->RPort->Layer );
      UnlockLayerRom( sbm_win->RPort->Layer );
  }
}

void write_opts( void )
{
  char opt_line[128];
  FILE *opt_file;

  opt_file = fopen( opt_name, "w" );
  if( opt_file != NULL )
  {
    print( "Writing %s", opt_name );

    sprintf( opt_line, "%sCOPIES=%d\n", COMMENT( op.copies <= 0 ), op.copies );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sDENSITY=%lf\n", COMMENT( op.density == 0.0 ), op.density );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "EJECT=%s\n", ON_OFF( op.formfeed ) );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sHEIGHT=%lf in\n", COMMENT( op.height == 0.0 ), op.height );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sHMARGIN=%lf in\n", COMMENT( op.hmargin == 0.0 ), op.hmargin );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sHOFFSET=%lf in\n", COMMENT( op.hoffset == 0.0 ), op.hoffset );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sHRESOLUTION=%ld\n", COMMENT( op.hres <= 0 ), op.hres );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sHSPREAD=%lf in\n", COMMENT( op.hspread == 0.0 ), op.hspread );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sIMGPATH=%s\n", COMMENT( *op.img_path == '\0' ), op.img_path );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "LANDSCAPE=%s\n", ON_OFF( op.landscape ) );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sLOGFILE=%s\n", COMMENT( *op.log_name == '\0' ), op.log_name );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sMAGNIFICATION=%ld\n", COMMENT( op.new_mag <= 0 ), op.new_mag );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "MEMORY=%s\n", ON_OFF( op.dvimemory ) );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sOUTPUT=%s\n", COMMENT( *(p_defs[op.shipno].name) == '\0' ), p_defs[op.shipno].name );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sPATH=%s\n", COMMENT( *op.dvi_path == '\0' ), op.dvi_path );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sPATHMEM=%ld\n", COMMENT( op.pathmem <= 0 ), op.pathmem );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "PICTURES=%s\n", ON_OFF( op.show_img ) );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sPIXMEM=%ld\n", COMMENT( op.pixmem <= 0 ), op.pixmem );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sPKPATH=%s\n", COMMENT( *op.pk_path == '\0' ), op.pk_path );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sREDIRECT=%s\n", COMMENT( *op.redirect == '\0' ), op.redirect );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "SEPARATE=%s\n", ON_OFF( op.separate ) );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "SHOWFONTS=%s\n", ON_OFF( op.showfonts ) );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "SINGLESHEET=%s\n", ON_OFF( op.singlesheet ) );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sTFMPATH=%s\n", COMMENT( *op.tfm_path == '\0' ), op.tfm_path );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "THINOUT=%s\n", ON_OFF( op.thin_out ) );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "TRACECHARS=%s\n", ON_OFF( op.tracechars ) );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "TRACEMEM=%s\n", ON_OFF( op.tracemem ) );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sVFPATH=%s\n", COMMENT( *op.vf_path == '\0' ), op.vf_path );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sVMARGIN=%lf in\n", COMMENT( op.vmargin == 0.0 ), op.vmargin );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sVOFFSET=%lf in\n", COMMENT( op.voffset == 0.0 ), op.voffset );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sVRESOLUTION=%ld\n", COMMENT( op.vres <= 0 ), op.vres );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sVSPREAD=%lf in\n", COMMENT( op.vspread == 0.0 ), op.vspread );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sWIDTH=%lf in\n", COMMENT( op.width == 0.0 ), op.width );
    fputs( opt_line, opt_file );
    sprintf( opt_line, "%sCALLMF=%s\n", COMMENT( *op.pk_callmf == '\0' ), op.pk_callmf );
    fputs( opt_line, opt_file );

    fclose( opt_file );
  }
}

int main( int argc, char *argv[] )
{
  BPTR old_dir = NULL, lock = NULL;
  char *path, ch;

  shipout = NULL; /* shipout Hahn von dvi abdrehen */

  process = (struct Process *)FindTask( NULL ); /* eigenen Prozess ermitteln */

  sprintf( scr_title, "This is DVI Version %s" , DVI_VER );

  get_opts( argc );

  if( dvi_name[0] != '\0' )
  {
    path = PathPart( dvi_name );
    ch = *path;
    *path = '\0';
    lock = Lock( dvi_name, ACCESS_READ );
    *path = ch;
  }
  else
    lock = DupLock( process->pr_CurrentDir );

  if( BADDR( lock ) == NULL )
  {
    print( "Unable to create directory lock!" );
    return( RETURN_ERROR );
  }

  old_dir = CurrentDir( lock ); /* In das entsprechende Verzeichnis wechseln */

  if( gr_install() )
  {
    if( open_all() )
    {
      main_loop();
      dvi_clean_amiga();
    }
    close_all();
  }

  gr_destall();
  mem_test();

  UnLock( CurrentDir( old_dir ) );

  return( 0 );
}
