/*
*  dvichk.c:  An utility to print out the page numbers of a `.dvi'
*             resp., `.log' file created by the TeX formatter.
*
*
*  Copyright (c) 1990-1993, 2000 Thomas Esken
*
*  This software doesn't claim completeness, correctness or usability.
*  On principle I will not be liable for ANY damages or losses (implicit
*  or explicit), which result from using or handling my software.
*  If you use this software, you agree without any exception to this
*  agreement, which binds you LEGALLY !!
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the `GNU General Public License' as published by
*  the `Free Software Foundation'; either version 2, or (at your option)
*  any later version.
*
*  You should have received a copy of the `GNU General Public License'
*  along with this program; if not, write to the:
*
*    Free Software Foundation, Inc.
*    59 Temple Place - Suite 330
*    Boston, MA 02111-1307,  USA
*/



#ifdef RCSID
static char rcsid[]="$Id: dvichk.c 1.91 2000/06/07 01:09:01 tom Exp $";
#endif



/*
*  Include header files.
*/
#include "tailor.h"
#if defined(MSDOS) && !defined(DJG)
#  include <alloc.h>
#  include <mem.h>
#endif
#if HAVE_CTYPE_H
#  include <ctype.h>
#endif
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif
#if HAVE_SYS_STAT_H
#  include <sys/stat.h>
#endif
#include "common.h"
#include "utils.h"
#include "dvichk.h"
#include "globals.h"



/*
*  LOCAL functions prototypes.
*/
__BEGIN_DECLARATIONS
LOCAL Slint
decode_4byte __P_((const int b1,
                   const int b2,
                   const int b3,
                   const int b4));
LOCAL int
decode_2byte __P_((const int b1,
                   const int b2));
LOCAL char *
manage_dvi_pagenumber __P_((const Uchar *pagenumber_vector,
                                  int   *pagenumber_maxlen));
LOCAL Bool
check_log_pagenumber __P_((const char *pagenumber));
__END_DECLARATIONS




/*
*  Function implementations.
*/
   LOCAL Slint
decode_4byte (b1, b2, b3, b4)
   const int b1;
   const int b2;
   const int b3;
   const int b4;
/*
   Returns the four bytes B1 B2 B3 B4 which embody a 32bit value
     (High-Order...Low-Order) converted to a signed long decimal integer number.
*/
{
   return((Slint)   ((Uint)b1 >> 4) * 0x10000000L
                  + ((Uint)b1 & 15) * 0x1000000L
                  + ((Uint)b2 >> 4) * 0x100000L
                  + ((Uint)b2 & 15) * 0x10000L
                  + ((Uint)b3 >> 4) * 0x1000
                  + ((Uint)b3 & 15) * 0x100
                  + (Uint)b4);
}



   LOCAL int
decode_2byte (b1, b2)
   const int b1;
   const int b2;
/*
   Returns the two bytes B1 B2 which embody a 16bit value
     (High-Order...Low-Order) converted to a signed long decimal integer number.
*/
{
   return((int)   ((Uint)b1 >> 4) * 0x1000
                + ((Uint)b1 & 15) * 0x100
                + (Uint)b2);
}



   LOCAL char *
manage_dvi_pagenumber (pagenumber_vector, pagenumber_maxlen)
   const Uchar *pagenumber_vector;
         int   *pagenumber_maxlen;
/*
   Returns the decoded PAGENUMBER_VECTOR as a pointer to string
     and the maximum length of all pagenumbers processed
     by passing it via PAGENUMBER_MAXLEN.
*/
{
   static   int    maxlen;
   register int    i=DVI_PAGENUMBER_BYTES*DVI_PAGENUMBER_MAX;
   register int    n;
   static   char  *s;
   static   char  *the_pagenumber;
   static   Bool   is_initialized=FALSE;
   auto     Bool   subnumber_found=FALSE;


   if (!is_initialized)
    {
      the_pagenumber = (char *)my_malloc(128,
                                         ERR_NO_MEMORY_AVAILABLE,
                                         __FILE__, (long)__LINE__);
      s = (char *)my_malloc(MAXLEN_MAX,
                            ERR_NO_MEMORY_AVAILABLE,
                            __FILE__, (long)__LINE__);
      is_initialized = TRUE;
    }
   *s = '\0';
   maxlen = *pagenumber_maxlen;
   for (n=DVI_PAGENUMBER_MAX ; n > 0 ; n--,i-=DVI_PAGENUMBER_BYTES)
    {
      /*
         Convert the four bytes of COUNT? which embody a 64bit value
         (High-Order...Low-Order) to a signed long decimal number text.
      */
      sprintf(the_pagenumber, "%ld",
              (Slint)  ((Uint)pagenumber_vector[i-4] >> 4) * 0x10000000L
                     + ((Uint)pagenumber_vector[i-4] & 15) * 0x1000000L
                     + ((Uint)pagenumber_vector[i-3] >> 4) * 0x100000L
                     + ((Uint)pagenumber_vector[i-3] & 15) * 0x10000L
                     + ((Uint)pagenumber_vector[i-2] >> 4) * 0x1000L
                     + ((Uint)pagenumber_vector[i-2] & 15) * 0x100L
                     + (Uint)pagenumber_vector[i-1]);
      if (   subnumber_found
          || *the_pagenumber != '0')
       {
         if (subnumber_found)
           strcat(s, DVI_SUBNUMBER_DELIMITER);
         if (*(the_pagenumber + 1))
          {
            auto char  *tmp=my_strrev (the_pagenumber);


            strcat(s, tmp);
            free(tmp);
          }
         else
           strcat(s, the_pagenumber);
         subnumber_found = TRUE;
       }
    }
   i = (int)strlen(s);
   if (i > maxlen)
     maxlen = i;
   *pagenumber_maxlen = maxlen;

   return(s);
}



   LOCAL Bool
check_log_pagenumber (pagenumber)
   const char *pagenumber;
/*
   Returns TRUE in case PAGENUMBER is a syntactially valid pagenumber
     as it can be found in a `.log' file.  If PAGENUMBER is a construct
     like `[20pt]' or else, return FALSE.
*/
{
   register       int    j=0;
   auto     const char  *ptr_char=pagenumber;


   while (*ptr_char)
    {
      if (   (   !j
              && (   *ptr_char == '-'
                  || isdigit(*ptr_char)))
          || (   j
              && (   *ptr_char == *DVI_SUBNUMBER_DELIMITER
                  || isdigit(*ptr_char))))
       {
         j++;
         ptr_char++;
       }
      else
        return(FALSE);
    }

   return(TRUE);
}



   PUBLIC int
run_dvichk (argc, argv)
   int    argc;
   char  *argv[];
/*
   The DVIChk utility is realized here!
*/
{
#if HAVE_SYS_STAT_H && defined(S_IFMT) && defined(S_IFREG)
   auto     struct stat      statbuf;
#endif
   auto     Checkmode_enum   checkmode;
   auto     FILE            *f;
   register Uint             fonts_max=FONTS_MAX;
   register Uint             pages_max=PAGES_MAX;
   register int              last_error=EXIT_SUCCESS;
   register int              j;
   auto     char           **used_fonts;
   auto     char           **document_pages;
   auto     char            *logline;
   auto     char            *fn;
   auto     Uchar           *pagenumber_vector;
   auto     Bool             is_error=FALSE;
   auto     Bool             is_tty=isatty(1);


   if (the_options->total_only)
    {
      the_options->font_info = FALSE;
      the_options->columns = COLUMNS_MIN;
    }
   if (   !the_options->no_info
       && the_options->font_info)
      used_fonts = (char **)my_malloc (FONTS_MAX*sizeof(char *),
                                       ERR_NO_MEMORY_AVAILABLE,
                                       __FILE__, ((long)__LINE__)-2L);
   document_pages = (char **)my_malloc (PAGES_MAX*sizeof(char *),
                                        ERR_NO_MEMORY_AVAILABLE,
                                        __FILE__, ((long)__LINE__)-2L);
   logline = (char *)my_malloc (LINESIZE+1,
                                ERR_NO_MEMORY_AVAILABLE,
                                __FILE__, ((long)__LINE__)-2L);
   pagenumber_vector = (Uchar *)my_malloc (DVI_PAGENUMBER_BYTES*DVI_PAGENUMBER_MAX,
                                           ERR_NO_MEMORY_AVAILABLE,
                                           __FILE__, ((long)__LINE__)-2L);
   while (argv[argc] != (char *)NULL)
    {
      register int     fonts=0;
      register int     pages=0;
      register int     size;
      register int     i;
      auto     int     column_width=COLUMN_WIDTH;
      static   Uchar   dvibuf[(Uint)BLOCKSIZE];
      auto     char   *logbuf=(char *)dvibuf;
      auto     char   *dvi_id_text=(char *)NULL;
      auto     char   *tmp;
      auto     Bool    bop_found=FALSE;
      auto     Bool    eop_found=FALSE;
#if HAVE_SYS_STAT_H && defined(S_IFMT) && defined(S_IFREG)
      auto     Bool    is_regular_file=TRUE;
#endif


      f = fopen(argv[argc], "r");
      if (f != (FILE *)NULL)
       {
         if (fclose(f) == EOF)
           (void)my_error (stderr, ERR_CLOSE_FILE, argv[argc], 0L);
         fn = strdup (argv[argc]);
       }
      else
       {
         sprintf(logline, "%s%s", argv[argc], LOG_FN_EXTENSION);
         f = fopen(logline, "r");
         if (f != (FILE *)NULL)
          {
            if (fclose(f) == EOF)
              (void)my_error (stderr, ERR_CLOSE_FILE, logline, 0L);
            fn = strdup (logline);
          }
         else
          {
            sprintf(logline, "%s%s", argv[argc], DVI_FN_EXTENSION);
            f = fopen(logline, "r");
            if (f != (FILE *)NULL)
             {
               if (fclose(f) == EOF)
                 (void)my_error (stderr, ERR_CLOSE_FILE, logline, 0L);
               fn = strdup (logline);
             }
            else
              fn = strdup (argv[argc]);
          }
       }
      f = (FILE *)NULL;
#if HAVE_SYS_STAT_H && defined(S_IFMT) && defined(S_IFREG)
      /*
         Test if the file is a regular file, if not, this is an error!
      */
      if (!stat(fn, &statbuf))
       {
#  if defined(MSDOS) && !defined(DJG) && defined(S_IREAD)
         if (!(statbuf.st_mode & S_IREAD))
           chmod(fn, S_IREAD);
#  endif
         if ((statbuf.st_mode & S_IFMT) == S_IFREG)
           f = fopen(fn, "rb");
         else
           is_regular_file = FALSE;
       }
#else /* !HAVE_SYS_STAT_H || !S_IFMT || !S_IFREG */
      f = fopen(fn, "rb");
#endif /* !HAVE_SYS_STAT_H || !S_IFMT || !S_IFREG */
      if (   !is_regular_file
          || f == (FILE *)NULL)
       {
         if (!is_regular_file)
           last_error = my_error (stderr, ERR_IRREGULAR_FILE, fn, 0L);
         else
           last_error = my_error (stderr, ERR_FILE_NOT_FOUND, fn, 0L);
       }
      else
       {
         size = fread(dvibuf, 1, (Uint)(BLOCKSIZE)*sizeof(Uchar), f);
         if (!size)
          {
            last_error = my_error (stderr, ERR_ZERO_FILE, fn, 0L);
            if (fclose(f) == EOF)
              (void)my_error (stderr, ERR_CLOSE_FILE, fn, 0L);
          }
         else
          {
            /*
               Check for I[1], the DVI file magic.
            */
            if (   (*dvibuf == DVI_MAGIC1)
                && (   dvibuf[1] == DVI_MAGIC2
                    || dvibuf[1] == DVI_MAGIC3
                    || dvibuf[1] == DVI_MAGIC4))
             {
               register int     dvifile_size=size;
               register int     n;
               register int     k=0;
               auto     Bool    dvifile_mmapped=FALSE;
               auto     Bool    dvi_preamble=TRUE;
               auto     Bool    buffer_finished=FALSE;
               auto     Bool    pagenumber_split=FALSE;
               auto     char   *the_dvifile=(char *)NULL;
               auto     char    dvi_magic=dvibuf[1];


               /*
                  DVI file magic found.
               */
               checkmode = check_dvi;
#if HAVE_SYS_STAT_H && defined(S_IFMT) && defined(S_IFREG)
               /*
                  Check whether we can map the DVI file into memory.
               */
#  if defined(MSDOS) && !defined(DJG)
               if (statbuf.st_size <= testval)
                {
                  dvifile_size = (int)statbuf.st_size;
                  the_dvifile = (char *)malloc(dvifile_size);
                }
#  else /* !MSDOS || DJG */
               dvifile_size = statbuf.st_size;
               the_dvifile = (char *)malloc(dvifile_size);
#  endif /* !MSDOS || DJG */
               if (the_dvifile == (char *)NULL)
                {
                  /*
                     Let's process the DVI file from BOF to EOF block by block.
                  */
                  if (fclose(f) == EOF)
                    (void)my_error (stderr, ERR_CLOSE_FILE, fn, 0L);
                }
               else
                {
                  register int  the_size=size;


                  /*
                     Read the complete DVI file into memory,
                       used method:  WITHOUT reallocation.
                  */
                  dvifile_mmapped = TRUE;
                  tmp = (char *)memcpy(the_dvifile, dvibuf, size);
                  if (tmp == (char *)NULL)
                    (void)my_error (stderr, ERR_C_FUNCTION_FAILURE, __FILE__, (long)__LINE__);
                  while (size == BLOCKSIZE)
                   {
                     size = fread(dvibuf, 1, (Uint)(BLOCKSIZE)*sizeof(Uchar), f);
                     if (size)
                      {
                        tmp = (char *)memcpy(the_dvifile+the_size, dvibuf, size);
                        if (tmp == (char *)NULL)
                          (void)my_error (stderr, ERR_C_FUNCTION_FAILURE, __FILE__, (long)__LINE__);
                        the_size += size;
                      }
                   }
                }
#else /* !HAVE_SYS_STAT_H || !S_IFMT || !S_IFREG */
#  if defined(MSDOS) && !defined(DJG)
               /*
                  Default mode is to process the DVI file from BOF to EOF block by block.
               */
               if (fclose(f) == EOF)
                 (void)my_error (stderr, ERR_CLOSE_FILE, fn, 0L);
#  else /* !MSDOS || DJG */
               /*
                  Check whether we can map the DVI file into memory.
               */
               the_dvifile = (char *)malloc(size);
               if (the_dvifile != (char *)NULL)
                {
                  /*
                     Read the complete DVI file into memory,
                       used method:  WITH reallocation.
                  */
                  dvifile_mmapped = TRUE;
                  tmp = (char *)memcpy(the_dvifile, dvibuf, size);
                  if (tmp == (char *)NULL)
                    (void)my_error (stderr, ERR_C_FUNCTION_FAILURE, __FILE__, (long)__LINE__);
                  while (size == BLOCKSIZE)
                   {
                     size = fread(dvibuf, 1, (Uint)(BLOCKSIZE)*sizeof(Uchar), f);
                     if (size)
                      {
                        dvifile_size += size;
                        the_dvifile = (char *)realloc ((VOID_PTR)the_dvifile, dvifile_size);
                        if (the_dvifile == (char *)NULL)
                         {
                           /*
                              Error, so let's process the DVI file from BOF to EOF block by block.
                           */
                           size = 0;
                           dvifile_mmapped = FALSE;
                           free(the_dvifile);
                           if (fclose(f) == EOF)
                             (void)my_error (stderr, ERR_CLOSE_FILE, fn, 0L);
                         }
                        else
                         {
                           tmp = (char *)memcpy(the_dvifile+dvifile_size-size, dvibuf, size);
                           if (tmp == (char *)NULL)
                             (void)my_error (stderr, ERR_C_FUNCTION_FAILURE, __FILE__, (long)__LINE__);
                         }
                      }
                   }
                }
#  endif /* !MSDOS || DJG */
#endif /* !HAVE_SYS_STAT_H || !S_IFMT || !S_IFREG */
               if (dvifile_mmapped)
                {
                  /*
                     Now let's process the DVI file mapped into memory.
                  */
                  i = dvifile_size - 1;
                  /*
                     Decode the POSTAMBLE, so jump to last byte in DVI file.
                  */
                  j = (Uchar)*(the_dvifile + i);
                  if (j == DVI_FILLER)
                   {
                     /*
                        Skip all trailing 223's fillers.
                     */
                     i -= 4;
                     n = 0;
                     while ((Uchar)*(the_dvifile + i) == DVI_FILLER)
                      {
                        i--;
                        n++;
                      }
                     /*
                        Check for the correct number of 223's fillers.
                     */
                     if (n <= 3)
                       /*
                          Check for the correct DVI file magic (DVI format ID I[1]).
                       */
                       if (*(the_dvifile + i) == dvi_magic)
                        {
                          /*
                             Get Q[4], the pointer to the start of the POSTAMBLE
                               and jump to it.
                          */
                          i = (int)decode_4byte ((Uchar)*(the_dvifile+i-4),
                                                 (Uchar)*(the_dvifile+i-3),
                                                 (Uchar)*(the_dvifile+i-2),
                                                 (Uchar)*(the_dvifile+i-1));
                          /*
                             Check if it is really the POSTAMBLE POST[1] we jumped to.
                          */
                          if ((Uchar)*(the_dvifile + i) == DVI_POST)
                           {
#if 0
#if defined(MSDOS) && !defined(DJG)
                             /*
                                Get NUM[4], the numerator of the fraction 10^-7 meter,
                                  which offset is +5 bytes from the start
                                  of the POSTAMBLE.
                             */
                             auto Slint  num=decode_4byte ((Uchar)*(the_dvifile+i+5),
                                                           (Uchar)*(the_dvifile+i+6),
                                                           (Uchar)*(the_dvifile+i+7),
                                                           (Uchar)*(the_dvifile+i+8));
                             /*
                                Get DEN[4], the denominator of the fraction 10^-7 meter,
                                  which offset is +9 bytes from the start
                                  of the POSTAMBLE.
                             */
                             auto Slint den=decode_4byte ((Uchar)*(the_dvifile+i+9),
                                                          (Uchar)*(the_dvifile+i+10),
                                                          (Uchar)*(the_dvifile+i+11),
                                                          (Uchar)*(the_dvifile+i+12));
#endif /* 0 */
                             /*
                                Get MAG[4], the font magnification factor,
                                  which offset is +13 bytes from the start
                                  of the POSTAMBLE.
                             */
                             auto Slint  mag=decode_4byte ((Uchar)*(the_dvifile+i+13),
                                                           (Uchar)*(the_dvifile+i+14),
                                                           (Uchar)*(the_dvifile+i+15),
                                                           (Uchar)*(the_dvifile+i+16));
#else /* !MSDOS || DJG */
#if 0
                             /*
                                Get NUM[4], the numerator of the fraction 10^-7 meter,
                                  which offset is +5 bytes from the start
                                  of the POSTAMBLE.
                             */
                             register int  num=(int)decode_4byte ((Uchar)*(the_dvifile+i+5),
                                                                  (Uchar)*(the_dvifile+i+6),
                                                                  (Uchar)*(the_dvifile+i+7),
                                                                  (Uchar)*(the_dvifile+i+8));
                             /*
                                Get DEN[4], the denominator of the fraction 10^-7 meter,
                                  which offset is +9 bytes from the start
                                  of the POSTAMBLE.
                             */
                             register int  den=(int)decode_4byte ((Uchar)*(the_dvifile+i+9),
                                                                  (Uchar)*(the_dvifile+i+10),
                                                                  (Uchar)*(the_dvifile+i+11),
                                                                  (Uchar)*(the_dvifile+i+12));
#endif /* 0 */
                             /*
                                Get MAG[4], the font magnification factor,
                                  which offset is +13 bytes from the start
                                  of the POSTAMBLE.
                             */
                             register int  mag=(int)decode_4byte ((Uchar)*(the_dvifile+i+13),
                                                                  (Uchar)*(the_dvifile+i+14),
                                                                  (Uchar)*(the_dvifile+i+15),
                                                                  (Uchar)*(the_dvifile+i+16));
#endif /* !MSDOS || DJG */


                             /*
                                Get T[2], the number of pages (BOP's) in that
                                  DVI file, which offset is +27 bytes
                                  from the actual BOP we are pointing to.
                             */
                             n=pages = decode_2byte ((Uchar)*(the_dvifile+i+27),
                                                     (Uchar)*(the_dvifile+i+28));
                             if ((Uint)pages >= pages_max)
                              {
                                /*
                                   Resize the `document_pages[]' table.
                                */
                                pages_max = pages;
                                if (pages_max*sizeof(char *) > testval)
                                  pages_max--;
                                document_pages = (char **)my_realloc ((VOID_PTR)document_pages,
                                                                      pages_max*sizeof(char *),
                                                                      ERR_NO_MEMORY_AVAILABLE,
                                                                      __FILE__, ((long)__LINE__)-3L);
                              }
                             /*
                                Save the address of the POSTAMBLE.
                             */
                             j = i;
                             if (   !the_options->no_info
                                 && the_options->font_info)
                              {
                                /*
                                   Jump to the FONT DEFINITION part
                                     which offset is +29 bytes from the POSTAMBLE POST[1].
                                */
                                i += 29;
                                /*
                                   List all N[a[1]+l[1]], the font names used,
                                     which are listed until we find a POST_POST[1] command.
                                */
                                while ((Uchar)*(the_dvifile + i) != DVI_POST_POST)
                                 {
                                   auto     double  x;
#if defined(MSDOS) && !defined(DJG)
                                   auto     Slint   c;
                                   auto     Slint   s;
                                   auto     Slint   d;
#else /* !MSDOS || DJG */
                                   register int     c;
                                   register int     s;
                                   register int     d;
#endif /* !MSDOS || DJG */
                                   register int     a;
                                   register int     l;


                                   /*
                                      Check whether we found a FNT_DEF1...FNT_DEF4 command.
                                   */
                                   if (   (Uchar)*(the_dvifile + i) == DVI_FNT_DEF1
                                       || (Uchar)*(the_dvifile + i) == DVI_FNT_DEF2
                                       || (Uchar)*(the_dvifile + i) == DVI_FNT_DEF3
                                       || (Uchar)*(the_dvifile + i) == DVI_FNT_DEF4)
                                    {
                                      /*
                                         Jump to the C[4], the checksum of the font
                                           by skipping +n bytes, which represent the font number K[n-1].
                                      */
                                      i += ((Uchar)*(the_dvifile + i) - DVI_FNT_DEF1 + 2);
#if defined(MSDOS) && !defined(DJG)
                                      /*
                                         Get C[4], the checksum of the font,
                                           which offset is +0 bytes from the start
                                           of each single FONT DEFINITION.
                                      */
                                      c = decode_4byte ((Uchar)*(the_dvifile + i),
                                                        (Uchar)*(the_dvifile + i + 1),
                                                        (Uchar)*(the_dvifile + i + 2),
                                                        (Uchar)*(the_dvifile + i + 3));
                                      /*
                                         Get S[4], the fixed point scale factor of the font,
                                           which offset is +4 bytes from the start
                                           of each single FONT DEFINITION.
                                      */
                                      s = decode_4byte ((Uchar)*(the_dvifile + i + 4),
                                                        (Uchar)*(the_dvifile + i + 5),
                                                        (Uchar)*(the_dvifile + i + 6),
                                                        (Uchar)*(the_dvifile + i + 7));
                                      /*
                                         Get D[4], the design size of the font,
                                           which offset is +8 bytes from the start
                                           of each single FONT DEFINITION.
                                      */
                                      d = decode_4byte ((Uchar)*(the_dvifile + i + 8),
                                                        (Uchar)*(the_dvifile + i + 9),
                                                        (Uchar)*(the_dvifile + i + 10),
                                                        (Uchar)*(the_dvifile + i + 11));
#else /* !MSDOS || DJG */
                                      /*
                                         Get C[4], the checksum of the font,
                                           which offset is +0 bytes from the start
                                           of each single FONT DEFINITION.
                                      */
                                      c = (int)decode_4byte ((Uchar)*(the_dvifile + i),
                                                             (Uchar)*(the_dvifile + i + 1),
                                                             (Uchar)*(the_dvifile + i + 2),
                                                             (Uchar)*(the_dvifile + i + 3));
                                      /*
                                         Get S[4], the fixed point scale factor of the font,
                                           which offset is +4 bytes from the start
                                           of each single FONT DEFINITION.
                                      */
                                      s = (int)decode_4byte ((Uchar)*(the_dvifile + i + 4),
                                                             (Uchar)*(the_dvifile + i + 5),
                                                             (Uchar)*(the_dvifile + i + 6),
                                                             (Uchar)*(the_dvifile + i + 7));
                                      /*
                                         Get D[4], the design size of the font,
                                           which offset is +8 bytes from the start
                                           of each single FONT DEFINITION.
                                      */
                                      d = (int)decode_4byte ((Uchar)*(the_dvifile + i + 8),
                                                             (Uchar)*(the_dvifile + i + 9),
                                                             (Uchar)*(the_dvifile + i + 10),
                                                             (Uchar)*(the_dvifile + i + 11));
#endif /* !MSDOS || DJG */
                                      /*
                                         Get A[1], the length of the font name area/directory.
                                      */
                                      a = (Uchar)*(the_dvifile + i + 12);
                                      /*
                                         Get L[1], the length of the font name itself.
                                      */
                                      l = (Uchar)*(the_dvifile + i + 13);
                                      /*
                                         Get the name of the font.
                                      */
                                      tmp = (char *)my_malloc (l+1,
                                                               ERR_NO_MEMORY_AVAILABLE,
                                                               __FILE__, ((long)__LINE__)-2L);
                                      strncpy(tmp, the_dvifile+i+14+a, l);
                                      tmp[l] = '\0';
                                      /*
                                         Compute the used magnification of the font.
                                      */
#if defined(MSDOS) && !defined(DJG)
                                      x = mag * ((double)mag*(Ulint)s) / (1000*(double)((Ulint)d));
                                      /*
                                         Create a single font information line.
                                      */
                                      sprintf(logline, "%s %05d %#8lx", tmp, (int)ROUND(x), c);
#else /* !MSDOS || DJG */
                                      x = mag * ((double)mag*(Uint)s) / (1000*(double)((Uint)d));
                                      /*
                                         Create a single font information line.
                                      */
                                      sprintf(logline, "%s %05d %#8x", tmp, (int)ROUND(x), c);
#endif /* !MSDOS || DJG */
                                      free(tmp);
                                      /*
                                          Store that line.
                                      */
                                      if ((Uint)fonts >= fonts_max)
                                       {
                                         /*
                                            Resize the `used_fonts[]' table.
                                         */
                                         fonts_max <<= 1;
                                         if (fonts_max*sizeof(char *) > testval)
                                           fonts_max--;
                                         used_fonts = (char **)my_realloc ((VOID_PTR)used_fonts,
                                                                           fonts_max*sizeof(char *),
                                                                           ERR_NO_MEMORY_AVAILABLE,
                                                                           __FILE__, ((long)__LINE__)-3L);
                                       }
                                      used_fonts[fonts] = (char *)my_malloc (strlen(logline)+1,
                                                                             ERR_NO_MEMORY_AVAILABLE,
                                                                             __FILE__, ((long)__LINE__)-2L);
                                      strcpy(used_fonts[fonts++], logline);
                                      /*
                                         Jump to the next FONT DEFINITION command,
                                           which offset +14+a+l bytes from here.
                                      */
                                      i += (14+a+l);
                                      /*
                                         Skip all possibly trailing NON-OPERATION codes.
                                      */
                                      while ((Uchar)*(the_dvifile + i) == DVI_NOP)
                                        i++;
                                    }
                                   else
                                    {
                                      /*
                                         Error, corrupted font definition found,
                                           but keep it as a secret :)
                                      */
                                      fonts = 0;
                                      break;
                                    }
                                 }
                                if (fonts > 1)
                                  qsort((VOID_PTR)used_fonts, fonts, sizeof *used_fonts, (Cmp_func)asc_sort);
                              }
                             /*
                                Get P[4], the pointer to the final BOP
                                  and jump to it.
                             */
                             i = (int)decode_4byte ((Uchar)*(the_dvifile+j+1),
                                                    (Uchar)*(the_dvifile+j+2),
                                                    (Uchar)*(the_dvifile+j+3),
                                                    (Uchar)*(the_dvifile+j+4));
                             /*
                                And start getting all page numbers of the DVI file.
                             */
                             while (i != -1)
                              {
                                /*
                                   Check if it is really the BOP[1] we jumped to.
                                */
                                if ((Uchar)*(the_dvifile + i) == DVI_BOP)
                                 {
                                   /*
                                      Copy COUNT0[4]...COUNT9[4] to a buffer.
                                   */
                                   tmp = (char *)memcpy(pagenumber_vector, the_dvifile+i+1, DVI_PAGENUMBER_BYTES*DVI_PAGENUMBER_MAX);
                                   if (tmp == (char *)NULL)
                                     (void)my_error (stderr, ERR_C_FUNCTION_FAILURE, __FILE__, (long)__LINE__);
                                   /*
                                      Decode that buffer.
                                   */
                                   tmp = my_strrev (manage_dvi_pagenumber (pagenumber_vector, &column_width));
                                   /*
                                      Store that buffer.
                                   */
                                   document_pages[--pages] = (char *)my_malloc (strlen(tmp)+1,
                                                                                ERR_NO_MEMORY_AVAILABLE,
                                                                                __FILE__, ((long)__LINE__)-2L);
                                   strcpy(document_pages[pages], tmp);
                                   free(tmp);
                                   /*
                                       Get P[4], the pointer to the previous BOP[1]
                                         and jump to it.
                                   */
                                   i = (int)decode_4byte ((Uchar)*(the_dvifile+i+1+DVI_PAGENUMBER_BYTES*DVI_PAGENUMBER_MAX),
                                                          (Uchar)*(the_dvifile+i+2+DVI_PAGENUMBER_BYTES*DVI_PAGENUMBER_MAX),
                                                          (Uchar)*(the_dvifile+i+3+DVI_PAGENUMBER_BYTES*DVI_PAGENUMBER_MAX),
                                                          (Uchar)*(the_dvifile+i+4+DVI_PAGENUMBER_BYTES*DVI_PAGENUMBER_MAX));
                                 }
                                else
                                 {
                                   /*
                                      Error, DVI file corrupt.
                                   */
                                   n = 0;
                                   break;
                                 }
                              }
                             if (   n
                                 && !the_options->no_info)
                              {
                                /*
                                   Decode K[1] of the PREAMBLE to get the length of the comment X.
                                */
                                j = (int)*(the_dvifile + DVI_COMMENT_OFFSET);
                                dvi_id_text = (char *)my_malloc (j+1,
                                                                 ERR_NO_MEMORY_AVAILABLE,
                                                                 __FILE__, ((long)__LINE__)-2L);
                                /*
                                   Store the comment X[K[1]], where 0<=K<256.
                                */
                                strncpy(dvi_id_text, (char *)(the_dvifile+DVI_COMMENT_OFFSET+1), j);
                                dvi_id_text[j] = '\0';

                              }
                             /*
                                Finally, store the number of pages into PAGES
                                  which is 0 in case an error occured during
                                  former processing of the DVI file.
                             */
                             pages = n;
                           }
                        }
                   }
                  free(the_dvifile);
                }
               else
                {
                  /*
                     Now let's process the DVI file from BOF to EOF block by block.
                  */
                  f = fopen(fn, "rb");
                  size = fread(dvibuf, 1, (Uint)(BLOCKSIZE)*sizeof(Uchar), f);
                  /*
                     Decode K[1] of the PREAMBLE to get the length of the comment X.
                  */
                  i = (int)*(dvibuf + DVI_COMMENT_OFFSET);
                  if (!the_options->no_info)
                   {
                     dvi_id_text = (char *)my_malloc (i+1,
                                                      ERR_NO_MEMORY_AVAILABLE,
                                                      __FILE__, ((long)__LINE__)-2L);
                     /*
                        Store the comment X[K[1]], where 0<=K<256.
                     */
                     strncpy(dvi_id_text, (char *)(dvibuf+DVI_COMMENT_OFFSET+1), i);
                     dvi_id_text[i] = '\0';
                   }
                  /*
                  Jump to first COUNT0[4] byte after the first BOP[1].
                  */
                  i += (DVI_COMMENT_OFFSET + 2);
                  /*
                     Copy COUNT0[4]...COUNT9[4] to a buffer.
                  */
                  tmp = (char *)memcpy(pagenumber_vector, dvibuf+i, DVI_PAGENUMBER_BYTES*DVI_PAGENUMBER_MAX);
                  if (tmp == (char *)NULL)
                    (void)my_error (stderr, ERR_C_FUNCTION_FAILURE, __FILE__, (long)__LINE__);
                  /*
                     Decode that buffer.
                  */
                  tmp = my_strrev (manage_dvi_pagenumber (pagenumber_vector, &column_width));
                  /*
                     Store that buffer.
                  */
                  document_pages[pages] = (char *)my_malloc (strlen(tmp)+1,
                                                             ERR_NO_MEMORY_AVAILABLE,
                                                             __FILE__, ((long)__LINE__)-2L);
                  strcpy(document_pages[pages++], tmp);
                  free(tmp);
                  /*
                     Jump to the first real text byte of first page (where P[4]==-1)
                       for detecting the next DVI_EOP/DVI_BOP commands.
                  */
                  i += (DVI_PAGENUMBER_BYTES * (DVI_PAGENUMBER_MAX + 1));
                  do
                   {
                     /*
                        Read again only if we've processed the actual file buffer
                          in which we've stored the PREAMBLE.
                     */
                     if (!dvi_preamble)
                      {
                        size = fread(dvibuf, 1, (Uint)(BLOCKSIZE)*sizeof(Uchar), f);
                        buffer_finished = !size;
                      }
                     else
                       dvi_preamble = FALSE;
                     while (!buffer_finished)
                      {
                        /*
                           Check whether we can find a page
                             in the actual file buffer.
                        */
                        if (   dvibuf[i] == DVI_EOP
                            || eop_found)
                         {
                           if (!eop_found)
                             i++;
                           eop_found = TRUE;
                           if (i < size)
                            {
                              if (dvibuf[i] == DVI_BOP)
                                bop_found = TRUE;
                              else
                               {
                                 if (dvibuf[i] == DVI_NOP)
                                  {
                                    while (dvibuf[i] == DVI_NOP)
                                     {
                                       i++;
                                       if (i >= size)
                                        {
                                          buffer_finished = TRUE;
                                          break;
                                        }
                                       else
                                         if (dvibuf[i] == DVI_BOP)
                                          {
                                            bop_found = TRUE;
                                            break;
                                          }
                                     }
                                  }
                                 else
                                   eop_found = FALSE;
                               }
                            }
                         }
                        if (   eop_found
                            && bop_found)
                         {
                           /*
                              We found a page in the actual file buffer,
                                so let's compute how many bytes are
                                still remaining in the file buffer.
                           */
                           n = size - (i + 1);
                           if (n)
                            {
                              /*
                                 Found the first byte of COUNT0[4] minimum,
                                   so check if we just managed the first
                                   bytes of COUNT0[4]...COUNT9[4] by reason
                                   the previous file buffer hasn't contained
                                   COUNT0[4]...COUNT9[4] completely.
                              */
                              if (!pagenumber_split)
                               {
                                 /*
                                    Jump to the first byte of COUNT0[4].
                                 */
                                 i++;
                                 /*
                                    Check if all bytes of COUNT0[4]...COUNT9[4]
                                      are contained in the actual file buffer.
                                 */
                                 if (n >= DVI_PAGENUMBER_BYTES*DVI_PAGENUMBER_MAX)
                                  {
                                    /*
                                       All bytes of COUNT0[4]...COUNT9[4] are
                                       contained in the actual file buffer,
                                       so copy COUNT0[4]...COUNT9[4] to a buffer.
                                    */
                                    tmp = (char *)memcpy(pagenumber_vector, dvibuf+i, DVI_PAGENUMBER_BYTES*DVI_PAGENUMBER_MAX);
                                    if (tmp == (char *)NULL)
                                      (void)my_error (stderr, ERR_C_FUNCTION_FAILURE, __FILE__, (long)__LINE__);
                                    /*
                                       Decode that buffer.
                                    */
                                    tmp = my_strrev (manage_dvi_pagenumber (pagenumber_vector, &column_width));
                                    /*
                                       Store that buffer.
                                    */
                                    if ((Uint)pages >= pages_max)
                                     {
                                       /*
                                          Resize the `document_pages[]' table.
                                       */
                                       pages_max <<= 1;
                                       if (pages_max*sizeof(char *) > testval)
                                         pages_max--;
                                       document_pages = (char **)my_realloc ((VOID_PTR)document_pages,
                                                                             pages_max*sizeof(char *),
                                                                             ERR_NO_MEMORY_AVAILABLE,
                                                                             __FILE__, ((long)__LINE__)-3L);
                                     }
                                    document_pages[pages] = (char *)my_malloc (strlen(tmp)+1,
                                                                               ERR_NO_MEMORY_AVAILABLE,
                                                                               __FILE__, ((long)__LINE__)-2L);
                                    strcpy(document_pages[pages++], tmp);
                                    free(tmp);
                                    /*
                                       Jump to the first real text byte of the next page
                                         for detecting the next DVI_EOP/DVI_BOP commands.
                                    */
                                    i += ((DVI_PAGENUMBER_BYTES * (DVI_PAGENUMBER_MAX + 1)) - 1);
                                    eop_found=bop_found = FALSE;
                                  }
                                 else
                                  {
                                    /*
                                       Only some bytes of COUNT0[4]...COUNT9[4] are
                                       contained in the actual file buffer, so copy
                                       them to a buffer and manage the remaining
                                       bytes of them after we've read the next
                                       file buffer.
                                    */
                                    pagenumber_split = TRUE;
                                    k = n;
                                    /*
                                       Copy the first part of COUNT0[4]...COUNT9[4]
                                         to a buffer.
                                    */
                                    tmp = (char *)memcpy(pagenumber_vector, dvibuf+i, n);
                                    if (tmp == (char *)NULL)
                                      (void)my_error (stderr, ERR_C_FUNCTION_FAILURE, __FILE__, (long)__LINE__);
                                    /*
                                       Jump to end of the actual file buffer
                                         so the next file buffer is read.
                                    */
                                    i += (n - 1);
                                  }
                               }
                              else
                               {
                                 /*
                                    The previous file buffer contained only
                                      some bytes of COUNT0[4]...COUNT9[4], so manage
                                      the remaining bytes of them now, which
                                      are listed at the beginning of the actual
                                      file buffer.
                                 */
                                 pagenumber_split = FALSE;
                                 /*
                                    Copy the remaining COUNT0[4]...COUNT9[4] bytes
                                      to a buffer.
                                 */
                                 n = (DVI_PAGENUMBER_BYTES * DVI_PAGENUMBER_MAX) - k;
                                 tmp = (char *)memcpy(pagenumber_vector+k, dvibuf+i, n);
                                 if (tmp == (char *)NULL)
                                   (void)my_error (stderr, ERR_C_FUNCTION_FAILURE, __FILE__, (long)__LINE__);
                                 /*
                                    Decode that buffer.
                                 */
                                 tmp = my_strrev (manage_dvi_pagenumber (pagenumber_vector, &column_width));
                                 /*
                                    Store that buffer.
                                 */
                                 if ((Uint)pages >= pages_max)
                                  {
                                    /*
                                       Resize the `document_pages[]' table.
                                    */
                                    pages_max <<= 1;
                                    if (pages_max*sizeof(char *) > testval)
                                      pages_max--;
                                    document_pages = (char **)my_realloc ((VOID_PTR)document_pages,
                                                                          pages_max*sizeof(char *),
                                                                          ERR_NO_MEMORY_AVAILABLE,
                                                                          __FILE__, ((long)__LINE__)-3L);
                                  }
                                 document_pages[pages] = (char *)my_malloc (strlen(tmp)+1,
                                                                            ERR_NO_MEMORY_AVAILABLE,
                                                                            __FILE__, ((long)__LINE__)-2L);
                                 strcpy(document_pages[pages++], tmp);
                                 free(tmp);
                                 /*
                                    Jump to the first real text byte of the next page
                                      for detecting the next DVI_EOP/DVI_BOP commands.
                                 */
                                 i += ((n - 1) + DVI_PAGENUMBER_BYTES);
                                 eop_found=bop_found = FALSE;
                               }
                            }
                         }
                        i++;
                        if (i >= size)
                          buffer_finished = TRUE;
                      }
                     i = 0;
                   } while (size);
                }
             }
            else
             {
               /*
                  Let's check whether the LOG file is a text file, so let's
                    search for NUL characters, which means, if we find any of
                    them, the file is no text file by reason text files
                    typically do not contain NUL characters.
               */
               for (i=0 ; i < size ; i++)
                 /*
                    We found a NUL character.
                 */
                 if (!dvibuf[i])
                   break;
               if (i == size)
                {
                  static char  str3[3]={LOG_BOP, LOG_EOP, '\0'};
                  auto   Bool  read_logline=TRUE;
                  auto   Bool  logline_finished=FALSE;
                  auto   Bool  delete_next_bracket=FALSE;


                  if (fclose(f) == EOF)
                    (void)my_error (stderr, ERR_CLOSE_FILE, fn, 0L);
                  f = fopen(fn, "r");
                  setvbuf(f, logbuf, _IOFBF, (Uint)BLOCKSIZE);
                  if (fgets(logline, LINESIZE, f) != (char *)NULL)
                   {
                     tmp = strchr(logline, '\n');
                     if (tmp != (char *)NULL)
                       *tmp = '\0';
                   }
                  checkmode = check_log;
                  while (!feof(f))
                   {
                     if (read_logline)
                      {
                        if (fgets(logline, LINESIZE, f) != (char *)NULL)
                         {
                           tmp = strchr(logline, '\n');
                           if (tmp != (char *)NULL)
                             *tmp = '\0';
                         }
                      }
                     else
                       read_logline = TRUE;
                     tmp = "";
                     while (tmp != (char *)NULL)
                      {
                        tmp = strstr (logline, str3);
                        if (tmp != (char *)NULL)
                         {
                           *tmp = ' ';
                           tmp[1] = *tmp;
                         }
                      }
                     logline_finished = FALSE;
                     i = 0;
                     tmp = strchr(logline, LOG_BOP);
                     if (tmp != (char *)NULL)
                      {
                        i = (int)(tmp - logline);
                        bop_found = TRUE;
                      }
                     else
                       bop_found = FALSE;
                     j = 0;
                     tmp = strchr(logline, LOG_EOP);
                     if (tmp != (char *)NULL)
                      {
                        j = (int)(tmp - logline);
                        eop_found = TRUE;
                      }
                     else
                       eop_found = FALSE;
                     if (   (   bop_found
                             && (   (   logline[i+1] > '9'
                                     || logline[i+1] < '0')
                                 && (logline[i+1] != '-')))
                         || (   (i > 0)
                             && (logline[i-1] != ' ')))
                      {
                        bop_found=eop_found = FALSE;
                        logline[i] = ' ';
                      }
                     if (bop_found)
                      {
                        if (   logline[i+1]
                            && !eop_found)
                         {
                           tmp = logline + i + 1;
                           if (check_log_pagenumber (tmp))
                            {
                              if ((Uint)pages >= pages_max)
                               {
                                 /*
                                    Resize the `document_pages[]' table.
                                 */
                                 pages_max <<= 1;
                                 if (pages_max*sizeof(char *) > testval)
                                   pages_max--;
                                 document_pages = (char **)my_realloc ((VOID_PTR)document_pages,
                                                                       pages_max*sizeof(char *),
                                                                       ERR_NO_MEMORY_AVAILABLE,
                                                                       __FILE__, ((long)__LINE__)-3L);
                               }
                              document_pages[pages] = (char *)my_malloc (strlen(tmp)+1,
                                                                         ERR_NO_MEMORY_AVAILABLE,
                                                                         __FILE__, ((long)__LINE__)-2L);
                              strcpy(document_pages[pages++], tmp);
                            }
                           bop_found = FALSE;
                           logline_finished=delete_next_bracket = TRUE;
                        }
                      }
                     while (   bop_found
                            && !logline_finished)
                      {
                        j = 0;
                        tmp = strchr(logline, LOG_EOP);
                        if (tmp != (char *)NULL)
                         {
                           j = (int)(tmp - logline);
                           eop_found = TRUE;
                         }
                        else
                          eop_found = FALSE;
                        if (   eop_found
                            && delete_next_bracket)
                         {
                           strcpy(logline, tmp+1);
                           delete_next_bracket = FALSE;
                           j = 0;
                           tmp = strchr(logline, LOG_EOP);
                           if (tmp != (char *)NULL)
                            {
                              j = (int)(tmp - logline);
                              eop_found = TRUE;
                            }
                           else
                             eop_found = FALSE;
                           if (bop_found)
                            {
                              i = 0;
                              tmp = strchr(logline, LOG_BOP);
                              if (tmp != (char *)NULL)
                                i = (int)(tmp - logline);
                            }
                           else
                             logline_finished = TRUE;
                         }
                        if (   eop_found
                            && !logline_finished)
                         {
                           if (j > i+1)
                            {
                              if ((Uint)pages >= pages_max)
                               {
                                 /*
                                    Resize the `document_pages[]' table.
                                 */
                                 pages_max <<= 1;
                                 if (pages_max*sizeof(char *) > testval)
                                   pages_max--;
                                 document_pages = (char **)my_realloc ((VOID_PTR)document_pages,
                                                                       pages_max*sizeof(char *),
                                                                       ERR_NO_MEMORY_AVAILABLE,
                                                                       __FILE__, ((long)__LINE__)-3L);
                               }
                              document_pages[pages] = (char *)my_malloc (j-i,
                                                                         ERR_NO_MEMORY_AVAILABLE,
                                                                         __FILE__, ((long)__LINE__)-2L);
                              strncpy(document_pages[pages], logline+i+1, j-i-1);
                              document_pages[pages][j-i-1] = '\0';
                              if (check_log_pagenumber (document_pages[pages]))
                                pages++;
                              else
                                free(document_pages[pages]);
                              strncpy(logline, logline+j+1, strlen(logline)-j);
                              i = 0;
                              tmp = strchr(logline, LOG_BOP);
                              if (tmp != (char *)NULL)
                               {
                                 i = (int)(tmp - logline);
                                 bop_found = TRUE;
                               }
                              else
                                bop_found = FALSE;
                              if (   !*logline
                                  && !bop_found)
                                logline_finished = TRUE;
                            }
                           else
                            {
                              bop_found=eop_found = FALSE;
                              strcpy(logline, logline+j+1);
                            }
                         }
                        else
                          if (bop_found)
                           {
                             if (   logline[i+1]
                                 && !eop_found)
                              {
                                tmp = logline + i + 1;
                                if ((Uint)pages >= pages_max)
                                 {
                                   /*
                                      Resize the `document_pages[]' table.
                                   */
                                   pages_max <<= 1;
                                   if (pages_max*sizeof(char *) > testval)
                                     pages_max--;
                                   document_pages = (char **)my_realloc ((VOID_PTR)document_pages,
                                                                          pages_max*sizeof(char *),
                                                                          ERR_NO_MEMORY_AVAILABLE,
                                                                          __FILE__, ((long)__LINE__)-3L);
                                 }
                                document_pages[pages] = (char *)my_malloc (strlen(tmp)+1,
                                                                           ERR_NO_MEMORY_AVAILABLE,
                                                                           __FILE__, ((long)__LINE__)-2L);
                                strcpy(document_pages[pages], tmp);
                                if (eop_found)
                                 {
                                   if (check_log_pagenumber (document_pages[pages]))
                                     pages++;
                                   bop_found = FALSE;
                                   logline_finished=delete_next_bracket = TRUE;
                                 }
                                else
                                 {
                                   while (   !eop_found
                                          && !feof(f))
                                    {
                                      if (fgets(logline, LINESIZE, f) != (char *)NULL)
                                       {
                                         tmp = strchr(logline, '\n');
                                         if (tmp != (char *)NULL)
                                           *tmp = '\0';
                                       }
                                      j = 0;
                                      tmp = strchr(logline, LOG_EOP);
                                      if (tmp != (char *)NULL)
                                       {
                                         j = (int)(tmp - logline);
                                         eop_found = TRUE;
                                       }
                                      else
                                     eop_found = FALSE;
                                    }
                                   if (eop_found)
                                    {
                                      logline[j] = '\0';
                                      strcat(document_pages[pages], logline);
                                      strcpy(logline, logline+j+1);
                                      if (check_log_pagenumber (document_pages[pages]))
                                        pages++;
                                      logline_finished = TRUE;
                                      bop_found=read_logline = FALSE;
                                    }
                                 }
                              }
                             else
                               logline_finished = TRUE;
                           }
                      }
                     if (   delete_next_bracket
                         && eop_found)
                          delete_next_bracket = FALSE;
                   }
                }
             }
            if (fclose(f) == EOF)
              (void)my_error (stderr, ERR_CLOSE_FILE, fn, 0L);
            if (!pages)
             {
               last_error = my_error (stderr, ERR_CORRUPTED_FILE, fn, 0L);
               if (dvi_id_text != (char *)NULL)
                 free(dvi_id_text);
             }
            else
              if (!the_options->total_only)
               {
                 register int  rows=1;


                 if (!the_options->no_info)
                  {
                    if (   is_tty
                        && is_error)
                      S_NEWLINE(stdout);
                    fprintf(stdout, "%s %s  ---  %s\n", PRGR_NAME, VERSION_NO, COPYRIGHT_TXT);
                    fputs("------------------------------------------------------------\n", stdout);
#if USE_DE
                    if (checkmode == check_dvi)
                     {
                       i = 0;
                       while (isspace(*(dvi_id_text + i)))
                         i++;
                       fprintf(stdout, "Dokumentdatei          = %s (%s)\n", fn, dvi_id_text+i);
                       free(dvi_id_text);
                       if (   the_options->font_info
                           && fonts)
                        {
                          rows <<= 2;
                          for (i=0 ; i < fonts ; i++)
                           {
                             dvi_id_text = strrchr(used_fonts[i], ' ');
                             *dvi_id_text = '\0';
                             tmp = strchr(used_fonts[i], ' ');
                             *tmp = '\0';
                             fprintf(stdout, "Benutzter Zeichensatz  = %-10s skaliert %5d, crc=%s\n",
                                     used_fonts[i], atoi(tmp+1), dvi_id_text+1);
                             free(used_fonts[i]);
                             if (is_tty)
                               if (   the_options->pager
                                   && (++rows >= SCREEN_ROWS))
                                {
                                  GETKEY(stdin, stdout, "%s: Weiter mit <Return>, <%s> zum Beenden...", prgr_name, QUITCHAR);
                                  rows = 1;
                                }
                           }
                        }
                     }
                    else
                      fprintf(stdout, "Dokumentdatei          = %s\n", fn);
                    fprintf(stdout, "Seitenanzahl insgesamt = %d\n", pages);
                    fputs("Im Dokument vorhandene Seitennummern:\n", stdout);
                    fputs("-------------------------------------\n", stdout);
#else /* !USE_DE */
                    if (checkmode == check_dvi)
                     {
                       i = 0;
                       while (isspace(*(dvi_id_text + i)))
                         i++;
                       fprintf(stdout, _("Document file      = %s (%s)\n"), fn, dvi_id_text+i);
                       free(dvi_id_text);
                       if (   the_options->font_info
                           && fonts)
                        {
                          rows <<= 2;
                          for (i=0 ; i < fonts ; i++)
                           {
                             dvi_id_text = strrchr(used_fonts[i], ' ');
                             *dvi_id_text = '\0';
                             tmp = strchr(used_fonts[i], ' ');
                             *tmp = '\0';
                             fprintf(stdout, _("Used font          = %-10s scaled %5d, crc=%s\n"),
                                     used_fonts[i], atoi(tmp+1), dvi_id_text+1);
                             free(used_fonts[i]);
                             if (is_tty)
                               if (   the_options->pager
                                   && (++rows >= SCREEN_ROWS))
                                {
                                  GETKEY(stdin, stdout, _("%s: <Return> for more, <%s> to quit..."), prgr_name, QUITCHAR);
                                  rows = 1;
                                }
                           }
                        }
                     }
                    else
                      fprintf(stdout, _("Document file      = %s\n"), fn);
                    fprintf(stdout, _("Total page numbers = %d\n"), pages);
                    fputs(_("Page numbers existing in document:\n"), stdout);
                    fputs(_("----------------------------------\n"), stdout);
#endif /* !USE_DE */
                  }
                 last_error = EXIT_SUCCESS;
                 fn = (char *)NULL;
                 i=j = 0;
                 if (!the_options->no_info)
                  {
                    if (   the_options->font_info
                        && (rows != 1))
                      rows += 3;
                    else
                      if (rows == 1)
                       {
                         rows <<= 3;
                         rows--;
                       }
                    if (rows >= SCREEN_ROWS)
                      rows = 1;
                  }
                 if (column_width != COLUMN_WIDTH)
                   column_width++;
                 while (i < pages)
                  {
                    if (the_options->columns == COLUMNS_MIN)
                      fprintf(stdout, "%s ", document_pages[i]);
                    else
                     {
                       if (the_options->columns == COLUMNS_MIN+1)
                         fputs(document_pages[i], stdout);
                       else
                         fprintf(stdout, "%*s", column_width, document_pages[i]);
                       if (++j == the_options->columns)
                        {
                          j = 0;
                          fputc('\n', stdout);
                          if (is_tty)
                            if (   the_options->pager
                                && (++rows >= SCREEN_ROWS))
                             {
#if USE_DE
                               GETKEY(stdin, stdout, "%s: Weiter mit <Return>, <%s> zum Beenden...", prgr_name, QUITCHAR);
#else /* !USE_DE */
                               GETKEY(stdin, stdout, _("%s: <Return> for more, <%s> to quit..."), prgr_name, QUITCHAR);
#endif /* !USE_DE */
                               rows = 1;
                             }
                        }
                     }
                    free(document_pages[i]);
                    i++;
                  }
               }
              else
               {
                 free(dvi_id_text);
                 last_error = EXIT_SUCCESS;
                 fn = (char *)NULL;
                 i=j = 0;
                 while (i < pages)
                  {
                    free(document_pages[i]);
                    i++;
                  }
                 fprintf(stdout, "%d", pages);
               }
          }
       }
      if (fn == (char *)NULL)
        is_error = FALSE;
      else
        is_error = TRUE;
      if (argv[++argc] != (char *)NULL)
       {
         free(fn);
         if (!is_error)
          {
            if (   j
                || the_options->columns == COLUMNS_MIN)
              fputc('\n', stdout);
            fputc('\n', stdout);
          }
       }
    }
   if (   !is_error
       && (   j
           || the_options->columns == COLUMNS_MIN))
     fputc('\n', stdout);

  return(last_error);
}
