
IMPLEMENTATION MODULE VDUInterface;

(* 
   Author:         Andrew Trevorrow
   Implementation: University of Hamburg Modula-2 under VAX/VMS version 4
   Date Started:   August, 1984

   Description:
   This module initializes the generic VDU routines and parameters according
   to the vdu value set by DCLInterface.
   We do this here so that new VDUs can be added without having to
   recompile DVItoVDU (search for "XXX" below).

   Revised:
   May, 1985
 - All terminal i/o now done with routines from ScreenIO.

   June, 1985
 - Import vdu from DCLInterface; no longer need GetVDUType.

   September, 1985
 - Added code for REGIS VDUs.
 - Changed VT200 to VT220.
 - Added InitTeXtoASCII.

   March, 1986
 - Added code for new VT220 driver (from John Mann, Monash University)
 - Added code for VIS240/241 (modifies some REGIS parameters).

   November, 1986
 - Updated InitTeXtoASCII to convert 200C..377C to '?'.
 - VT200 is now a synonym for VT220.

   November, 1988 (Niel Kempson <rmcs-tex@uk.ac.cranfield.cdvc>,
		  RMCS Shrivenham, SWINDON SN6 8LA, United Kingdom)
 - Added code for SG480 VDU.
 - Added code for VTTek VDU.
 - VT300, VT320 are now synonyms for VT220.
 - VT330, VT340 are now synonyms for ReGIS.
 - Converted string comparison to be case insensitive.

    December, 1988
  - Incorporate changes made by Phil Taylor (CHAA006@UK.AC.RHBNC.VAXB).
    i.e. added code for Tektronix VDUs & BBC micro running TERMULATOR.
*)

  FROM ScreenIO IMPORT 
                       ReadString,
                       Write, WriteString, WriteLn,
                       Halt;

  FROM DCLInterface IMPORT 
                           stringvalue,
			   Cap,
                           vdu;
  (* /VDU value; already translated if it was a logical name *)

  FROM AED512VDU IMPORT 
                        screenht,
                        (* AED screen can be 512 or 483 pixels high *)
                        InitAED512;
  (* assigns AED512 specific values to generic VDU variables *)

  FROM ANSIVDU IMPORT 
                      InitANSI;
  (* assigns ANSI specific values to generic VDU variables   *)

  FROM BBCVDU IMPORT 
                      InitBBC;
  (* assigns BBC micro specific values to generic VDU variables   *)

  FROM REGISVDU IMPORT 
                       InitREGIS;
  (* assigns REGIS specific values to generic VDU variables  *)

  FROM SG480VDU IMPORT 
                       InitSG480;
  (* assigns SG480 specific values to generic VDU variables  *)

  FROM TektronixVDU IMPORT 
                       InitTektronix;
  (* assigns Tektronix specific values to generic VDU variables  *)

  FROM VIS500VDU IMPORT 
                        InitVIS500;
  (* assigns VIS500 specific values to generic VDU variables *)

  FROM VIS550VDU IMPORT 
                        InitVIS550;
  (* assigns VIS550 specific values to generic VDU variables *)

  FROM VT640VDU IMPORT 
                       InitVT640;
  (* assigns VT640 specific values to generic VDU variables  *)

  FROM VT220VDU IMPORT 
                       InitVT220;
  (* assigns VT220 specific values to generic VDU variables  *)

  FROM VTTekVDU IMPORT 
                       InitVTTek;
  	   (* assigns VT200 tektronix  specific values to generic VDU variables  *)

(* Add other VDU specific modules as more terminals are implemented.
FROM XXXVDU IMPORT
   InitXXX;
*)

(******************************************************************************)

  PROCEDURE InitTeXtoASCII;

(* Initialize TeXtoASCII array used in specific ShowChar/Rectangle routines
   to map a given TeX char into a similar, displayable ASCII char.
*)

    VAR
      ch : CHAR;

  BEGIN
    FOR ch :=   0C TO  17C DO
      TeXtoASCII[ch] := '?'
    END; (* Greeks and ligatures *)
    TeXtoASCII[20C] := 'i';                              (* dotless i *)
    TeXtoASCII[21C] := 'j';                              (* dotless j *)
    TeXtoASCII[22C] := '`';                              (* grave accent *)
    TeXtoASCII[23C] := "'";                              (* acute accent *)
    FOR ch :=  24C TO  27C DO
      TeXtoASCII[ch] := '~'
    END; (* high accents *)
    TeXtoASCII[30C] := ',';                              (* cedilla *)
    FOR ch :=  31C TO  40C DO
      TeXtoASCII[ch] := '?'
    END; (* diphthongs, foreigns *)
    FOR ch :=  41C TO 133C DO
      TeXtoASCII[ch] := ch
    END; (* same: !..0..9..A..[ *)
    TeXtoASCII[134C] := '"';                             (* open double quote *)
    TeXtoASCII[135C] := ']';                             (* same *)
    FOR ch := 136C TO 137C DO
      TeXtoASCII[ch] := '^'
    END; (* more high accents *)
    FOR ch := 140C TO 172C DO
      TeXtoASCII[ch] := ch
    END; (* same: `..z *)
    FOR ch := 173C TO 174C DO
      TeXtoASCII[ch] := '-'
    END; (* en dash, em dash *)
    FOR ch := 175C TO 177C DO
      TeXtoASCII[ch] := '~'
    END; (* more high accents *)
    FOR ch := 200C TO 377C DO
      TeXtoASCII[ch] := '?'
    END; (* the rest *)
  END InitTeXtoASCII;

(******************************************************************************)

  PROCEDURE StringsEqual (s1, s2 : ARRAY OF CHAR) : BOOLEAN;

(* Returns TRUE iff s1 = s2. 
   Comparison is based on the underlying order of character codes.
   E.g., for the ASCII code, ' ' < '0'..'9' < 'A'..'Z' < 'a'..'z'.

   NOTE: string comparison is case insensitive.
*)

    VAR
      i, s1length, s2length : CARDINAL;

  BEGIN
(* SYSDEP: LEN assumes end of string is first NULL, or string is full *)
    s1length := LEN(s1);
    s2length := LEN(s2);
    IF s1length <> s2length THEN
      RETURN FALSE
    ELSE 
      i := 0;
      WHILE (i < s1length) DO   (* s1length = s2length *)
        IF Cap(s1[i]) <> Cap(s2[i]) THEN
          RETURN FALSE
        ELSE 
          INC(i)
        END
      END;
      RETURN TRUE   (* s1 = s2 *)
    END
  END StringsEqual;

(******************************************************************************)

BEGIN
  InitTeXtoASCII;
  IF StringsEqual(vdu,'AED483') THEN
    screenht := 483;   (* override initial implementation value of 512 *)
    InitAED512;        (* uses screenht, so must be called after above line *)
  ELSIF StringsEqual(vdu,'AED512') THEN
    InitAED512;
  ELSIF StringsEqual(vdu,'ANSI')  OR
        StringsEqual(vdu,'VT100') THEN
    InitANSI;
  ELSIF StringsEqual(vdu,'BBC') THEN
    InitBBC;
  ELSIF StringsEqual(vdu,'SG480') THEN
    InitSG480;
  ELSIF StringsEqual(vdu,'REGIS') OR
        StringsEqual(vdu,'VT240') THEN
    InitREGIS;
(* SYSDEP: had to split to avoid compiler limit on number of OR branches! *)
  ELSIF StringsEqual(vdu,'GIGI')  OR
        StringsEqual(vdu,'VK100') OR
        StringsEqual(vdu,'VT125') THEN
    InitREGIS;
  ELSIF StringsEqual(vdu,'TEKTRONIX') THEN
    InitTektronix;
  ELSIF StringsEqual(vdu,'VT100132') THEN
    InitANSI;          (* it is up to user to set VDU to 132 columns *)
    windowwd := 132;   (* just increase window width *)
  ELSIF StringsEqual(vdu,'VIS240') OR
        StringsEqual(vdu,'VIS241') THEN
    InitREGIS;
   (* the VIS240/241 has more text lines and a bigger window region: *)
    bottoml := 29;
    windowwd := 800;
    windowht := 580 - windowv;
  ELSIF StringsEqual(vdu,'VIS500') THEN
    InitVIS500;
  ELSIF StringsEqual(vdu,'VIS550') THEN
    InitVIS550;
  ELSIF StringsEqual(vdu,'VT220') OR
        StringsEqual(vdu,'VT200') OR
        StringsEqual(vdu,'VT300') OR
        StringsEqual(vdu,'VT320') THEN
    InitVT220;
  ELSIF StringsEqual(vdu,'VT640') THEN
    InitVT640;
  ELSIF StringsEqual(vdu,'VTTEK') THEN
    InitVTTek;
(* add a new VDU here (keep order alphabetical)
ELSIF StringsEqual(vdu,'XXX') THEN
   InitXXX;
*)
  ELSE 
    WriteString('Bad /VDU value! (=');
    WriteString(vdu);
    Write(')');
    WriteLn;
    WriteString('Specify one of the following:');
    WriteLn;
    WriteString('   AED483    (AED 512 with 512 by 483 screen)');
    WriteLn;
    WriteString('   AED512    (AED 512 with 512 by 512 screen)');
    WriteLn;
    WriteString('   ANSI      (any ANSI compatible VDU; synonym = VT100)');
    WriteLn;
    WriteString('   BBC       (BBC model B running Termulator)');
    WriteLn;
    WriteString('   REGIS     (any REGIS compatible VDU; synonyms = GIGI,');
    WriteLn;
    WriteString('              VK100, VT125, VT240, VT241, VT330, VT340)');
    WriteLn;
    WriteString('   SG480     (VT100 with Selanar graphics)');
    WriteLn;
    WriteString('   Tektronix (4006, 4010, 4014 etc.)');
    WriteLn;
    WriteString('   VIS240    (VISUAL 240; synonym = VIS241)');
    WriteLn;
    WriteString('   VIS500    (VISUAL 500)');
    WriteLn;
    WriteString('   VIS550    (VISUAL 550)');
    WriteLn;
    WriteString('   VT100132  (any VT100 compatible VDU in 132 column mode)');
    WriteLn;
    WriteString('   VT220     (VT220 using down-loaded chunky graphics; ');
    WriteLn;
    WriteString('              synonyms = VT200, VT300, VT320)');
    WriteLn;
    WriteString('   VT640     (VT100 with Retro-Graphics)');
    WriteLn;
    WriteString('   VTTek     (VT200/300 series in Tektronix mode)');
    WriteLn;
   (* add a new VDU here (keep order alphabetical)
   WriteString('   XXX');
   WriteLn;
   *)
    Halt(2);
  END;
END VDUInterface.
