(* Routines and data structures for use in a TeX82 DVI translator.
   Much of the code in DVIReader is based on DVITYPE 2.7 by Donald Knuth.
   DVITYPE is a program for verifying a DVI file and also serves as a model
   for other DVI-reading programs.  See the TeXWARE manual by Knuth for a
   complete description of DVITYPE and the format of DVI files.
   For efficiency reasons we assume the given DVI file is formatted correctly;
   it is the job of DVITYPE to diagnose bad DVI files.
*)

#include 'globals.h';
#include 'files.h';
#include 'dvireader.h';
#include 'screenio.h';
#include 'fontreader.h';   (* only PixelTableRoutine *)
#include 'options.h';      (* hoffset and voffset *)

(*******************************************************************************
   DECLARATIONS FOR ACCESSING BYTES IN A DVI FILE

   A DVI file is a stream of 8-bit bytes.  Once opened, we can access any byte
   by setting DVIoffset to the required position and then calling GetDVIByte.
*)

VAR
   DVIfile : integer;                 (* DVI file descriptor                  *)
   DVIoffset : INTEGER;               (* current byte offset in DVIfile       *)
   currDVIbuff : INTEGER;             (* starting byte offset in buffer       *)
   DVIbuffer : buffer;                (* input buffer                         *)
   postpostid : INTEGER;              (* offset of postpost's id byte         *)

(*******************************************************************************
   DECLARATIONS FOR GETTING TO A DVI PAGE

   The user can select a particular page by specifying a DVI page
   number (from 1 to totalpages), or a TeX page number (based on the
   values of \count0,\count1,...,\count9), or simply requesting the next page
   in the DVI file (which depends on whether we are ascending or not).
   We will often need to follow the DVI backpointers to locate the bop byte
   of a selected page.
*)

VAR
   curreop,                     (* position of eop byte of current page   *)
   currbop,                     (* position of bop byte of current page   *)
   lastbop : INTEGER;           (* position of last bop byte              *)
   prevbop : INTEGER;           (* position of bop byte of previous page;
                                   note that prevbop of first page = -1   *)

(*******************************************************************************
   DECLARATIONS FOR INTERPRETING A DVI PAGE

   The commands between the bop and eop bytes for a particular page need to be
   translated (based on the method used by DVITYPE) before we can determine the
   the position and shape of all rules on that page, as well as the position
   of all characters and which fonts they belong to.
*)

CONST
   (* Use symbolic names for the opcodes of DVI commands:                     *)
   setchar0 = 0;         (* setchar1..setchar127 = 1..127                     *)
   set1     = 128;       (* set2,set3,set4 = 129,130,131                      *)
   setrule  = 132;
   put1     = 133;       (* put2,put3,put4 = 134,135,136                      *)
   putrule  = 137;
   nop      = 138;
   bop      = 139;
   eop      = 140;
   push     = 141;
   pop      = 142;
   right1   = 143;  w0 = 147;  x0 = 152;  down1 = 157;  y0 = 161;  z0 = 166;
   right2   = 144;  w1 = 148;  x1 = 153;  down2 = 158;  y1 = 162;  z1 = 167;
   right3   = 145;  w2 = 149;  x2 = 154;  down3 = 159;  y2 = 163;  z2 = 168;
   right4   = 146;  w3 = 150;  x3 = 155;  down4 = 160;  y3 = 164;  z3 = 169;
                    w4 = 151;  x4 = 156;                y4 = 165;  z4 = 170;
   fntnum0  = 171;       (* fntnum1..fntnum63 = 172..234                      *)
   fnt1     = 235;       (* fnt2,fnt3,fnt4 = 236,237,238                      *)
   xxx1     = 239;       (* xxx2,xxx3,xxx4 = 240,241,242                      *)
   fntdef1  = 243;       (* fntdef2,fntdef3,fntdef4 = 244,245,246             *)
   pre      = 247;
   post     = 248;
   postpost = 249;
   (* undefined commands = 250..255 *)

   maxstacksize  = 100;              (* maximum stack size for state values   *)
   maxstacksizem = maxstacksize - 1;
   maxdrift      = 2;                (* prevent hh & vv from drifting too far *)
   maxint        = 2147483647;       (* 2^31 - 1                              *)

VAR
   DVIcommand : 0..255;              (* holds next DVI command                *)
   maxstack,                         (* max pushes over pops in DVI file      *)
   num,                              (* DVI numerator                         *)
   den : INTEGER;                    (* DVI denominator                       *)
   conv : REAL;                      (* converts DVI units to pixels          *)
   h, v,                             (* current pos on page in DVI units      *)
   w, x,                             (* horizontal increments in DVI units    *)
   y, z,                             (* vertical increments in DVI units      *)
   hh, vv : INTEGER;                 (* h and v in pixels (approx)            *)
   hhh, vvv : INTEGER;               (* h and v rounded to nearest pixel      *)
   hstack, vstack,                   (* push down stacks for state values     *)
   wstack, xstack,
   ystack, zstack,
   hhstack, vvstack : ARRAY [0..maxstacksizem] OF INTEGER;
   stackpos : 0..maxstacksize;       (* stacks empty when stackpos = 0,
                                        i.e., top of stacks = stackpos - 1    *)
   fontspace  : INTEGER;             (* used in DoRight and DoDown            *)
   thisrule   : ruleinfoptr;         (* temporary pointer to node in rulelist *)
   thischar   : charinfoptr;         (* temporary pointer to node in charlist *)
   thisspecial : specialinfoptr;     (* temporary ptr to node in speciallist  *)


(******************************************************************************)

(* Here are the functions used to get byte/s from DVIfile.
   They are essentially the same as those used in DVITYPE.
*)

FUNCTION GetDVIByte : INTEGER;

(* Returns the value (unsigned) of the byte at DVIoffset and
   advances DVIoffset for the next GetDVIByte.
*)

VAR buffstart, result : INTEGER;

BEGIN
buffstart := (DVIoffset DIV bufflen) * bufflen;   (* 0, bufflen, 2*bufflen... *)
IF buffstart <> currDVIbuff THEN BEGIN
   currDVIbuff := buffstart;
   result := lseek(DVIfile, buffstart, 0);
   { DEBUG
     IF result <> buffstart THEN BEGIN
        writeln('Lseek failed in GetDVIByte!'); RestoreTerminal; exit(1);
     END;
   GUBED }
   result := read(DVIfile, DVIbuffer, bufflen);
   { DEBUG
     IF result = -1 THEN BEGIN
        writeln('Read failed in GetDVIByte!'); RestoreTerminal; exit(1);
     END;
   GUBED }
END;
GetDVIByte := ORD(DVIbuffer[DVIoffset - buffstart]);
DVIoffset := DVIoffset + 1;
END; (* GetDVIByte *)

(******************************************************************************)

FUNCTION SignedDVIByte : INTEGER;      (* the next byte, signed *)

VAR b : INTEGER;

BEGIN
b := GetDVIByte;
IF b < 128 THEN
   SignedDVIByte := b
ELSE
   SignedDVIByte := b - 256;
END; (* SignedDVIByte *)

(******************************************************************************)

FUNCTION GetTwoDVIBytes : INTEGER;     (* the next 2 bytes, unsigned *)

VAR a, b : INTEGER;

BEGIN
a := GetDVIByte;
b := GetDVIByte;
GetTwoDVIBytes := a * 256 + b;
END; (* GetTwoDVIBytes *)

(******************************************************************************)

FUNCTION SignedDVIPair : INTEGER;      (* the next 2 bytes, signed *)

VAR a, b : INTEGER;

BEGIN
a := GetDVIByte;
b := GetDVIByte;
IF a < 128 THEN
   SignedDVIPair := a * 256 + b
ELSE
   SignedDVIPair := (a - 256) * 256 + b;
END; (* SignedDVIPair *)

(******************************************************************************)

FUNCTION GetThreeDVIBytes : INTEGER;   (* the next 3 bytes, unsigned *)

VAR a, b, c : INTEGER;

BEGIN
a := GetDVIByte;
b := GetDVIByte;
c := GetDVIByte;
GetThreeDVIBytes := (a * 256 + b) * 256 + c;
END; (* GetThreeDVIBytes *)

(******************************************************************************)

FUNCTION SignedDVITrio : INTEGER;      (* the next 3 bytes, signed *)

VAR a, b, c : INTEGER;

BEGIN
a := GetDVIByte;
b := GetDVIByte;
c := GetDVIByte;
IF a < 128 THEN
   SignedDVITrio := (a * 256 + b) * 256 + c
ELSE
   SignedDVITrio := ((a - 256) * 256 + b) * 256 + c;
END; (* SignedDVITrio *)

(******************************************************************************)

FUNCTION SignedDVIQuad : INTEGER;      (* the next 4 bytes, signed *)

TYPE int_or_bytes = RECORD
                    CASE b : BOOLEAN OF
                       TRUE  : (int : INTEGER);
                       FALSE : (byt : PACKED ARRAY [0..3] OF CHAR);
                    END;

VAR w : int_or_bytes;

BEGIN
WITH w DO BEGIN
   w.byt[0] := CHR(GetDVIByte);
   w.byt[1] := CHR(GetDVIByte);
   w.byt[2] := CHR(GetDVIByte);
   w.byt[3] := CHR(GetDVIByte);
END;
SignedDVIQuad := w.int;
END; (* SignedDVIQuad *)

(******************************************************************************)

PROCEDURE ProcessPostamble;

(* Having successfully opened the DVI file, we find the postamble
   and initialize these global variables:
   lastbop, num, den, DVImag, maxstack, totalpages.
   The font definitions are read by ProcessFontDefs.
*)

VAR postamblepos, postamble, pagehtplusdp, pagewidth : INTEGER;

BEGIN
DVIoffset    := postpostid - 4;
postamblepos := SignedDVIQuad;   (* get post_post's postamble ptr *)
DVIoffset    := postamblepos;
postamble    := GetDVIByte;
lastbop      := SignedDVIQuad;
num          := SignedDVIQuad;
den          := SignedDVIQuad;
DVImag       := SignedDVIQuad;
pagehtplusdp := SignedDVIQuad;
pagewidth    := SignedDVIQuad;
maxstack     := SignedDVIPair;
totalpages   := SignedDVIPair;
IF maxstack > maxstacksize THEN BEGIN
   writeln('Stack capacity exceeded!');
   RestoreTerminal; exit(1);
   (* now we don't need to test for stack overflow in DoPush *)
END;
{ DEBUG
   writeln('postamble opcode      = ', postamble:1);
   writeln('postion of last bop   = ', lastbop:1);
   writeln('num                   = ', num:1);
   writeln('den                   = ', den:1);
   writeln('DVI mag               = ', DVImag:1);
   writeln('ht+dp of tallest page = ', pagehtplusdp:1);
   writeln('width of widest page  = ', pagewidth:1);
   writeln('max stack depth       = ', maxstack:1);
   writeln('total # of pages      = ', totalpages:1);
GUBED }
END; (* ProcessPostamble *)

(******************************************************************************)

PROCEDURE ProcessFontDefs;

(* Read the fntdef commands in the postamble (fntdef commands in the DVI
   pages will be skipped) and store the information in the font list.
   (Note that complete fontspecs are NOT built here because DVIReader does not
   want to know about the format or naming conventions of font files.)
   Since ProcessPostamble ended by reading the totalpages parameter, the
   next GetDVIByte should return nop or first fntdef.
*)

VAR f, c, s, d, a, l : INTEGER;        (* hold fntdef parameters *)
    i : INTEGER;     ch : CHAR;        (* for getting farea and fname *)
    farea, fname : string;             (* a and l bytes long respectively *)

BEGIN
totalfonts := 0;     (* number of nodes in font list *)
fontlist   := NIL;
REPEAT
   DVIcommand := GetDVIByte;
   IF (DVIcommand >= fntdef1) AND (DVIcommand <= fntdef1+3) THEN BEGIN
      CASE DVIcommand - fntdef1 OF
         0 : f := GetDVIByte;
         1 : f := GetTwoDVIBytes;
         2 : f := GetThreeDVIBytes;
         3 : f := SignedDVIQuad
      END;
      c := SignedDVIQuad;   (* checksum; ignore it *)
      s := SignedDVIQuad;   (* scaled size *)
      d := SignedDVIQuad;   (* design size *)
      a := GetDVIByte;      (* length of font area *)
      l := GetDVIByte;      (* length of font name *)
      farea := '';          (* initialize with blanks *)
      FOR i := 0 TO a-1 DO BEGIN    (* read and store font area *)
         ch := CHR(GetDVIByte);
         IF i < maxfontspec THEN farea[i] := ch;
      END;
      fname := '';
      FOR i := 0 TO l-1 DO BEGIN    (* read and store font name *)
         ch := CHR(GetDVIByte);
         IF i < maxfontspec THEN fname[i] := ch;
      END;
      NEW(currfont);
      WITH currfont^ DO BEGIN
         fontused    := FALSE;
         fontnum     := f;
         scaledsize  := s;
         designsize  := d;
         fontarea    := farea; fontarealen := a;
         fontname    := fname; fontnamelen := l;
         fontspec    := '';
         fontspeclen := 0;        (* fontspec is built in FontReader *)
         fontexists  := FALSE;    (* becomes TRUE if fontspec can be opened *)
         totalchars  := 0;
         charlist    := NIL;      (* first node allocated in DoFont *)
         chartail    := NIL;      (* nodes are added to tail of char list *)
         pixelptr    := NIL;      (* allocated once per font; see DoFont *)
         nextfont    := fontlist;
      END;
      fontlist := currfont;       (* add new font to head of list *)
      totalfonts := totalfonts + 1;
   END
   ELSE IF DVIcommand = nop THEN
      (* nop commands can occur between DVI commands *)
   ELSE IF DVIcommand = postpost THEN
      (* we have reached end of postamble *)
   ELSE BEGIN
      writeln('Unexpected DVI command in postamble = ', DVIcommand:1);
      RestoreTerminal; exit(1);
   END;
UNTIL DVIcommand = postpost;
END; (* ProcessFontDefs *)

(******************************************************************************)

PROCEDURE OpenDVIFile (name : string);

(* If the given file can be opened and is a valid TeX82 DVI file then
   the following global variables are initialized:

      DVImag      := magnification value stored in DVI file (TeX's \mag)
      totalpages  := total number of pages in DVI file
      currDVIpage := 0      (and remains so until a page is selected)
      currTeXpage := ten 0s (ditto)
      totalfonts  := total number of fonts in DVI file (= nodes in font list)
      fontlist^.  (nodes are added to head of list)
         fontused   := FALSE
         fontnum    := internal DVI font number
         scaledsize := scaled size of font (in DVI units)
         designsize := design size of font (in DVI units)
         fontarea   := a string of min(fontarealen,maxfontspec) characters
         fontname   := a string of min(fontnamelen,maxfontspec) characters
         fontspec   := a null string (fontspeclen := 0)
         fontexists := FALSE
         totalchars := 0
         charlist   := NIL
         chartail   := NIL
         pixelptr   := NIL
         nextfont   := next node in font list (if not NIL)
*)

LABEL 666, 777, 888;

VAR length, i : integer;

BEGIN
currDVIbuff := -1;   (* impossible value for first GetDVIByte *)
length := 0;
WHILE length < maxstring DO BEGIN
   IF name[length] = ' ' THEN goto 888;
   length := length + 1;
END;
888:
IF length < maxstring THEN name[length] := CHR(0);   (* terminate with NULL *)
DVIfile := open(name, O_RDONLY, 0);
IF length < maxstring THEN name[length] := ' ';      (* restore space *)
IF DVIfile >= 0 THEN BEGIN
   (* get offset of last DVI byte *)
   DVIoffset := lseek(DVIfile, 0, 2);
   IF DVIoffset = -1 THEN BEGIN
      writeln('Failed to find end of file!'); RestoreTerminal; exit(1);
   END;
   IF DVIoffset = 0 THEN BEGIN
      writeln('File is empty!'); RestoreTerminal; exit(1);
   END;
   { DEBUG
     writeln('total bytes = ', DVIoffset:1);
   GUBED }
   DVIoffset := DVIoffset - 1;
   WHILE TRUE DO BEGIN                          (* skip any NULs *)
      IF GetDVIByte <> 0 THEN goto 777;
      IF DVIoffset = 1 THEN goto 777;           (* start of file! *)
      DVIoffset := DVIoffset - 2;               (* GetDVIByte increments *)
   END;
   777:
   DVIoffset := DVIoffset - 1;
   WHILE TRUE DO BEGIN                          (* skip 223s *)
      IF GetDVIByte <> 223 THEN goto 666;
      IF DVIoffset = 1 THEN goto 666;           (* start of file! *)
      DVIoffset := DVIoffset - 2;               (* GetDVIByte increments *)
   END;
   666:
   DVIoffset := DVIoffset - 1;
   postpostid := DVIoffset;         (* remember offset of id byte *)
   IF GetDVIByte <> 2 THEN BEGIN
      writeln('Not a valid DVI file!'); RestoreTerminal; exit(1);
   END
   ELSE BEGIN
      ProcessPostamble;             (* get DVImag, totalpages, etc *)
      ProcessFontDefs;              (* build and initialize font list *)
      currDVIpage := 0;             (* we haven't processed a page yet *)
      FOR i := 0 TO 9 DO currTeXpage[i] := 0;
   END;
END
ELSE BEGIN
   writeln('Couldn''t open file: ', name:length);
   RestoreTerminal; exit(1);
END;
END; (* OpenDVIFile *)

(******************************************************************************)

PROCEDURE SkipFntdef (which : INTEGER);

(* Read past a fntdef command without interpreting it. *)

VAR dummy, a, l, i : INTEGER;

BEGIN
CASE which OF   (* which = DVIcommand - fntdef1 *)
   0 : dummy := GetDVIByte;
   1 : dummy := GetTwoDVIBytes;
   2 : dummy := GetThreeDVIBytes;
   3 : dummy := SignedDVIQuad
END;
dummy := SignedDVIQuad;
dummy := SignedDVIQuad;
dummy := SignedDVIQuad;
a := GetDVIByte;        (* length of directory *)
l := GetDVIByte;        (* length of font name *)
FOR i := 1 TO l+a DO
   dummy := GetDVIByte;
END; (* SkipFntdef *)

(******************************************************************************)

PROCEDURE ReadFirstBop;

(* Read first bop by skipping past preamble; update currbop and currDVIpage. *)

VAR k, i, dummy : INTEGER;

BEGIN
DVIoffset := 14;            (* position of preamble's k parameter *)
k := GetDVIByte;            (* length of x parameter *)
FOR i := 1 TO k DO
   dummy := GetDVIByte;     (* skip preamble comment *)
REPEAT   (* skip any nops and fntdefs *)
   DVIcommand := GetDVIByte;
   IF (DVIcommand = nop) OR (DVIcommand = bop) THEN
      (* do nothing *)
   ELSE IF (DVIcommand >= fntdef1) AND (DVIcommand <= fntdef1+3) THEN
      SkipFntdef(DVIcommand - fntdef1)
   ELSE BEGIN
      WriteLine;
      WriteString('Unexpected DVI command before first bop=');
      WriteInt(DVIcommand); WriteLine;
      RestoreTerminal; exit(1);
   END;
UNTIL DVIcommand = bop;
currbop := DVIoffset - 1;   (* position in DVI file of first bop *)
currDVIpage := 1;
END; (* ReadFirstBop *)

(******************************************************************************)

PROCEDURE ReadNextBop;

(* We are currently positioned after an eop byte which we know is not the
   last.  This routine positions us after the next bop byte and updates
   currbop and currDVIpage.
*)

BEGIN
REPEAT   (* skip any nops and fntdefs *)
   DVIcommand := GetDVIByte;
   IF (DVIcommand = nop) OR (DVIcommand = bop) THEN
      (* do nothing *)
   ELSE IF (DVIcommand >= fntdef1) AND (DVIcommand <= fntdef1+3) THEN
      SkipFntdef(DVIcommand - fntdef1)
   ELSE BEGIN
      WriteLine;
      WriteString('Unexpected DVI command between eop and bop=');
      WriteInt(DVIcommand); WriteLine;
      RestoreTerminal; exit(1);
   END;
UNTIL DVIcommand = bop;
currbop := DVIoffset - 1;   (* position in DVI file of this bop *)
currDVIpage := currDVIpage + 1;
END; (* ReadNextBop *)

(******************************************************************************)

PROCEDURE ReadBopParameters;

(* We should now be positioned after the bop of desired page, so read
   the 10 TeX counters and update currTeXpage and prevbop.
   At the end of this routine we will be at the byte after currbop's parameters
   and ready to InterpretPage.
*)

VAR i : INTEGER;

BEGIN
FOR i := 0 TO 9 DO
   currTeXpage[i] := SignedDVIQuad;
prevbop := SignedDVIQuad;   (* position of previous bop in DVI file *)
END; (* ReadBopParameters *)

(******************************************************************************)

PROCEDURE MoveToNextPage (ascending : BOOLEAN);

(* MoveToNextPage will select the next page, depending on the current page
   and the specified direction.  If the value of currDVIpage is 0 (set in
   OpenDVIFile), then MoveToNextPage will select the first page if ascending is
   TRUE and the last page if ascending is FALSE.  If currDVIpage is > 0 then
   MoveToNextPage will select currDVIpage+1 if ascending (unless currDVIpage =
   totalpages, in which case it does nothing), or currDVIpage-1 if descending
   (unless currDVIpage = 0).

   Before calling InterpretPage, DVItoVDU must position DVIReader to the
   desired page by calling one of the MoveTo... routines.
   The global variables updated if a page is located by any MoveTo... call are:
      currDVIpage := the current DVI page (1..totalpages)
      currTeXpage := the ten TeX counter values stored with this page
   Note that currDVIpage is initially 0 until one of these routines succeeds.
*)

LABEL 999;

BEGIN
IF (currDVIpage = 1) AND (NOT ascending) THEN
   goto 999
ELSE IF (currDVIpage = totalpages) AND ascending THEN
   goto 999
ELSE IF currDVIpage = 0 THEN
   (* we haven't processed a page yet *)
   IF ascending THEN         (* get first page *)
      ReadFirstBop
   ELSE BEGIN                (* get last page *)
      currbop := lastbop;
      DVIoffset := currbop + 1;
      currDVIpage := totalpages;
   END
ELSE
   IF ascending THEN
      (* currently positioned after eop of currDVIpage, so get next bop *)
      ReadNextBop
   ELSE BEGIN
      (* move to bop pointed to by currbop's backpointer *)
      currbop := prevbop;
      DVIoffset := currbop + 1;   (* move to byte after previous bop *)
      currDVIpage := currDVIpage - 1;
   END;
ReadBopParameters;   (* update currTeXpage and prevbop *)
999:
END; (* MoveToNextPage *)

(******************************************************************************)

PROCEDURE MoveToDVIPage (n : INTEGER);

(* Move to nth DVI page; n should be in 1..totalpages. *)

LABEL 888, 999;

BEGIN
IF (n < 1) OR (n > totalpages) THEN   (* do nothing *)
   goto 999
ELSE IF n = 1 THEN
   (* Note that this test must come before next test so that we avoid any
      problems when currDVIpage initially = 0. *)
   ReadFirstBop
ELSE IF n = currDVIpage + 1 THEN
   ReadNextBop
ELSE BEGIN
   IF n < currDVIpage THEN BEGIN
      currbop := prevbop;   (* start searching backwards from previous page *)
      currDVIpage := currDVIpage - 1;
   END
   ELSE IF n > currDVIpage THEN BEGIN
      currbop := lastbop;   (* start searching backwards from last page *)
      currDVIpage := totalpages;
   END;
   (* if n = currDVIpage we'll just move back to currbop *)
   (* n is now <= currDVIpage so search by following backpointers *)
   WHILE TRUE DO
      IF n = currDVIpage THEN BEGIN
         DVIoffset := currbop + 1;      (* move to byte after currbop *)
         goto 888;
      END
      ELSE BEGIN
         DVIoffset := currbop + 41;     (* move to location of backpointer *)
         currbop := SignedDVIQuad;      (* get location of previous page *)
         currDVIpage := currDVIpage - 1;
      END;
   888:
END;
ReadBopParameters;   (* update currTeXpage and prevbop *)
999:
END; (* MoveToDVIPage *)

(******************************************************************************)

FUNCTION CurrMatchesNew (VAR newTeXpage : TeXpageinfo) : BOOLEAN;

(* Return TRUE iff currTeXpage matches newTeXpage. *)

VAR i : 0..9;

BEGIN
CurrMatchesNew := TRUE;
WITH newTeXpage DO
   FOR i := 0 TO lastvalue DO
      IF present[i] THEN
         IF value[i] <> currTeXpage[i] THEN
            CurrMatchesNew := FALSE;
END; (* CurrMatchesNew *)

(******************************************************************************)

FUNCTION MoveToTeXPage (VAR newTeXpage : TeXpageinfo) : BOOLEAN;

(* MoveToTeXPage will search for the lowest DVI page matching the given TeX page
   specification.  (TeX stores the values of \count0,\count1,...,\count9 with
   every DVI page.  Plain TeX uses \count0 to control page numbering.)
   newTeXpage is a VAR parameter only for efficiency; it won't be changed.
   The value array stores the requested counter values, the present array
   indicates which counters are relevant and lastvalue indicates
   the position (0..9) of the last relevant counter.

   DVItoVDU converts a more friendly representation
   of a TeX page request into the TeXpageinfo format.  For example,
   [2..5] would be converted to:     and [] would be converted to:
   value     = [2,?,5,?,?,?,?,?,?,?]     value     = [?,?,?,?,?,?,?,?,?,?]
   present   = [T,F,T,?,?,?,?,?,?,?]     present   = [F,?,?,?,?,?,?,?,?,?]
   lastvalue =      2                    lastvalue =  0
   MoveToTeXPage returns TRUE iff the requested TeX page is located.
*)

LABEL 999;

VAR savecurrbop, savecurrDVIpage : INTEGER;
    nextbop : INTEGER;
    i : INTEGER;
    atleastone : BOOLEAN;

BEGIN
(* save away current page and DVI position *)
savecurrDVIpage := currDVIpage;
IF currDVIpage <> 0 THEN           (* only if we've processed a page *)
   savecurrbop := currbop;
   (* note that curreop is saved in last InterpretPage *)
(* search backwards through all DVI pages for lowest matching page *)
atleastone := FALSE;
nextbop := lastbop;
FOR i := totalpages DOWNTO 1 DO BEGIN
   DVIoffset := nextbop + 1;
   ReadBopParameters;              (* update currTeXpage and prevbop *)
   IF CurrMatchesNew(newTeXpage) THEN BEGIN
      currbop := nextbop;
      currDVIpage := i;
      atleastone := TRUE;
   END;
   nextbop := prevbop;
END;
IF NOT atleastone THEN BEGIN       (* no match, so restore currDVIpage *)
   currDVIpage := savecurrDVIpage;
   IF currDVIpage <> 0 THEN BEGIN  (* restore page and positioning info *)
      currbop := savecurrbop;
      DVIoffset := currbop + 1;
      ReadBopParameters;           (* restore currTeXpage and prevbop *)
      DVIoffset := curreop + 1;
      (* we should now be after the eop byte of the original page *)
   END;
   MoveToTeXPage := FALSE;
   goto 999;
END
ELSE                               (* we found lowest matching page *)
   DVIoffset := currbop + 1;
ReadBopParameters;                 (* update currTeXpage and prevbop *)
MoveToTeXPage := TRUE;
999:
END; (* MoveToTeXPage *)

(******************************************************************************)

PROCEDURE SetConversionFactor (resolution, magnification : INTEGER);

(* This routine must be called before the first InterpretPage call.
   DVIReader needs to know the resolution and magnification values
   before it attempts to convert DVI units into pixel values.
*)

BEGIN
conv := num/254000.0 * resolution/den * magnification/1000.0;
END; (* SetConversionFactor *)

(******************************************************************************)

PROCEDURE InitStateValues;

(* Initialize state values and stack. *)

BEGIN
hh := hoffset;         (* 0 if no horizontal shift specified *)
vv := voffset;         (* 0 if no vertical shift specified *)
IF hoffset >= 0 THEN
   h :=   TRUNC(hoffset / conv + 0.5)
ELSE
   h := - TRUNC(ABS(hoffset) / conv + 0.5);
IF voffset >= 0 THEN
   v :=   TRUNC(voffset / conv + 0.5)
ELSE
   v := - TRUNC(ABS(voffset) / conv + 0.5);
w := 0; x := 0;
y := 0; z := 0;
stackpos  := 0;
fontspace := 0;        (* for DoRight and DoDown before a DoFont call *)
END; (* InitStateValues *)

(******************************************************************************)

PROCEDURE InitPage;

(* Initialize page so that there are no fonts, chars, rules, specials. *)

BEGIN
(* page edges will change if there is at least one char or rule on page *)
minhp := maxint;
minvp := maxint;
maxhp := -maxint;
maxvp := -maxint;
currfont := fontlist;
WHILE currfont <> NIL DO
   WITH currfont^ DO BEGIN
      IF fontused THEN BEGIN  (* only reset those fonts used in last page *)
         fontused := FALSE;
         totalchars := 0;
         (* deallocate char list completely; DoFont will allocate first node *)
         WHILE charlist <> NIL DO BEGIN
            thischar := charlist;
            charlist := thischar^.nextchar;
            DISPOSE(thischar);
         END;
         chartail := NIL;
         (* pixel table remains allocated *)
      END;
      currfont := nextfont;
   END;
currfont   := NIL;            (* current font is undefined at start of page *)
totalrules := 0;
(* deallocate rule information except for one node (for DoSet/PutRule) *)
WHILE rulelist <> ruletail DO BEGIN
   thisrule := rulelist;
   rulelist := thisrule^.nextrule;
   DISPOSE(thisrule);
END;
rulelist^.rulecount := 0;     (* no rules in this node *)
rulelist^.nextrule  := NIL;
(* deallocate \special information *)
WHILE speciallist <> NIL DO BEGIN
   thisspecial := speciallist;
   speciallist := speciallist^.nextspecial;
   DISPOSE(thisspecial);
END;
END; (* InitPage *)

(******************************************************************************)

FUNCTION PixelRound (DVIunits : INTEGER) : INTEGER;

(* Return the nearest number of pixels in the given DVI dimension. *)

BEGIN
IF DVIunits > 0 THEN
   PixelRound :=   TRUNC(conv * DVIunits + 0.5)
ELSE
   PixelRound := - TRUNC(conv * ABS(DVIunits) + 0.5);
END; (* PixelRound *)

(******************************************************************************)

PROCEDURE DoSetChar (ch : INTEGER);

(* Add char info to current chartable, update our horizontal
   position on the page and check the page edges.
*)

LABEL 999;

BEGIN
WITH currfont^ DO BEGIN
   IF ch > maxTeXchar THEN BEGIN
      WriteString('Unknown character from'); WriteChar(' ');
      WriteString(fontspec);
      goto 999;                           (* ignore ch *)
   END;
   WITH chartail^ DO
      IF charcount = chartablesize THEN BEGIN
         (* allocate a new chartable *)
         NEW(nextchar);                   (* add new node to end of char list *)
         nextchar^.charcount := 0;        (* reset charcount *)
         nextchar^.nextchar  := NIL;
         chartail := nextchar;
      END;
   WITH chartail^ DO BEGIN                (* may be new chartable *)
      WITH chartable[charcount] DO BEGIN
         hp := hh;
         vp := vv;
         code := ch;
      END;
      WITH pixelptr^[ch] DO BEGIN
         (* do page edges increase? *)
         IF vv - yo < minvp THEN minvp := vv - yo;
         IF hh - xo < minhp THEN minhp := hh - xo;
         IF vv + (ht - yo - 1) > maxvp THEN maxvp := vv + (ht - yo - 1);
         IF hh + (wd - xo - 1) > maxhp THEN maxhp := hh + (wd - xo - 1);
         (* the above checks ensure that page edges include all black
            pixels in glyph, but we also want to include reference point *)
         IF hh < minhp THEN minhp := hh;
         IF vv < minvp THEN minvp := vv;
         IF hh > maxhp THEN maxhp := hh;
         IF vv > maxvp THEN maxvp := vv;
         (* add pixel width calculated in PixelTableRoutine *)
         hh := hh + pwidth;
         (* use hhh and maxdrift to prevent hh drifting too far from h *)
         hhh := PixelRound(h + dwidth);
         IF ABS(hhh - hh) > maxdrift THEN
            IF hhh > hh THEN
               hh := hhh - maxdrift
            ELSE
               hh := hhh + maxdrift;
         (* add DVI width calculated in PixelTableRoutine *)
         h := h + dwidth;
      END;
      totalchars := totalchars + 1;
      charcount := charcount + 1;
   END;
END;
999:
END; (* DoSetChar *)

(******************************************************************************)

PROCEDURE DoPutChar (ch : INTEGER);

(* Exactly the same as DoSetChar, but we DON'T update the horizontal
   position on the page.  (We still have to check page edges.)
*)

LABEL 999;

BEGIN
WITH currfont^ DO BEGIN
   IF ch > maxTeXchar THEN BEGIN
      WriteString('Unknown character from'); WriteChar(' ');
      WriteString(fontspec);
      goto 999;                           (* ignore ch *)
   END;
   WITH chartail^ DO
      IF charcount = chartablesize THEN BEGIN
         (* allocate a new chartable *)
         NEW(nextchar);                   (* add new node to end of char list *)
         nextchar^.charcount := 0;        (* reset charcount *)
         nextchar^.nextchar  := NIL;
         chartail := nextchar;
      END;
   WITH chartail^ DO BEGIN                (* may be new chartable *)
      WITH chartable[charcount] DO BEGIN
         hp := hh;
         vp := vv;
         code := ch;
      END;
      WITH pixelptr^[ch] DO BEGIN
         (* do page edges increase? *)
         IF vv - yo < minvp THEN minvp := vv - yo;
         IF hh - xo < minhp THEN minhp := hh - xo;
         IF vv + (ht - yo - 1) > maxvp THEN maxvp := vv + (ht - yo - 1);
         IF hh + (wd - xo - 1) > maxhp THEN maxhp := hh + (wd - xo - 1);
         (* the above checks ensure that page edges include all black
            pixels in glyph, but we also want to include reference point *)
         IF hh < minhp THEN minhp := hh;
         IF vv < minvp THEN minvp := vv;
         IF hh > maxhp THEN maxhp := hh;
         IF vv > maxvp THEN maxvp := vv;
      END;
      totalchars := totalchars + 1;
      charcount := charcount + 1;
   END;
END;
999:
END; (* DoPutChar *)

(******************************************************************************)

PROCEDURE DoPush;

(* Push state values onto stack.
   No need to test for stack overflow since we compare maxstack and
   maxstacksize in ProcessPostamble.
*)

BEGIN
hstack[stackpos] := h;   vstack[stackpos] := v;
wstack[stackpos] := w;   xstack[stackpos] := x;
ystack[stackpos] := y;   zstack[stackpos] := z;
hhstack[stackpos] := hh; vvstack[stackpos] := vv;
stackpos := stackpos + 1;
END; (* DoPush *)

(******************************************************************************)

PROCEDURE DoPop;

(* Pop state values from top of stack. *)

BEGIN
{ DEBUG
  IF stackpos = 0 THEN BEGIN
     WriteLine; WriteString('Stack empty!'); WriteLine;
     RestoreTerminal; exit(1);
  END;
GUBED }
stackpos := stackpos - 1;
h := hstack[stackpos];   v := vstack[stackpos];
w := wstack[stackpos];   x := xstack[stackpos];
y := ystack[stackpos];   z := zstack[stackpos];
hh := hhstack[stackpos]; vv := vvstack[stackpos];
END; (* DoPop *)

(******************************************************************************)

PROCEDURE DoRight (amount : INTEGER);

(* Move the reference point horizontally by given amount (usually +ve).
   When the amount is small, like a kern, hh changes
   by rounding the amount; but when the amount is large, hh changes by rounding
   the true position h so that accumulated rounding errors disappear.
*)

BEGIN
IF (amount < fontspace) AND (amount > -4 * fontspace) THEN BEGIN
   hh  := hh + PixelRound(amount);
   (* use hhh and maxdrift to prevent hh drifting too far from h *)
   hhh := PixelRound(h + amount);
   IF ABS(hhh - hh) > maxdrift THEN
      IF hhh > hh THEN
         hh := hhh - maxdrift
      ELSE
         hh := hhh + maxdrift;
END
ELSE
   hh := PixelRound(h + amount);
h := h + amount;
END; (* DoRight *)

(******************************************************************************)

PROCEDURE DoDown (amount : INTEGER);

(* Move the reference point vertically by given amount (usually +ve).
   Rounding is done similarly to DoRight but with the threshold between
   small and large amounts increased by a factor of 5.
*)

BEGIN
IF ABS(amount) < 5 * fontspace THEN BEGIN
   vv := vv + PixelRound(amount);
   (* use vvv and maxdrift to prevent vv drifting too far from v *)
   vvv := PixelRound(v + amount);
   IF ABS(vvv - vv) > maxdrift THEN
      IF vvv > vv THEN
         vv := vvv - maxdrift
      ELSE
         vv := vvv + maxdrift;
END
ELSE
   vv := PixelRound(v + amount);
v := v + amount;
END; (* DoDown *)

(******************************************************************************)

FUNCTION RulePixels (DVIunits : INTEGER) : INTEGER;

(* Return the number of pixels in the given height or width of a rule
   using the method recommended in DVITYPE.
*)

VAR n : INTEGER;

BEGIN
n := TRUNC(conv * DVIunits);
IF n < conv * DVIunits THEN RulePixels := n + 1 ELSE RulePixels := n;
END; (* RulePixels *)

(******************************************************************************)

PROCEDURE DoSetRule (height, width : INTEGER);

(* Add rule information to current ruletable, update page edges, h and hh
   (but only if width and height are > 0).
*)

BEGIN
IF (height > 0) AND (width > 0) THEN BEGIN
   WITH ruletail^ DO
      IF rulecount = ruletablesize THEN BEGIN
         (* allocate a new ruletable *)
         NEW(nextrule);                   (* add new node to end of rule list *)
         nextrule^.rulecount := 0;        (* reset rulecount *)
         nextrule^.nextrule  := NIL;
         ruletail := nextrule;
      END;
   WITH ruletail^ DO BEGIN                (* may be new ruletable *)
      WITH ruletable[rulecount] DO BEGIN
         hp := hh;
         vp := vv;
         wd := RulePixels(width);
         ht := RulePixels(height);
         (* do page edges increase? *)
         IF vv - (ht - 1) < minvp THEN minvp := vv - (ht - 1);
         IF hh + (wd - 1) > maxhp THEN maxhp := hh + (wd - 1);
         (* ref pt of rule is bottom left black pixel *)
         IF vv > maxvp THEN maxvp := vv;
         IF hh < minhp THEN minhp := hh;
         hh := hh + wd;
         (* use hhh and maxdrift to prevent hh drifting too far from h *)
         hhh := PixelRound(h + width);
         IF ABS(hhh - hh) > maxdrift THEN
            IF hhh > hh THEN
               hh := hhh - maxdrift
            ELSE
               hh := hhh + maxdrift;
         h := h + width;
      END;
      totalrules := totalrules + 1;
      rulecount := rulecount + 1;
   END;
END;
END; (* DoSetRule *)

(******************************************************************************)

PROCEDURE DoPutRule (height, width : INTEGER);

(* Exactly the same as DoSetRule, but we DON'T update the horizontal
   position on the page.  (We still have to check page edges.)
*)

BEGIN
IF (height > 0) AND (width > 0) THEN BEGIN
   WITH ruletail^ DO
      IF rulecount = ruletablesize THEN BEGIN
         (* allocate a new ruletable *)
         NEW(nextrule);                   (* add new node to end of rule list *)
         nextrule^.rulecount := 0;        (* reset rulecount *)
         nextrule^.nextrule := NIL;
         ruletail := nextrule;
      END;
   WITH ruletail^ DO BEGIN                (* may be new ruletable *)
      WITH ruletable[rulecount] DO BEGIN
         hp := hh;
         vp := vv;
         wd := RulePixels(width);
         ht := RulePixels(height);
         (* do page edges increase? *)
         IF vv - (ht - 1) < minvp THEN minvp := vv - (ht - 1);
         IF hh + (wd - 1) > maxhp THEN maxhp := hh + (wd - 1);
         (* ref pt of rule is bottom left black pixel *)
         IF vv > maxvp THEN maxvp := vv;
         IF hh < minhp THEN minhp := hh;
      END;
      totalrules := totalrules + 1;
      rulecount := rulecount + 1;
   END;
END;
END; (* DoPutRule *)

(******************************************************************************)

PROCEDURE DoFont (externf : INTEGER);

(* Search font list for externf, setting currfont and fontspace.
   If this is the first time we've seen this font (on current page) then
   we need to allocate the first chartable.
   If this is the first time we've seen this font used at all then we
   allocate a pixeltable and call routine to fill it in.
*)

LABEL 888;

BEGIN
currfont := fontlist;
WHILE currfont <> NIL DO
   IF currfont^.fontnum <> externf THEN
      currfont := currfont^.nextfont
   ELSE
      goto 888;
888:
{ DEBUG
  IF currfont = NIL THEN BEGIN
     WriteLine; WriteString('Failed to find font #'); WriteInt(externf);
     WriteLine; RestoreTerminal; exit(1);
  END;
GUBED }
WITH currfont^ DO BEGIN
   IF fontused THEN
      (* do nothing since we've already used this font on this page *)
   ELSE BEGIN
      fontused := TRUE;
      NEW(charlist);           (* allocate first chartable *)
      WITH charlist^ DO BEGIN
         charcount := 0;       (* for DoSet/PutChar *)
         nextchar  := NIL;     (* this node is also last *)
      END;
      chartail := charlist;
      IF pixelptr = NIL THEN BEGIN  (* first time we've seen this font *)
         NEW(pixelptr);
         PixelTableRoutine;
      END;
   END;
   fontspace  := scaledsize DIV 6;
   (* See DVITYPE; a 3-unit thin space.
      Note that a thin space is 1/6 of a quad, where a quad is
      1 em in the current font and usually equals the design size.
   *)
END;
END; (* DoFont *)

(******************************************************************************)

PROCEDURE DoSpecial (hpos, vpos, totalbytes : INTEGER);   (* in *)

(* DVIReader has seen a \special command while interpreting the current page.
   It will pass the current page position and number of bytes in the command.
   We save the info away in speciallist for later use by the main program.
*)

VAR i, flush : INTEGER;     temp : specialinfoptr;

BEGIN
NEW(temp);
WITH temp^ DO BEGIN
   special := '';                     (* SYSDEP: fill with spaces *)
   FOR i := 0 TO totalbytes-1 DO
      IF i < maxspeciallen THEN special[i] := CHR(GetDVIByte);
   (* we must read all the \special bytes *)
   IF totalbytes > maxspeciallen THEN BEGIN
      FOR i := 1 TO totalbytes - maxspeciallen DO flush := GetDVIByte;
   END;
   hp := hpos;
   vp := vpos;
   nextspecial := speciallist;
END;
speciallist := temp;                  (* add new info to head of list *)
END; (* DoSpecial *)

(******************************************************************************)

PROCEDURE InterpretPage;

(* When this routine is called we are positioned after the bytes of a bop
   command (i.e., at currbop + 45).  At the end of this routine we will be
   positioned after the eop byte for the current page.  In between we carry
   out the important task of translating the DVI description of this page
   and filling in the following global data structures:

      totalrules := number of rules on page
      rulelist^. (nodes are added to tail of rule list)
         rulecount  := number of rules in this ruletable
         ruletable[0..rulecount-1].
            hp, vp  := reference point of a rule
            wd, ht  := pixel dimensions of a rule (both > 0)
         nextrule   := next node in rule list (if not NIL)
      ruletail   := pointer to last node in rule list

      speciallist^. (nodes are added to head of this list)
         hp, vp      := reference point on page
         special     := \special bytes
         nextspecial := next node in list (if not NIL)

      fontlist^.
         (the following fontinfo is relevant only if fontused is TRUE)
         totalchars := number of chars on page from this font
         charlist^. (nodes are added to tail of char list)
            charcount  := number of chars in this chartable
            chartable[0..charcount-1].
               hp, vp  := reference point of a character
               code    := TeX character code (and index into pixel table)
            nextchar   := next node in char list (if not NIL)
         chartail   := pointer to last node in char list
         pixelptr^[0..maxTeXchar].
            (filled in by FontReader's PixelTableRoutine)
            wd, ht  := glyph width and height in pixels
            xo, yo  := offsets from the character's reference point
            dwidth  := advance width in DVI units
            pwidth  := advance width in pixels
            mapadr  := offset in fontspec of the glyph's bitmap info
            bitmap  := NIL
         nextfont   := next node in font list (if not NIL)

      pageempty  := TRUE iff the page has no rules and no characters
      minhp, minvp, maxhp, maxvp
                 := the edges of the page (undefined if pageempty is TRUE)
                    They define the smallest rectangle containing all black
                    pixels on the page;(minhp,minvp) is the top left corner.

   Reference points for rules and characters are stored as a pair of
   horizontal and vertical pixel coordinates.  The point (0,0) is assumed
   to be the pixel 1 inch in from the top and left edges of an imaginary sheet
   of paper.  Horizontal coordinates increase to the right and vertical
   coordinates increase down the paper.
   The number of pixels per inch is defined by the resolution parameter
   given to SetConversionFactor.
*)

VAR param, ht, wd : INTEGER;

BEGIN
InitStateValues;
InitPage;
REPEAT
   DVIcommand := GetDVIByte;
   (* For efficiency reasons the most frequent commands should be tested 1st.
      The following order is the result of frequency testing on typical
      DVI files.  Note that the most frequent commands in the DVI file
      generated by TeX 1.3 for the Dec. 1983 LaTeX manual were:
      <set1, w0, right3, push/pop, x0, w3, y0, fntnum25, right2, fntnum31,
      down3, x2, right4, w2, x3, down4, z0, fntnum8, setrule, y3, etc.
   *)
   IF DVIcommand < set1       THEN DoSetChar(DVIcommand)   (* 0..127 *)
   ELSE IF DVIcommand = w0      THEN DoRight(w)
   ELSE IF DVIcommand = right3  THEN DoRight(SignedDVITrio)
   ELSE IF DVIcommand = push    THEN DoPush
   ELSE IF DVIcommand = pop     THEN DoPop
   ELSE IF DVIcommand = x0      THEN DoRight(x)
   ELSE IF DVIcommand = w3      THEN BEGIN w := SignedDVITrio; DoRight(w) END
   ELSE IF DVIcommand = y0      THEN DoDown(y)

   ELSE IF (DVIcommand > z4) AND (DVIcommand < fnt1)   (* fntnum0..fntnum63 *)
                              THEN DoFont(DVIcommand - fntnum0)

   (* catch all the remaining movement commands *)
   ELSE IF (DVIcommand > pop) AND (DVIcommand < fntnum0) THEN BEGIN
      IF DVIcommand = right2     THEN DoRight(SignedDVIPair)
      ELSE IF DVIcommand = right4  THEN DoRight(SignedDVIQuad)
      ELSE IF DVIcommand = x2      THEN BEGIN x := SignedDVIPair; DoRight(x) END
      ELSE IF DVIcommand = x3      THEN BEGIN x := SignedDVITrio; DoRight(x) END
      ELSE IF DVIcommand = down3   THEN DoDown(SignedDVITrio)
      ELSE IF DVIcommand = down4   THEN DoDown(SignedDVIQuad)
      ELSE IF DVIcommand = w2      THEN BEGIN w := SignedDVIPair; DoRight(w) END
      ELSE IF DVIcommand = z0      THEN DoDown(z)
      ELSE IF DVIcommand = y3      THEN BEGIN y := SignedDVITrio; DoDown(y) END
      ELSE IF DVIcommand = z3      THEN BEGIN z := SignedDVITrio; DoDown(z) END
      ELSE IF DVIcommand = down2   THEN DoDown(SignedDVIPair)
      (* the next DVI commands are used very rarely (by TeX 1.3 at least) *)
      ELSE IF DVIcommand = w1      THEN BEGIN w := SignedDVIByte; DoRight(w) END
      ELSE IF DVIcommand = w4      THEN BEGIN w := SignedDVIQuad; DoRight(w) END
      ELSE IF DVIcommand = x1      THEN BEGIN x := SignedDVIByte; DoRight(x) END
      ELSE IF DVIcommand = x4      THEN BEGIN x := SignedDVIQuad; DoRight(x) END
      ELSE IF DVIcommand = y1      THEN BEGIN y := SignedDVIByte; DoDown(y) END
      ELSE IF DVIcommand = y2      THEN BEGIN y := SignedDVIPair; DoDown(y) END
      ELSE IF DVIcommand = y4      THEN BEGIN y := SignedDVIQuad; DoDown(y) END
      ELSE IF DVIcommand = z1      THEN BEGIN z := SignedDVIByte; DoDown(z) END
      ELSE IF DVIcommand = z2      THEN BEGIN z := SignedDVIPair; DoDown(z) END
      ELSE IF DVIcommand = z4      THEN BEGIN z := SignedDVIQuad; DoDown(z) END
      ELSE IF DVIcommand = right1  THEN DoRight(SignedDVIByte)
      ELSE IF DVIcommand = down1   THEN DoDown(SignedDVIByte)
      ELSE BEGIN
           WriteLine; WriteString('Bug in InterpretPage!'); WriteLine;
           RestoreTerminal; exit(1);
      END;
   END
   ELSE IF DVIcommand = setrule THEN BEGIN
           ht := SignedDVIQuad; wd := SignedDVIQuad; DoSetRule(ht,wd);
   END
   ELSE IF DVIcommand = putrule THEN BEGIN
           ht := SignedDVIQuad; wd := SignedDVIQuad; DoPutRule(ht,wd);
   END
   ELSE IF (DVIcommand >= put1) AND (DVIcommand <= put1+3) THEN
      CASE DVIcommand - put1 OF
         0 : DoPutChar(GetDVIByte);
         1 : DoPutChar(GetTwoDVIBytes);
         2 : DoPutChar(GetThreeDVIBytes);
         3 : DoPutChar(SignedDVIQuad)
      END

   ELSE IF (DVIcommand >= set1) AND (DVIcommand <= set1+3) THEN
      CASE DVIcommand - set1 OF
         0 : DoSetChar(GetDVIByte);
         1 : DoSetChar(GetTwoDVIBytes);
         2 : DoSetChar(GetThreeDVIBytes);
         3 : DoSetChar(SignedDVIQuad)
      END

   ELSE IF (DVIcommand >= fnt1) AND (DVIcommand <= fnt1+3) THEN
      CASE DVIcommand - fnt1 OF
         0 : DoFont(GetDVIByte);
         1 : DoFont(GetTwoDVIBytes);
         2 : DoFont(GetThreeDVIBytes);
         3 : DoFont(SignedDVIQuad)
      END

   ELSE IF (DVIcommand >= xxx1) AND (DVIcommand <= xxx1+3) THEN BEGIN
      CASE DVIcommand - xxx1 OF
         0 : param := GetDVIByte;
         1 : param := GetTwoDVIBytes;
         2 : param := GetThreeDVIBytes;
         3 : param := SignedDVIQuad
      END;
      (* pass current pixel position and number of bytes *)
      DoSpecial(hh, vv, param);
   END

   (* skip fntdef command since we've got this info from postamble *)
   ELSE IF (DVIcommand >= fntdef1) AND (DVIcommand <= fntdef1+3)
                              THEN SkipFntdef(DVIcommand - fntdef1)

   ELSE IF DVIcommand = nop   THEN (* do nothing *)
   ELSE IF DVIcommand = eop   THEN (* do nothing *)
   ELSE BEGIN
      WriteLine;
      WriteString('Unexpected DVI command while interpreting page=');
      WriteInt(DVIcommand); WriteLine;
      RestoreTerminal; exit(1);
   END;
UNTIL DVIcommand = eop;
(* save position of eop byte for use in MoveToTeXPage *)
curreop := DVIoffset - 1;
IF stackpos <> 0 THEN BEGIN
   WriteLine;
   WriteString('Stack not empty at eop!'); WriteLine;
   RestoreTerminal; exit(1);
END;
pageempty := (minhp = maxint) AND (minvp = maxint) AND
             (maxhp = -maxint) AND (maxvp = -maxint); (* InitPage values *)
END; (* InterpretPage *)

(******************************************************************************)

PROCEDURE SortFonts (VAR unusedlist : fontinfoptr);   (* out *)

(* Sort fontlist in order of ascending totalchars.
   Fonts with least characters can then be accessed first.
   Since the number of fonts used on a typical page is quite small, a simple
   sorting algorithm should be good enough.  Note that unused fonts are moved
   to the end of the list and unusedlist points to the first such node.
   DVItoVDU need only process fonts up to (but excluding) unusedlist
   and does not have to worry about checking the fontused flag.
   If unusedlist is NIL then either 1) all fonts are used on the current page
   or 2) fontlist is also NIL (totalfonts = 0).
*)

VAR newfontlist, prevfont, largest, prevlargest : fontinfoptr;
    mostchars : INTEGER;

BEGIN
newfontlist  := NIL;
(* go thru fontlist once and move all unused fonts to head of newfontlist *)
prevfont := NIL;
currfont := fontlist;
WHILE currfont <> NIL DO
   WITH currfont^ DO
      IF fontused THEN BEGIN
         prevfont := currfont;      (* remember previous node *)
         currfont := nextfont;
      END
      ELSE
         (* move node from fontlist to head of newfontlist
            and don't change prevfont
         *)
         IF prevfont = NIL THEN BEGIN
            fontlist := nextfont;   (* remove first node in fontlist *)
            nextfont := newfontlist;
            newfontlist := currfont;
            currfont := fontlist;
         END
         ELSE BEGIN
            prevfont^.nextfont := nextfont;
            nextfont := newfontlist;
            newfontlist := currfont;
            currfont := prevfont^.nextfont;
         END;
(* unusedlist will be last unused font moved to newfontlist.  It will be NIL
   if either fontlist is NIL or all fonts are used.
*)
unusedlist := newfontlist;
(* Now go thru fontlist repeatedly moving node with max totalchars to
   head of newfontlist until fontlist is exhausted.
*)
WHILE fontlist <> NIL DO BEGIN
   prevfont := NIL;
   currfont := fontlist;
   prevlargest := NIL;
   largest     := fontlist;
   mostchars   := 0;
   WHILE currfont <> NIL DO   (* search for largest totalchars *)
      WITH currfont^ DO BEGIN
         IF totalchars > mostchars THEN BEGIN
            prevlargest := prevfont;
            largest     := currfont;
            mostchars   := totalchars;
         END;
         prevfont := currfont;
         currfont := nextfont;
      END;
   (* move largest node from fontlist to head of newfontlist *)
   WITH largest^ DO BEGIN
      IF prevlargest = NIL THEN
         fontlist := nextfont   (* remove first node in fontlist *)
      ELSE
         prevlargest^.nextfont := nextfont;
      nextfont := newfontlist;
      newfontlist := largest;
   END;
END;
fontlist := newfontlist;   (* used fonts now sorted and unused fonts at end *)
END; (* SortFonts *)

(******************************************************************************)

PROCEDURE CloseDVIFile;

(* Close the currently open DVI file and deallocate dynamic data structures. *)

VAR result : integer;

BEGIN
result := close(DVIfile);
WHILE fontlist <> NIL DO BEGIN
   currfont := fontlist;
   WITH currfont^ DO BEGIN
      WHILE charlist <> NIL DO BEGIN
         thischar := charlist;
         charlist := thischar^.nextchar;
         DISPOSE(thischar);      (* deallocate char list *)
      END;
      IF pixelptr <> NIL THEN
         DISPOSE(pixelptr);      (* deallocate pixel table *)
      fontlist := nextfont;
   END;
   DISPOSE(currfont);            (* deallocate font information *)
END;
(* Deallocate rule information except for one node (in case DVItoVDU ever
   opens another DVI file).
*)
WHILE rulelist <> ruletail DO BEGIN
   thisrule := rulelist;
   rulelist := thisrule^.nextrule;
   DISPOSE(thisrule);
END;
WHILE speciallist <> NIL DO BEGIN
   thisspecial := speciallist;
   speciallist := speciallist^.nextspecial;
   DISPOSE(thisspecial);
END;
END; (* CloseDVIFile *)

(******************************************************************************)

PROCEDURE InitDVIReader;

BEGIN
totalrules := 0;
NEW(rulelist);             (* for first InitPage *)
ruletail := rulelist;      (* ditto *)
speciallist := NIL;        (* ditto *)
fontlist := NIL;           (* safer for CloseDVIFile *)
END; (* InitDVIReader *)
