/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains routines for manipulating the AST Queue.
 */

#include "types.h"
#include "ast.h"

/*
 *	The AST queue is a circular linked list of entries containing
 *	AST descriptor blocks. The queue entries are allocated dynamically
 *	as needed, so the queue is initially null:
 */

struct AST_Block {
	struct AST_Block *FLink;	/* Pointer to next one */
	struct AST_Block *Blink;	/* Pointer to previous one */
	unsigned long AST_Priority;	/* This is a priority-ordered queue */
	unsigned long AST_Type;		/* What was it? */
	unsigned long AST_Arg1;		/* One longword argument */
	unsigned long AST_Arg2;		/* Another longword argument */
};

struct AST_Block AST_Queue_Header = { &AST_Queue_Header, &AST_Queue_Header, 0 };

/*
 *	The AST Queue Free list is a list of available queue nodes.
 *	This is used to minimize the amount of dynamic memory allocation
 *	used for handling the AST queue:
 */

struct AST_Block *AST_Free_List = 0;

/*
 *	Routine Initialize_AST_Queue initializes the AST Queue. This
 *	consists solely of pre-allocating a few AST blocks, since the
 *	queue is initialized by compile-time declarations. This routine
 *	does not need to be called in order for things to work:
 */

Initialize_AST_Queue ()
{
	auto   struct AST_Block *AST_Ptr;
	auto   unsigned int Count;
	extern char *Mem_Alloc();

	for (Count = 10; Count > 0; Count--) {
		AST_Ptr = (struct AST_Block *) Mem_Alloc (sizeof (struct AST_Block));
		AST_Ptr->FLink = AST_Free_List;
		AST_Free_List = AST_Ptr;
	}
}

EnQueue_AST_Locked (Type, Arg1, Arg2, Priority)
unsigned long Type, Arg1, Arg2, Priority;
{
	extern unsigned long Sys$SetAST();

	Sys$SetAST (0);
	EnQueue_AST (Type, Arg1, Arg2, Priority);
	Sys$SetAST (1);
}

/*
 *	Routine EnQueue_AST adds an entry to the AST queue. This is always
 *	done at AST level, so no interlocking logic is necessary:
 */

EnQueue_AST (Type, Arg1, Arg2, Priority)
unsigned long Type, Arg1, Arg2, Priority;
{
	auto   struct AST_Block *AST_Entry_Ptr, *AST_Pred_Ptr;
	extern char *Mem_Alloc();
/*
 *	Check the priorities for consistency just in case something
 *	got clobbered. This is necessary since the header entry MUST
 *	have a priority of zero, and no others may have a
 *	priority of zero, in order to avoid an endless loop:
 */
	if (AST_Queue_Header.AST_Priority != 0)
		AST_Queue_Header.AST_Priority = 0;
	if (Priority == 0)
		Priority = 1;
/*
 *	Allocate some space for the entry (if none in the free list),
 *	and fill it in:
 */
	if ((AST_Entry_Ptr = AST_Free_List) != 0)
		AST_Free_List = AST_Entry_Ptr->FLink;
	else
		AST_Entry_Ptr = (struct AST_Block *) Mem_Alloc (sizeof (struct AST_Block));
	AST_Entry_Ptr->AST_Type = Type;
	AST_Entry_Ptr->AST_Arg1 = Arg1;
	AST_Entry_Ptr->AST_Arg2 = Arg2;
	AST_Entry_Ptr->AST_Priority = Priority;
/*
 *	Find the first entry with a priority less than the required priority
 *	(if the priorities are the same, insert in FIFO order):
 */
	for (AST_Pred_Ptr = &AST_Queue_Header; AST_Pred_Ptr->FLink->AST_Priority >= Priority;
				AST_Pred_Ptr = AST_Pred_Ptr->FLink)
		;
	Insert_into_Queue (AST_Entry_Ptr, AST_Pred_Ptr);
}

/*
 *	Routine DeQueue_AST returns the values for the entry at the
 *	head of the queue, then evaporates the entry. We must be
 *	careful here, since this routine can be interrupted by an
 *	AST anytime during execution:
 */

unsigned long DeQueue_AST (Arg1_Ptr, Arg2_Ptr)
unsigned long *Arg1_Ptr, *Arg2_Ptr;
{
	auto   struct AST_Block *AST_Ptr;
	auto   unsigned long Type;
	extern char ***Remove_from_Queue();
	extern unsigned long Sys$SetAST();

	Sys$SetAST (0);
	AST_Ptr = (struct AST_Block *) Remove_from_Queue (AST_Queue_Header.FLink);
	if (AST_Ptr == 0)
		Type = AST_NULL;
	else {
		Type = AST_Ptr->AST_Type;
		*Arg1_Ptr = AST_Ptr->AST_Arg1;
		*Arg2_Ptr = AST_Ptr->AST_Arg2;
		AST_Ptr->FLink = AST_Free_List;
		AST_Free_List = AST_Ptr;
	}
	Sys$SetAST (1);
	return (Type);
}

/*
 *	The following two routines are general purpose queue manipulation
 *	routines:
 */

Insert_into_Queue (Entry, Pred)
char ***Entry, ***Pred;
{
	Entry[0] = Pred[0];
	Entry[1] = Pred;
	Pred[0][1] = Entry;
	Pred[0] = Entry;
}

char ***Remove_from_Queue (Entry)
char ***Entry;
{
	if (Entry[0] == Entry)		/* Queue is empty */
		return (0);
	Entry[1][0] = Entry[0];
	Entry[0][1] = Entry[1];
	return (Entry);
}
