/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains miscellaneous VMS dependent functions.
 */

#include signal
#include "ast.h"

/*
 *	Routine Enable_Interrupt sets up an interrupt handler which will
 *	set the user-supplied flag when an interrupt (Control-C) is entered.
 *	This makes use of the C run-time signal handling, an unfortunate
 *	necessity.
 */

unsigned long *Interrupt_Flag_Ptr;

Enable_Interrupt (Flag_Ptr)
unsigned long *Flag_Ptr;
{
	auto   int (*Old_Handler)();
	extern unsigned int Interrupt_Handler();
	extern int (*ssignal())();

	Interrupt_Flag_Ptr = Flag_Ptr;
	*Interrupt_Flag_Ptr = 0;
	Old_Handler = ssignal (SIGINT, &Interrupt_Handler);
}

/*
 *	The Interrupt Handler sets the flag longword to a non-zero
 *	value returns. Another explicit call to Enable_Interrupt is
 *	necessary to process more than one interrupt.
 */

unsigned int Interrupt_Handler (Sig)
unsigned int Sig;
{
	globalvalue SS$_NORMAL;

	if (Sig == SIGINT)
		*Interrupt_Flag_Ptr = 1;
	EnQueue_AST (AST_INTERRUPT, 0, 0, PRI_INTERRUPT);
	return (SS$_NORMAL);
}

/*
 *	Routine Check_System_Status reports on a bad status from a VMS
 *	system service, something that should never normally happen
 *	(except for running out of dynamic memory):
 */

int Check_System_Status (Sys_Status)
unsigned long Sys_Status;
{
	globalvalue ERROR_SYSERROR;

	if ((Sys_Status & 0x01) == 0) {
		Message (ERROR_SYSERROR, Sys_Status, 0);
		Message (Sys_Status);
		return (0);
	}
	return (1);
}
